# 機能設計書 94-均一量子化操作

## 概要

本ドキュメントは、TensorFlowにおける均一量子化操作（Uniform Quantization Operations）機能の設計を記述する。INT8等への均一量子化・逆量子化操作、および量子化テンソルに対する演算（ドット積、畳み込み、加算、クリッピング）を提供する。

### 本機能の処理概要

**業務上の目的・背景**：モデルの推論速度向上とメモリ削減のため、FP32モデルをINT8等の低精度表現に変換する量子化が必要である。均一量子化は、浮動小数点値を等間隔のグリッドにマッピングする方式であり、量子化パラメータ（scales, zero_points）を用いて変換・逆変換を行う。

**機能の利用シーン**：モデルデプロイメント時のINT8量子化、TFLiteへの変換前の量子化対応モデル構築、量子化対応の学習（Quantization-Aware Training）で利用される。

**主要な処理内容**：
1. `UniformQuantize` - 浮動小数点テンソルをINT8/UINT8/INT32量子化テンソルに変換
2. `UniformDequantize` - 量子化テンソルを浮動小数点に逆変換
3. `UniformRequantize` - 量子化テンソルを異なる量子化パラメータで再量子化
4. `UniformQuantizedDot` - 量子化テンソル間のドット積
5. `UniformQuantizedDotHybrid` - 浮動小数点と量子化テンソルのハイブリッドドット積
6. `UniformQuantizedConvolution` - 量子化テンソル間の畳み込み
7. `UniformQuantizedConvolutionHybrid` - ハイブリッド畳み込み
8. `UniformQuantizedAdd` - 量子化テンソルの加算
9. `UniformQuantizedClipByValue` - 量子化テンソルのクリッピング

**関連システム・外部連携**：TFLiteモデル変換パイプライン、XLAコンパイラの量子化サポートと連携する。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はAPI機能であり、直接関連する画面はない |

## 機能種別

計算処理（量子化変換・量子化演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| input | Tin (float/qint8/quint8/qint32) | Yes | 量子化対象テンソル | - |
| scales | float | Yes | 量子化スケール | スカラーまたは1次元 |
| zero_points | int32 | Yes | 量子化ゼロポイント | scalesと同じrank |
| quantization_axis | int | No | 量子化軸（デフォルト: -1、テンソル全体量子化） | -1以上 |
| quantization_min_val | int | Yes | 量子化最小値 | - |
| quantization_max_val | int | Yes | 量子化最大値 | min_val < max_val |
| lhs / rhs | Tin | Yes（演算系） | 左辺/右辺テンソル | Dot: rank 2, Conv: rank >= 2 |

### 入力データソース

計算グラフ内のテンソル。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output (Quantize) | qint8/quint8/qint32 | 量子化されたテンソル |
| output (Dequantize) | float | 逆量子化されたテンソル |
| output (Dot/Conv) | qint32 | 量子化演算結果 |
| output (DotHybrid/ConvHybrid) | float | ハイブリッド演算結果 |
| output (Add) | qint32 | 量子化加算結果 |
| output (ClipByValue) | qint32 | クリッピング結果 |

### 出力先

計算グラフ内の後続操作への入力。

## 処理フロー

### 処理シーケンス

```
1. 入力テンソルの検証
   └─ rank制約、scales/zero_pointsの形状一致
2. 量子化パラメータの検証
   └─ scales, zero_pointsのrank一致、quantization_axisの妥当性
3. 量子化/逆量子化/演算の実行
   └─ 量子化: round(input / scale) + zero_point
   └─ 逆量子化: (input - zero_point) * scale
   └─ 演算: 量子化ドメインでの演算
4. 結果の返却
```

### フローチャート

```mermaid
flowchart TD
    A[入力テンソル] --> B{操作種別}
    B -->|Quantize| C[float → qint8/quint8/qint32]
    B -->|Dequantize| D[qint8/quint8/qint32 → float]
    B -->|Requantize| E[qint → qint（再量子化）]
    B -->|Dot/Conv| F[量子化演算]
    B -->|DotHybrid/ConvHybrid| G[ハイブリッド演算]
    B -->|Add| H[量子化加算]
    B -->|ClipByValue| I[量子化クリッピング]
    C --> J[出力テンソル]
    D --> J
    E --> J
    F --> J
    G --> J
    H --> J
    I --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-94-01 | scales/zero_points一致 | scalesとzero_pointsは同じrankで同じサイズ | 全Op |
| BR-94-02 | per-tensor量子化 | quantization_axis=-1の場合、scales/zero_pointsはスカラー | axis=-1時 |
| BR-94-03 | per-channel量子化 | quantization_axis >= 0の場合、scales/zero_pointsは1次元 | axis >= 0時 |
| BR-94-04 | Dot入力制約 | lhs, rhsはrank 2の行列 | Dot系Op |
| BR-94-05 | Conv入力制約 | lhs, rhsはrank 2以上 | Conv系Op |
| BR-94-06 | lhs量子化制約 | Dot/Convのlhs scalesはスカラー（per-tensor量子化） | Dot/Conv |

### 計算ロジック

均一量子化の基本式：
- 量子化: `q = clamp(round(r / s) + z, qmin, qmax)`
- 逆量子化: `r = (q - z) * s`
  - r: 実数値、q: 量子化値、s: スケール、z: ゼロポイント

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| InvalidArgument | rank不一致 | lhsとrhsのrankが一致しない（Conv） | 同じrankのテンソルを入力 |
| InvalidArgument | scales/zero_points不一致 | scalesとzero_pointsのrankが異なる | 同じrankの値を入力 |
| InvalidArgument | 内積次元不一致 | Dotのlhs列数とrhs行数が不一致 | 互換な行列を入力 |

### リトライ仕様

リトライ不要。

## トランザクション仕様

該当なし。ステートレスなテンソル演算。

## パフォーマンス要件

- INT8演算はFP32に対して理論上4倍のスループット向上が見込める
- GPU/TPU上では専用の量子化演算ユニットを活用可能

## セキュリティ考慮事項

- 量子化による精度損失がモデルの公平性・安全性に影響する可能性がある

## 備考

- `UniformQuantizedConvolutionParams`クラスが畳み込みパラメータの検証と出力形状計算を担当
- dimension_numbers属性でXLA互換の次元指定が可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | uniform_quant_ops_params.h | `tensorflow/core/util/quantization/uniform_quant_ops_params.h` | `UniformQuantizedConvolutionParams`クラスの定義 |

**読解のコツ**: 量子化パラメータ（scales, zero_points, quantization_axis）と畳み込みパラメータ（strides, padding, dilation等）が分離されている。

#### Step 2: Op定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | uniform_quant_ops.cc | `tensorflow/core/ops/uniform_quant_ops.cc` | 全9種のOp定義 |

**主要処理フロー**:
1. **290-300行目**: `UniformQuantize` Op - float → qint8/quint8/qint32
2. **302-317行目**: `UniformRequantize` Op - 再量子化
3. **319-329行目**: `UniformDequantize` Op - 逆量子化
4. **331-352行目**: `UniformQuantizedDot` Op - 量子化ドット積
5. **354-366行目**: `UniformQuantizedDotHybrid` Op - ハイブリッドドット積
6. **368-397行目**: `UniformQuantizedConvolution` Op - 量子化畳み込み
7. **399-419行目**: `UniformQuantizedConvolutionHybrid` Op - ハイブリッド畳み込み
8. **421-441行目**: `UniformQuantizedAdd` Op - 量子化加算
9. **443-454行目**: `UniformQuantizedClipByValue` Op - クリッピング

#### Step 3: 形状推論ロジックを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | uniform_quant_ops.cc | `tensorflow/core/ops/uniform_quant_ops.cc` | DotShape (76-117行目), ConvolutionShape (241-270行目) |

**主要処理フロー**:
- **46-74行目**: `ScalesZeroPointsShapeValid` - scales/zero_pointsの形状検証
- **76-117行目**: `DotShape` - lhs[M,K] x rhs[K,N] → output[M,N]
- **181-239行目**: `ConvolutionShapeCommon` - 畳み込み出力形状計算

### プログラム呼び出し階層図

```
UniformQuantize / UniformDequantize / UniformRequantize
    └─ shape_inference::UnchangedShape

UniformQuantizedDot
    └─ DotShape
           ├─ WithRank(lhs, 2) / WithRank(rhs, 2)
           ├─ Merge(inner_lhs, inner_rhs)
           └─ ScalesZeroPointsShapeValid

UniformQuantizedConvolution
    └─ ConvolutionShape
           └─ ConvolutionShapeCommon
                  ├─ ToTensorShape
                  ├─ UniformQuantizedConvolutionParams.LoadFromAttrs
                  ├─ ValidateOrFillParamsAndValidateShape
                  ├─ CalculateOutputShape
                  └─ ScalesZeroPointsShapeValid
```

### データフロー図

```
[入力]                          [処理]                          [出力]

float Tensor          ───▶  UniformQuantize              ───▶  qint8 Tensor
                             (scales, zero_points)

qint8 Tensor          ───▶  UniformDequantize            ───▶  float Tensor
                             (scales, zero_points)

qint8 lhs, qint8 rhs  ───▶  UniformQuantizedDot          ───▶  qint32 Tensor
                             (scales, zero_points x 4)

float lhs, qint8 rhs  ───▶  UniformQuantizedDotHybrid    ───▶  float Tensor
                             (rhs_scales, rhs_zero_points)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| uniform_quant_ops.cc | `tensorflow/core/ops/uniform_quant_ops.cc` | ソース | Op定義と形状推論 |
| uniform_quant_ops_params.h | `tensorflow/core/util/quantization/uniform_quant_ops_params.h` | ヘッダ | 畳み込みパラメータクラス |
| uniform_quant_ops_params.cc | `tensorflow/core/util/quantization/uniform_quant_ops_params.cc` | ソース | パラメータ検証・形状計算 |
