# 通知設計書 24-XlaActivityListener

## 概要

本ドキュメントは、TensorFlowのXLAコンパイラにおける`XlaActivityListener`の通知設計について記載する。本リスナーは、XLA自動クラスタリング、JITコンパイル、最適化リマークイベントを登録されたリスナーへブロードキャストするためのC++インターフェースである。

### 本通知の処理概要

XlaActivityListenerは、XLAコンパイラの各種アクティビティ（自動クラスタリング、JITコンパイル、最適化リマーク）をオブザーバーパターンで監視するための抽象基底クラスである。BroadcastXlaActivity()関数を通じて全登録リスナーへイベントがブロードキャストされる。

**業務上の目的・背景**：XLA（Accelerated Linear Algebra）コンパイラは、TensorFlowグラフの一部をGPU/TPU向けに最適化コンパイルする。自動クラスタリングによるグラフ分割、JITコンパイル結果、パフォーマンス最適化に関するリマーク（警告）など、XLAの動作状況を外部モジュールから監視する必要がある。XlaActivityListenerはこの監視ポイントを提供し、テレメトリ収集やデバッグ情報の集約を可能にする。

**通知の送信タイミング**：(1) TensorFlowがグラフの自動クラスタリングを完了した後（XlaAutoClusteringActivity）、(2) XLAクラスタのJITコンパイルが完了した後（XlaJitCompilationActivity）、(3) パフォーマンス低下の可能性を検知した後（XlaOptimizationRemark）の3種類のイベントで通知が発行される。

**通知の受信者**：RegisterXlaActivityListener()を通じて登録されたXlaActivityListenerのサブクラスインスタンスが受信者となる。リスナーはグローバルなリストで管理され、プロセスのライフタイム中に蓄積される。

**通知内容の概要**：Protocol Buffers形式のメッセージとして、XlaAutoClusteringActivity（クラスタリング結果）、XlaJitCompilationActivity（コンパイル結果）、XlaOptimizationRemark（最適化警告とデバッグ情報）が送信される。

**期待されるアクション**：リスナーはイベントを受信し、テレメトリサービスへの送信、ログ記録、統計情報の集約などを行う。プログラム終了時にはFlush()が呼ばれ、リスナーは蓄積した情報をフラッシュする機会を得る。

## 通知種別

プロセス内コールバック（C++ virtualメソッド呼び出し）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期（ブロードキャスト関数内で全リスナーを順次呼び出し） |
| 優先度 | 中（テレメトリ・デバッグ用途） |
| リトライ | 無し（エラー時はStatusを返却） |

### 送信先決定ロジック

グローバルなXlaActivityListenerListに登録された全リスナーに対してブロードキャストされる。ReaderMutexLockによりリスナーリストへの同時アクセスを保護し、各リスナーのListen()メソッドを順次呼び出す。

## 通知テンプレート

### コールバック呼び出しの場合

| 項目 | 内容 |
|-----|------|
| 送信元 | BroadcastXlaActivity() / BroadcastOptimizationRemark() |
| 送信形式 | Protocol Buffers (xla_activity.pb.h) |
| 受信メソッド | XlaActivityListener::Listen() |

### 本文テンプレート

```cpp
// XlaAutoClusteringActivity
message XlaAutoClusteringActivity {
  // クラスタリングに関する詳細情報
}

// XlaJitCompilationActivity
message XlaJitCompilationActivity {
  // JITコンパイルに関する詳細情報
}

// XlaOptimizationRemark
message XlaOptimizationRemark {
  Warning warning = 1;
  string debug_information = 2;
}
```

### 添付ファイル

なし（プロセス内通信）

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| auto_clustering_activity | 自動クラスタリング結果 | XLAクラスタリングパス | Yes（クラスタリング通知時） |
| jit_compilation_activity | JITコンパイル結果 | XLA JITコンパイラ | Yes（コンパイル通知時） |
| optimization_remark | 最適化リマーク | XLA最適化パス | Yes（リマーク通知時） |
| optimization_warning | 最適化警告種別 | XlaOptimizationRemark::Warning enum | Yes（リマーク通知時） |
| debug_information | デバッグ情報文字列 | XLA最適化パス | Yes（リマーク通知時） |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| グラフ最適化 | 自動クラスタリング完了 | BroadcastXlaActivity(auto_clustering_activity)呼び出し | グラフのXLAクラスタ分割完了後 |
| JITコンパイル | XLAクラスタコンパイル完了 | BroadcastXlaActivity(jit_compilation_activity)呼び出し | XLAクラスタのコンパイル完了後 |
| 最適化警告 | パフォーマンス低下検知 | BroadcastOptimizationRemark()呼び出し | 最適化の機会損失を検知した場合 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| リスナー未登録 | リスナーリストが空の場合、ForEachListenerは何も行わずOkStatusを返す |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[XLAイベント発生] --> B[BroadcastXlaActivity / BroadcastOptimizationRemark]
    B --> C[GetXlaActivityListenerList]
    C --> D[ReaderMutexLock取得]
    D --> E[ForEachListener]
    E --> F{リスナーあり?}
    F -->|Yes| G[listener->Listen イベント]
    G --> H{Status OK?}
    H -->|Yes| I[次のリスナーへ]
    H -->|No| J[エラーStatus返却]
    I --> F
    F -->|No| K[OkStatus返却]
    J --> L[終了]
    K --> L
```

## データベース参照・更新仕様

### 参照テーブル一覧

本コンポーネントはデータベースを参照しない。

### 更新テーブル一覧

なし。リスナーの実装次第でテレメトリ送信やログ書き出しが行われるが、XlaActivityListener自体はデータ永続化を行わない。

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナーエラー | Listen()がエラーStatusを返した場合 | TF_RETURN_IF_ERRORにより即座にエラーStatusを返却し、残りのリスナーへの通知を中断 |
| Flush失敗 | プログラム終了時のFlush()でエラーが発生 | FlushAllListeners()内でCHECK(s.ok())により検査。失敗時はプロセスをクラッシュさせる |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 0（リトライなし） |
| リトライ間隔 | - |
| リトライ対象エラー | - |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 制限 | なし（イベント発生ごとにブロードキャスト） |

### 配信時間帯

制限なし。XLAの各処理フェーズで発生する。

## セキュリティ考慮事項

- XlaAutoClusteringActivityやXlaJitCompilationActivityにはモデルのグラフ構造に関する情報が含まれる可能性がある
- リスナーの登録はプロセス内のC++コードからのみ可能であり、外部からのリスナー注入リスクは低い
- リスナーはスレッドセーフである必要がある（クラスコメントに明記）

## 備考

- プログラム終了時にstd::atexit経由でFlushAllListeners()が呼ばれる（行36-37）
- XlaActivityListenerListはグローバルシングルトンとしてヒープに確保される（メモリリークは意図的）
- BroadcastOptimizationRemarkにはVLOG(2)でデバッグ出力が含まれる（行79）
- リスナーの登録にはWriterMutexLockが使用され、ブロードキャストにはReaderMutexLockが使用される（読み書きロック）

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Protocol Buffersで定義されたイベントメッセージ型を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | xla_activity.proto | `tensorflow/compiler/jit/xla_activity.proto` | XlaAutoClusteringActivity, XlaJitCompilationActivity, XlaOptimizationRemarkの各メッセージ定義 |
| 1-2 | xla_activity.pb.h | `tensorflow/compiler/jit/xla_activity.pb.h`（生成ファイル） | protobufから生成されたC++クラス |

**読解のコツ**: .protoファイルを先に読み、各フィールドの意味を理解してからC++コードを読む。

#### Step 2: エントリーポイントを理解する

ブロードキャスト関数群の実装を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | xla_activity_listener.h | `tensorflow/compiler/jit/xla_activity_listener.h` | BroadcastXlaActivity(), BroadcastOptimizationRemark(), XlaActivityListenerクラス、RegisterXlaActivityListener()の宣言（行25-74） |

**主要処理フロー**:
1. **行25-26**: BroadcastXlaActivity(XlaAutoClusteringActivity) - 自動クラスタリングイベントのブロードキャスト
2. **行29-30**: BroadcastXlaActivity(XlaJitCompilationActivity) - JITコンパイルイベントのブロードキャスト
3. **行33-34**: BroadcastOptimizationRemark(XlaOptimizationRemark) - 最適化リマークのブロードキャスト
4. **行39-41**: BroadcastOptimizationRemark(Warning, string) - 簡易版リマークブロードキャスト
5. **行49-70**: XlaActivityListenerクラス - Listen()仮想メソッド3種とFlush()
6. **行74**: RegisterXlaActivityListener() - リスナー登録関数

#### Step 3: 実装の詳細を理解する

ブロードキャスト関数とリスナー管理の実装を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | xla_activity_listener.cc | `tensorflow/compiler/jit/xla_activity_listener.cc` | リスナーリスト管理、ForEachListenerテンプレート関数、各Broadcast関数の実装（行1-105） |

**主要処理フロー**:
- **行26-30**: XlaActivityListenerList構造体 - mutexとlistenerベクタ
- **行34-39**: GetXlaActivityListenerList() - グローバルシングルトン取得、atexit登録
- **行41-52**: ForEachListener() - ReaderMutexLockで保護しながら全リスナーを巡回
- **行54-60**: FlushAllListeners() - 全リスナーのFlush()を呼び出し
- **行63-75**: BroadcastXlaActivity() - 2つのオーバーロード
- **行77-92**: BroadcastOptimizationRemark() - VLOG出力付き
- **行93-99**: RegisterXlaActivityListener() - WriterMutexLockでリスナー追加

### プログラム呼び出し階層図

```
[XLAコンパイラパイプライン]
    |
    +-- BroadcastXlaActivity(XlaAutoClusteringActivity)
    |       +-- ForEachListener()
    |               +-- ReaderMutexLock(listener_list->mutex)
    |               +-- listener->Listen(auto_clustering_activity) [各リスナー]
    |
    +-- BroadcastXlaActivity(XlaJitCompilationActivity)
    |       +-- ForEachListener()
    |               +-- listener->Listen(jit_compilation_activity) [各リスナー]
    |
    +-- BroadcastOptimizationRemark(XlaOptimizationRemark)
            +-- VLOG(2) << optimization_remark.DebugString()
            +-- ForEachListener()
                    +-- listener->Listen(optimization_remark) [各リスナー]

[プログラム終了時]
std::atexit()
    +-- FlushAllListeners()
            +-- ForEachListener()
                    +-- listener->Flush() [各リスナー]
```

### データフロー図

```
[入力]                         [処理]                          [出力]

XlaAutoClusteringActivity ---> BroadcastXlaActivity() -------> listener->Listen()
(protobuf)                     ForEachListener()               [各登録リスナー]

XlaJitCompilationActivity ---> BroadcastXlaActivity() -------> listener->Listen()
(protobuf)                     ForEachListener()               [各登録リスナー]

XlaOptimizationRemark -------> BroadcastOptimizationRemark() -> listener->Listen()
(protobuf)                     ForEachListener()               [各登録リスナー]
                               VLOG(2)出力                      -> ログ出力
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| xla_activity_listener.h | `tensorflow/compiler/jit/xla_activity_listener.h` | ソース | XlaActivityListenerインターフェース定義、Broadcast関数宣言 |
| xla_activity_listener.cc | `tensorflow/compiler/jit/xla_activity_listener.cc` | ソース | リスナー管理とブロードキャストの実装 |
| xla_activity.proto | `tensorflow/compiler/jit/xla_activity.proto` | 設定 | イベントメッセージのProtocol Buffers定義 |
| xla_activity.pb.h | `tensorflow/compiler/jit/xla_activity.pb.h` | 生成 | protobufから生成されたC++ヘッダ |
