# 通知設計書 6-BackupAndRestore

## 概要

本ドキュメントは、TensorFlow Kerasの`BackupAndRestore`コールバックに関する通知設計書である。`BackupAndRestore`はトレーニング状態のバックアップ・復元を管理し、障害耐性を提供する。

### 本通知の処理概要

`BackupAndRestore`は、トレーニング中にエポック終了時点のモデル状態をバックアップし、障害発生後のトレーニング再開時に最後のバックアップから復元する。TF `CheckpointManager`を基盤として一時チェックポイントを管理する。

**業務上の目的・背景**：大規模モデルのトレーニングは数時間から数日にわたることがあり、ハードウェア障害やプロセスの予期しない終了が発生する可能性がある。`BackupAndRestore`はこうした障害からの自動復旧を可能にし、計算リソースの無駄を最小限に抑える。

**通知の送信タイミング**：`on_train_begin`で前回のバックアップからの復元を実行し、各エポック終了時（`on_epoch_end`）に現在のトレーニング状態をバックアップする。`on_train_end`でバックアップファイルを削除する。

**通知の受信者**：ファイルシステム（バックアップチェックポイントの書き出し・読み込み）およびトレーニングループ（復元されたエポック番号による再開位置の通知）。

**通知内容の概要**：モデルの重み、オプティマイザの状態、現在のエポック番号を含むトレーニング状態全体のバックアップ。

**期待されるアクション**：障害発生後に`model.fit()`を再実行すると、自動的にバックアップから復元され、中断したエポックからトレーニングが再開される。

## 通知種別

ファイル出力通知（バックアップチェックポイントの保存・復元）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 高（障害耐性） |
| リトライ | 無し（バックアップ自体がリトライ機構） |

### 送信先決定ロジック

`backup_dir`引数で指定されたディレクトリにバックアップチェックポイントを保存する。分散トレーニング時は`WorkerTrainingState`が適切なパス管理を行う。

## 通知テンプレート

### メール通知の場合

該当なし

### 本文テンプレート

```
（標準出力への明示的なメッセージ出力なし。内部的にCheckpointManagerのログが出力される可能性あり）
```

### 添付ファイル

| ファイル名 | 形式 | 条件 | 説明 |
|----------|------|------|------|
| バックアップチェックポイント | TF Checkpoint | エポック終了時 | モデル・オプティマイザ・エポック状態 |

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| backup_dir | バックアップ保存先ディレクトリ | コンストラクタ引数 | Yes |
| epoch | 現在のエポック番号 | on_epoch_endの引数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| トレーニング開始 | on_train_begin | 常に発火 | WorkerTrainingState初期化・復元 |
| エポック終了 | on_epoch_end | 常に発火 | バックアップ保存 |
| トレーニング終了 | on_train_end | 常に発火 | バックアップファイル削除 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| Eager実行でない場合 | コンストラクタでValueError発生（Graph modeはサポート外） |
| サポート外のStrategy | on_train_beginでNotImplementedError発生 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[on_train_begin] --> B[WorkerTrainingState初期化]
    B --> C[前回バックアップ存在確認]
    C -->|存在する| D[バックアップから復元]
    C -->|存在しない| E[新規トレーニング開始]
    D --> F[エポックループ]
    E --> F
    F --> G[on_epoch_end]
    G --> H[_training_state.back_up: バックアップ保存]
    H --> I{次のエポック?}
    I -->|Yes| G
    I -->|No| J[on_train_end]
    J --> K[_training_state.delete_backup: バックアップ削除]
    K --> L[_training_state クリーンアップ]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（ファイルシステムのみ使用）

### 更新テーブル一覧

該当なし（ファイルシステムへの書き出し）

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ValueError | Eager実行でない場合 | コンストラクタでraiseし、ModelCheckpoint使用を案内 |
| ValueError | tf.function内で呼ばれた場合 | コンストラクタでraiseし、外部での呼び出しを案内 |
| NotImplementedError | サポート外のDistribution Strategy使用時 | on_train_beginでraiseし、サポート対象Strategyを案内 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 該当なし（バックアップ自体がリトライ機構として機能） |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | エポック数に依存 |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

- バックアップディレクトリにはモデルの重みとオプティマイザ状態が含まれるため、適切なアクセス権限管理が必要
- `backup_dir`は他のコールバック（ModelCheckpoint等）と共用してはならない

## 備考

- サポート対象Strategy: MirroredStrategy, CollectiveAllReduceStrategy (MultiWorkerMirroredStrategy), TPUStrategy, TPUStrategyV2, ParameterServerStrategyV2
- トレーニング正常完了時にはバックアップが自動削除される
- `_chief_worker_only = False`により全ワーカーが`on_train_begin`で復元を実行
- `_supports_tf_logs = True`が設定されている

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `BackupAndRestore.__init__`（行1644-1667）。backup_dir、_supported_strategies、Eager実行チェック |
| 1-2 | worker_training_state.py | `tensorflow/python/keras/distribute/worker_training_state.py` | `WorkerTrainingState`クラス。バックアップ・復元の実装詳細 |

**読解のコツ**: `BackupAndRestore`自体のロジックはシンプルで、実際のバックアップ・復元処理は`WorkerTrainingState`に委譲されている。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `on_train_begin`（行1669-1684）。Strategy検証とWorkerTrainingState初期化・復元 |
| 2-2 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `on_epoch_end`（行1696-1698）。`_training_state.back_up(epoch)`呼び出し |
| 2-3 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `on_train_end`（行1686-1694）。バックアップ削除とクリーンアップ |

**主要処理フロー**:
1. **行1674-1680**: Distribution Strategy検証
2. **行1681-1682**: `WorkerTrainingState`初期化
3. **行1684**: `self._training_state.restore()` で前回バックアップから復元
4. **行1698**: `self._training_state.back_up(epoch)` でバックアップ保存
5. **行1690**: `self._training_state.delete_backup()` でバックアップ削除

### プログラム呼び出し階層図

```
model.fit()
    |
    +-- CallbackList.on_train_begin()
    |       +-- BackupAndRestore.on_train_begin()
    |               +-- Strategy検証
    |               +-- WorkerTrainingState(model, backup_dir)
    |               +-- _training_state.restore()
    |
    +-- [エポックループ]
    |       +-- CallbackList.on_epoch_end()
    |               +-- BackupAndRestore.on_epoch_end()
    |                       +-- _training_state.back_up(epoch)
    |
    +-- CallbackList.on_train_end()
            +-- BackupAndRestore.on_train_end()
                    +-- _training_state.delete_backup()
                    +-- del self._training_state
```

### データフロー図

```
[入力]                     [処理]                              [出力]

backup_dir             ---> WorkerTrainingState.__init__() ---> CheckpointManager設定
model状態

前回バックアップ       ---> _training_state.restore()      ---> モデル状態復元
                                                                 エポック番号復元

現在のエポック         ---> _training_state.back_up(epoch) ---> バックアップファイル
model状態                                                        backup_dir/ckpt-N

トレーニング完了       ---> _training_state.delete_backup() --> バックアップファイル削除
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| callbacks.py | `tensorflow/python/keras/callbacks.py` | ソース | BackupAndRestoreクラス定義 |
| worker_training_state.py | `tensorflow/python/keras/distribute/worker_training_state.py` | ソース | WorkerTrainingStateクラス（バックアップ・復元実装） |
| distributed_file_utils.py | `tensorflow/python/keras/distribute/distributed_file_utils.py` | ソース | 分散ファイルパス管理 |
