# 通知設計書 9-LearningRateScheduler

## 概要

本ドキュメントは、TensorFlow Kerasの`LearningRateScheduler`コールバックに関する通知設計書である。`LearningRateScheduler`はエポック開始時にユーザ定義のスケジュール関数に基づき学習率を更新し、適用を通知する。

### 本通知の処理概要

`LearningRateScheduler`は、各エポック開始時にユーザが指定したスケジュール関数を呼び出し、返された値で学習率を更新するコールバックである。エポック終了時には現在の学習率を`logs`辞書に記録する。

**業務上の目的・背景**：学習率はモデルのトレーニングにおいて最も重要なハイパーパラメータの一つであり、トレーニングの進行に合わせて動的に変更することで収束速度と最終性能を向上させることができる。`LearningRateScheduler`はこの学習率スケジューリングをシンプルな関数インターフェースで実現する。

**通知の送信タイミング**：各エポック開始時（`on_epoch_begin`）にスケジュール関数を呼び出して学習率を更新する。各エポック終了時（`on_epoch_end`）に現在の学習率を`logs['lr']`に記録する。

**通知の受信者**：モデルのオプティマイザ（`model.optimizer.lr`属性の更新）。`verbose > 0`の場合は標準出力にもメッセージを出力。

**通知内容の概要**：更新後の学習率の値。

**期待されるアクション**：オプティマイザが更新された学習率を使用して以降のバッチ処理を行う。

## 通知種別

アプリ内通知（オプティマイザの学習率属性更新 + 標準出力メッセージ）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 高 |
| リトライ | 無し |

### 送信先決定ロジック

常にトレーニング中のモデルのオプティマイザ（`self.model.optimizer.lr`）に対して学習率を設定する。

## 通知テンプレート

### メール通知の場合

該当なし

### 本文テンプレート

```
# verbose > 0の場合
Epoch 00005: LearningRateScheduler setting learning rate to 0.001.
```

### 添付ファイル

該当なし

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| schedule | 学習率スケジュール関数 | コンストラクタ引数 | Yes |
| verbose | メッセージ出力の有無 | コンストラクタ引数（デフォルト: 0） | No |
| epoch | エポック番号 | on_epoch_beginの引数 | Yes |
| lr | 現在の学習率 | model.optimizer.lr | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| エポック開始 | on_epoch_begin | 常に発火 | スケジュール関数呼び出し・学習率更新 |
| エポック終了 | on_epoch_end | 常に発火 | logs['lr']に現在の学習率を記録 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | 常にスケジュール関数が呼び出される |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[on_epoch_begin] --> B{optimizer.lrが存在?}
    B -->|No| C[ValueError: lr属性なし]
    B -->|Yes| D[現在の学習率を取得]
    D --> E[schedule関数呼び出し: lr = schedule(epoch, lr)]
    E --> F{TypeError発生?}
    F -->|Yes| G[旧API: lr = schedule(epoch)]
    F -->|No| H[戻り値型チェック]
    G --> H
    H --> I{float/Tensor?}
    I -->|No| J[ValueError: float必須]
    I -->|Yes| K[backend.set_value: 学習率設定]
    K --> L{verbose > 0?}
    L -->|Yes| M[学習率メッセージ出力]
    L -->|No| N[on_epoch_end]
    M --> N
    N --> O[logs['lr'] = 現在の学習率]
```

## データベース参照・更新仕様

### 参照テーブル一覧

該当なし（インメモリ処理のみ）

### 更新テーブル一覧

該当なし

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| ValueError | optimizer.lr属性が存在しない | on_epoch_beginでValueErrorをraise |
| ValueError | schedule関数の戻り値がfloat/Tensorでない | on_epoch_beginでValueErrorをraise |
| ValueError | Tensorの戻り値がfloat型でない | on_epoch_beginでValueErrorをraise |
| TypeError | schedule関数が旧API（引数1つ）の場合 | try-exceptで旧API呼び出しにフォールバック |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | 該当なし |
| リトライ間隔 | 該当なし |
| リトライ対象エラー | 該当なし |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | エポック数に依存 |
| 1日あたり上限 | 制限なし |

### 配信時間帯

制限なし

## セキュリティ考慮事項

特になし。学習率値のみを扱う。

## 備考

- schedule関数は`(epoch, lr) -> new_lr`のシグネチャを推奨。旧API`(epoch) -> new_lr`も後方互換としてサポート
- on_epoch_endで`logs['lr']`に学習率を記録するため、他のコールバック（History等）で学習率の推移を追跡可能
- TensorFlowのLearningRateSchedule（オプティマイザに直接組み込むスケジューラ）とは異なるアプローチ

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `LearningRateScheduler.__init__`（行1943-1946）。schedule関数とverboseフラグ |

**読解のコツ**: `schedule`は`(epoch_index, current_lr) -> new_lr`のシグネチャを持つ関数オブジェクト。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `on_epoch_begin`（行1948-1964）。学習率更新の中核処理 |
| 2-2 | callbacks.py | `tensorflow/python/keras/callbacks.py` | `on_epoch_end`（行1966-1968）。logs['lr']への記録 |

**主要処理フロー**:
1. **行1949-1950**: `optimizer.lr`属性の存在チェック
2. **行1951-1952**: 現在の学習率取得と新API呼び出し `self.schedule(epoch, lr)`
3. **行1953-1955**: TypeError時の旧API呼び出し `self.schedule(epoch)`
4. **行1956-1960**: 戻り値の型チェック（float/Tensor）
5. **行1961**: `backend.set_value(self.model.optimizer.lr, ...)` で学習率設定
6. **行1968**: `logs['lr'] = backend.get_value(self.model.optimizer.lr)` で記録

### プログラム呼び出し階層図

```
model.fit()
    |
    +-- [エポックループ]
            +-- CallbackList.on_epoch_begin()
            |       +-- LearningRateScheduler.on_epoch_begin(epoch, logs)
            |               +-- backend.get_value(model.optimizer.lr)  # 現在LR取得
            |               +-- self.schedule(epoch, lr)               # 新LR計算
            |               +-- backend.set_value(model.optimizer.lr, new_lr)  # LR設定
            |               +-- print(...)  # verbose > 0
            |
            +-- [バッチループ: 更新された学習率で学習]
            |
            +-- CallbackList.on_epoch_end()
                    +-- LearningRateScheduler.on_epoch_end(epoch, logs)
                            +-- logs['lr'] = backend.get_value(model.optimizer.lr)
```

### データフロー図

```
[入力]                     [処理]                            [出力]

epoch番号              ---> schedule(epoch, current_lr)   ---> 新しい学習率
current_lr                                                     (float/Tensor)

新しい学習率           ---> backend.set_value()           ---> model.optimizer.lr更新

model.optimizer.lr     ---> on_epoch_end                  ---> logs['lr']
                            backend.get_value()                (他コールバック参照用)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| callbacks.py | `tensorflow/python/keras/callbacks.py` | ソース | LearningRateSchedulerクラス定義 |
| backend.py | `tensorflow/python/keras/backend.py` | ソース | get_value/set_value関数 |
| learning_rate_schedule.py | `tensorflow/python/keras/optimizer_v2/learning_rate_schedule.py` | ソース | LearningRateScheduleクラス（参考：オプティマイザ組み込み型） |
