# 非機能要件定義書

## 概要

本ドキュメントは、TensorFlow v2.21.0 プロジェクトの非機能要件を定義する。TensorFlowはオープンソースの機械学習フレームワークであり、CPU/GPU/TPU上での大規模分散計算、データパイプライン処理、モデルのトレーニング・推論を提供する。本書では、プロジェクトのソースコードから読み取れる性能、可用性、セキュリティ、拡張性、保守性、運用・監視に関する非機能要件を整理する。

## 性能要件

### レスポンスタイム

| 処理種別 | 目標値 | 備考 |
| --- | --- | --- |
| Collective通信タイムアウト | 120秒以内 | `CommunicationOptions`のデフォルト`timeout_seconds=120.0`。`tf.errors.DeadlineExceededError`が発生する（`collective_util.py`） |
| Profiler Trace | ユーザー指定（duration_ms） | `profiler_client.py`の`trace()`で指定。リトライ回数はデフォルト3回（`num_tracing_attempts=3`） |
| tf.data Serviceディスパッチャータイムアウト | Worker側デフォルト1時間 | `WorkerConfig.dispatcher_timeout_ms`のデフォルトが1時間（`server_lib.py`） |
| gRPCサーバーKeep Alive | 600秒 | `context.py`の`_KEEP_ALIVE_SECS = 600` |

### スループット

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| 同時スナップショット処理数 | `worker_max_concurrent_snapshots`で設定可能（デフォルト0=無制限） | `DispatcherConfig`にて設定（`server_lib.py`） |
| Collective通信パック分割サイズ | `bytes_per_pack`で設定可能（例: 50MB） | 0の場合は自動決定。勾配のall-reduceを複数パックに分割しオーバーラップ実行（`collective_util.py`） |
| 分散トレーニング並列度 | MirroredStrategy / MultiWorkerMirroredStrategy / ParameterServerStrategy にて設定 | 複数の分散戦略をサポート（`tensorflow/python/distribute/`配下） |
| CUDA Compute Capability | デフォルト3.5, 7.0 | `configure.py`の`_DEFAULT_CUDA_COMPUTE_CAPABILITIES = '3.5,7.0'`。XLAは3.5以上が必要 |

## 可用性要件

| 項目 | 目標値 | 備考 |
| --- | --- | --- |
| tf.data Service フォールトトレラントモード | ディスパッチャーの状態をジャーナルに同期書き込みし、再起動時に復旧可能 | `DispatcherConfig.fault_tolerant_mode=True`かつ`work_dir`設定が必要（`server_lib.py`） |
| ワーカーハートビート | `heartbeat_interval_ms`で設定可能。タイムアウト時にワーカーを「missing」と判定 | `WorkerConfig.heartbeat_interval_ms`、`DispatcherConfig.worker_timeout_ms`（`server_lib.py`） |
| DTensorハートビート | 定期的にワーカー間でハートビートシグナルを交換。異常検知時に`logging.fatal`で即時停止 | `heartbeat.py`にて実装。`timeout=max(period - 10, 2)`で待機 |
| クラスタコーディネーター障害復旧 | ワーカーのトランジェントタイムアウトを一定回数まで無視し、一時的なネットワーク障害に対応 | `cluster_coordinator.py`の`_transient_timeouts_threshold`環境変数で設定 |
| 非同期チェックポイント | トレーニング中に非同期でチェックポイントを保存し、障害時のデータ損失を最小化 | `async_checkpoint_helper.py`にて実装。専用スレッドでバックグラウンド保存 |
| 障害復旧時間（RTO） | アプリケーション依存 | チェックポイントからの復旧時間はモデルサイズ・ストレージ性能に依存 |
| 目標復旧時点（RPO） | 最終チェックポイント保存時点 | 非同期チェックポイントの保存間隔による |

## セキュリティ要件

### 入力データの安全性

TensorFlowは信頼できないモデル・チェックポイント・入力データを扱う際にセキュリティリスクが伴う。以下の要件がSECURITY.mdおよびソースコードから確認される。

- **信頼できる入力フォーマット**: PNG, BMP, GIF, WAV, RAW, RAW_PADDED, CSV, PROTOフォーマットは安全とみなされる。それ以外のフォーマットはサンドボックス内で処理する必要がある
- **モデルのサンドボックス実行**: 信頼できないモデル・GraphDef・SavedModel・チェックポイントの読み込みは任意コード実行と同等のリスクがあるため、サンドボックス内で実行する必要がある
- **TensorFlow Serverのネットワーク制限**: `tf.train.Server`は内部通信専用であり、認証プロトコルを含まず、メッセージは暗号化されない。信頼できないネットワーク環境での使用は禁止

### マルチテナント環境

- テナント間の分離機構はユーザーの責任で設計・実装する必要がある
- ネットワーク分離によるデータ・モデルへの不正アクセス防止が必要
- DoS対策としてレートリミット・ACL・監視によるサーバー再起動が推奨される

### セキュリティ脆弱性管理

- `tensorflow/security/advisory/` 配下にセキュリティアドバイザリを管理（2018年以降の脆弱性情報を記録）
- `tensorflow/security/fuzzing/` 配下にファジングテストを配置
- 脆弱性報告は Google Bug Hunters を通じて行う
- 脆弱性修正は四半期リリースにバッチ処理

### コンパイル時のセキュリティ

- XLA/JAXの推奨エントリポイント経由でのモデルコンパイルは安全
- テスト・デバッグツールは信頼できないデータとの使用に注意が必要

## 拡張性要件

### マルチプラットフォームサポート

- **OS**: Linux, macOS (Darwin), Windows, Cygwin をサポート（`configure.py`のプラットフォーム検出）
- **アーキテクチャ**: x86_64, ppc64le, s390x, ARM64（Android NDK API Level 21以上でARM64サポート）
- **モバイル**: Android（NDK v19/20/21/25対応）、iOS（macOSでのTensorFlow Lite対応）

### GPUアクセラレーション

- **CUDA**: Hermetic CUDA版、cuDNN版の指定が可能。Compute Capability 3.0以上をサポート（3.5以上推奨）
- **ROCm**: AMD GPU対応。CUDAとは排他的に設定（`configure.py`）
- **NCCL**: マルチGPU間の集合通信ライブラリ

### コンパイラサポート

- **Clang**: v16/17/18対応。デフォルトのCPU/CUDAコンパイラとして選択可能
- **GCC**: 代替コンパイラとして使用可能。ppc64leではgoldリンカーを使用
- **MSVC**: Windows環境でのビルドに使用可能
- **XLA (Accelerated Linear Algebra)**: JITコンパイルによる計算最適化

### 分散戦略の拡張性

- `MirroredStrategy`: 単一マシン内のマルチGPU分散
- `MultiWorkerMirroredStrategy`: 複数マシン間の分散トレーニング
- `ParameterServerStrategy`: パラメータサーバー方式の分散トレーニング
- `tf.data.experimental.service`: 分散データパイプライン処理（DispatchServer/WorkerServerアーキテクチャ）

## 保守性要件

### コードベースの構造

- **モジュール構成**: `tensorflow/core/`, `tensorflow/python/`, `tensorflow/compiler/`, `tensorflow/lite/`, `tensorflow/distribute/` 等の機能別モジュール構成
- **Python/C++レイヤー**: PythonフロントエンドとC++バックエンドの分離。`pywrap_*` によるバインディング
- **Bazelビルドシステム**: 再現可能なビルド環境。`.tf_configure.bazelrc`による設定管理

### テスト体制

- テストサイズフィルタリング: `small`, `medium` サイズのテストをデフォルト実行（`configure.py`の`system_specific_test_config`）
- プラットフォーム固有のテストタグ: `-no_windows`, `-nomac`, `-no_gpu` 等のタグによるフィルタリング
- v1/v2 APIテストの分離: `--config=v1` / `--config=v2` によるAPI別テスト
- ファジングテスト: `tensorflow/security/fuzzing/` でのセキュリティテスト

### バージョン管理

- セマンティックバージョニング（現行: v2.21.0）
- `tf_version.bzl` でバージョン一元管理
- v1/v2 API互換レイヤー: `compat_template_v1.__init__.py` による後方互換性維持

## 運用・監視要件

### プロファイリング・モニタリング

- **TensorFlow Profiler**: gRPC経由でオンデマンドプロファイリングを実行可能（`profiler_client.py`）
  - CPU/GPU/TPU対応
  - 複数ホストの同時プロファイリング
  - TensorBoard連携によるプロファイリング結果の可視化
  - リトライ機構: `num_tracing_attempts=3`（デフォルト）
- **モニタリングAPI**: `tf.profiler.experimental.client.monitor()` による軽量パフォーマンスサマリの取得（Cloud TPU対応）
- **メモリ管理**: GPU メモリの動的成長（`memory_growth`）、メモリ制限（`memory_limit`）、メモリ使用状況の取得（`get_memory_info`）、メモリ統計のリセット（`reset_memory_stats`）が可能（`context.py`）

### ロギング

- **abseil-py (absl) logging**: プロジェクト全体で`absl.logging`を使用
- **ハートビートログ**: DTensorハートビートスレッドの状態を定期的にログ出力
- **チェックポイントメトリクス**: 非同期チェックポイントの所要時間をマイクロ秒単位で計測

### ジョブ管理

- **ジョブGC（ガベージコレクション）**: `DispatcherConfig.job_gc_check_interval_ms` / `job_gc_timeout_ms` で未使用ジョブの自動回収間隔・タイムアウトを設定可能
- **Watchdog**: `cluster_coordinator.py`にてWatchdogによるタイムアウト監視を実装
- **非同期実行**: Eager Mode（デフォルト）とGraph Modeの切り替え。同期（SYNC）/非同期（ASYNC）実行モードの選択が可能

### キャッシュ

- **関数キャッシュ**: `function_cache.py` にてトレース済み関数のキャッシュを管理。関数シグネチャに基づくディスパッチ
- **デバイス解析キャッシュ**: `context.py`の`_device_parsing_cache`でデバイス名解析結果をキャッシュ。GILとべき等性による簡易スレッドセーフ

## 備考

- 本ドキュメントはTensorFlow v2.21.0のソースコードを解析して作成された。記載されている値はソースコードのデフォルト値および設定可能なパラメータに基づく
- TensorFlowはライブラリ/フレームワークとしての性質上、多くの非機能要件はユーザーのアプリケーション設計およびデプロイメント環境に依存する
- 分散トレーニングの具体的なスループット値やレイテンシは、ハードウェア構成・ネットワーク帯域・モデル規模に大きく依存するため、具体的な数値目標は記載していない
- セキュリティアドバイザリは`tensorflow/security/advisory/`配下で継続的に更新されている
