# API設計書

## 概要

本ドキュメントは、three.js JavaScript 3Dライブラリのパブリック API（クラス、メソッド、プロパティ）の設計仕様を記載した設計書です。three.jsは従来のREST API/HTTP APIを提供するバックエンドアプリケーションではなく、フロントエンドで使用される3Dグラフィックスライブラリであるため、本書ではJavaScript APIインターフェースとして設計内容を記載します。

**バージョン**: 0.182.0

## 共通仕様

### ライブラリインポート

```javascript
// ES Modules（推奨）
import * as THREE from 'three';

// 個別インポート
import { Scene, PerspectiveCamera, WebGLRenderer } from 'three';

// WebGPU対応
import * as THREE from 'three/webgpu';
```

### 座標系

three.jsは右手座標系を使用します：
- X軸: 右方向が正
- Y軸: 上方向が正
- Z軸: 手前方向が正

### 単位系

- 距離: ワールドユニット（アプリケーションで定義）
- 角度: ラジアン（デフォルト）/ 度（一部のプロパティ）
- 色: 0x000000 - 0xFFFFFF（16進数）または Color クラス

### 共通インターフェース

| インターフェース | 説明 |
| --- | --- |
| EventDispatcher | イベントリスナーの登録・解除・発火機能を提供 |
| clone() | オブジェクトの複製を作成 |
| copy(source) | ソースオブジェクトから値をコピー |
| toJSON(meta) | JSON形式でシリアライズ |
| dispose() | リソースを解放（GPUメモリ等） |

## API一覧

| カテゴリ | クラス名 | 説明 |
| --- | --- | --- |
| Core | Object3D | 全3Dオブジェクトの基底クラス |
| Core | BufferGeometry | ジオメトリデータの格納クラス |
| Core | Raycaster | レイキャスティングユーティリティ |
| Scenes | Scene | シーングラフのルートコンテナ |
| Cameras | PerspectiveCamera | 透視投影カメラ |
| Cameras | OrthographicCamera | 正投影カメラ |
| Renderers | WebGLRenderer | WebGL 2.0ベースレンダラー |
| Objects | Mesh | ポリゴンメッシュオブジェクト |
| Objects | Line | ラインオブジェクト |
| Objects | Points | ポイントクラウドオブジェクト |
| Materials | MeshBasicMaterial | 基本マテリアル（ライティング無効） |
| Materials | MeshStandardMaterial | PBRマテリアル |
| Geometries | BoxGeometry | ボックス形状ジオメトリ |
| Geometries | SphereGeometry | 球体形状ジオメトリ |
| Lights | DirectionalLight | 平行光源 |
| Lights | PointLight | 点光源 |
| Lights | SpotLight | スポットライト |
| Loaders | TextureLoader | テクスチャローダー |
| Loaders | ObjectLoader | JSON形式オブジェクトローダー |
| Math | Vector3 | 3次元ベクトル |
| Math | Matrix4 | 4x4行列 |
| Math | Quaternion | クォータニオン |

## 各APIエンドポイント定義

### Core

#### 1. Object3D

全ての3Dオブジェクトの基底クラスです。位置、回転、スケールなどの変換や、親子関係の管理機能を提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.Object3D` |
| 継承元 | EventDispatcher |
| ファイル | `src/core/Object3D.js` |

**コンストラクタ**

```javascript
const object = new THREE.Object3D();
```

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| id | number | 自動採番 | オブジェクトID（読み取り専用） |
| uuid | string | 自動生成 | UUID（読み取り専用） |
| name | string | '' | オブジェクト名 |
| parent | Object3D | null | 親オブジェクト参照 |
| children | Array<Object3D> | [] | 子オブジェクト配列 |
| position | Vector3 | (0,0,0) | ローカル座標位置 |
| rotation | Euler | (0,0,0) | ローカル回転（オイラー角） |
| quaternion | Quaternion | (0,0,0,1) | ローカル回転（クォータニオン） |
| scale | Vector3 | (1,1,1) | ローカルスケール |
| up | Vector3 | (0,1,0) | 上方向ベクトル |
| matrix | Matrix4 | 単位行列 | ローカル変換行列 |
| matrixWorld | Matrix4 | 単位行列 | ワールド変換行列 |
| visible | boolean | true | 可視状態 |
| castShadow | boolean | false | シャドウキャスト有効 |
| receiveShadow | boolean | false | シャドウ受け取り有効 |
| frustumCulled | boolean | true | 視錐台カリング有効 |
| renderOrder | number | 0 | レンダリング順序 |
| layers | Layers | - | レイヤー設定 |
| userData | Object | {} | カスタムデータ格納用 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| add(object) | Object3D | this | 子オブジェクトを追加 |
| remove(object) | Object3D | this | 子オブジェクトを削除 |
| clear() | - | this | 全ての子オブジェクトを削除 |
| attach(object) | Object3D | this | ワールド変換を維持して子として追加 |
| getObjectById(id) | number | Object3D/undefined | IDで子孫オブジェクトを検索 |
| getObjectByName(name) | string | Object3D/undefined | 名前で子孫オブジェクトを検索 |
| traverse(callback) | Function | void | 全子孫オブジェクトに対してコールバックを実行 |
| traverseVisible(callback) | Function | void | 可視な子孫のみに対してコールバックを実行 |
| lookAt(x, y, z) | number/Vector3 | void | 指定座標を向く |
| rotateX(angle) | number | this | X軸回りに回転 |
| rotateY(angle) | number | this | Y軸回りに回転 |
| rotateZ(angle) | number | this | Z軸回りに回転 |
| translateX(distance) | number | this | X軸方向に移動 |
| translateY(distance) | number | this | Y軸方向に移動 |
| translateZ(distance) | number | this | Z軸方向に移動 |
| localToWorld(vector) | Vector3 | Vector3 | ローカル座標をワールド座標に変換 |
| worldToLocal(vector) | Vector3 | Vector3 | ワールド座標をローカル座標に変換 |
| getWorldPosition(target) | Vector3 | Vector3 | ワールド座標位置を取得 |
| getWorldQuaternion(target) | Quaternion | Quaternion | ワールド回転を取得 |
| getWorldScale(target) | Vector3 | Vector3 | ワールドスケールを取得 |
| updateMatrix() | - | void | ローカル変換行列を更新 |
| updateMatrixWorld(force) | boolean | void | ワールド変換行列を更新 |
| clone(recursive) | boolean | Object3D | オブジェクトを複製 |
| copy(source, recursive) | Object3D, boolean | this | ソースからコピー |
| toJSON(meta) | Object | Object | JSON形式でシリアライズ |

**イベント**

| イベント名 | 説明 |
| --- | --- |
| added | 親オブジェクトに追加された時に発火 |
| removed | 親オブジェクトから削除された時に発火 |
| childadded | 子オブジェクトが追加された時に発火 |
| childremoved | 子オブジェクトが削除された時に発火 |

**使用例**

```javascript
const object = new THREE.Object3D();
object.position.set(1, 2, 3);
object.rotation.set(0, Math.PI / 2, 0);
object.scale.set(2, 2, 2);

scene.add(object);

object.traverse((child) => {
  console.log(child.name);
});
```

---

#### 2. BufferGeometry

メッシュ、ライン、ポイントのジオメトリデータを格納するクラスです。頂点位置、法線、UV座標などのバッファ属性を管理します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.BufferGeometry` |
| 継承元 | EventDispatcher |
| ファイル | `src/core/BufferGeometry.js` |

**コンストラクタ**

```javascript
const geometry = new THREE.BufferGeometry();
```

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| id | number | 自動採番 | ジオメトリID（読み取り専用） |
| uuid | string | 自動生成 | UUID（読み取り専用） |
| name | string | '' | ジオメトリ名 |
| index | BufferAttribute | null | 頂点インデックス |
| attributes | Object | {} | 属性辞書 |
| morphAttributes | Object | {} | モーフターゲット属性 |
| morphTargetsRelative | boolean | false | モーフターゲットを相対値として扱う |
| groups | Array | [] | ジオメトリグループ |
| boundingBox | Box3 | null | バウンディングボックス |
| boundingSphere | Sphere | null | バウンディング球 |
| drawRange | Object | {start: 0, count: Infinity} | 描画範囲 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| setAttribute(name, attribute) | string, BufferAttribute | this | 属性を設定 |
| getAttribute(name) | string | BufferAttribute | 属性を取得 |
| deleteAttribute(name) | string | this | 属性を削除 |
| hasAttribute(name) | string | boolean | 属性の存在確認 |
| setIndex(index) | BufferAttribute/Array | this | インデックスを設定 |
| getIndex() | - | BufferAttribute | インデックスを取得 |
| addGroup(start, count, materialIndex) | number, number, number | void | グループを追加 |
| clearGroups() | - | void | グループをクリア |
| setDrawRange(start, count) | number, number | void | 描画範囲を設定 |
| computeBoundingBox() | - | void | バウンディングボックスを計算 |
| computeBoundingSphere() | - | void | バウンディング球を計算 |
| computeVertexNormals() | - | void | 頂点法線を計算 |
| normalizeNormals() | - | void | 法線を正規化 |
| center() | - | this | ジオメトリを中心に配置 |
| applyMatrix4(matrix) | Matrix4 | this | 行列変換を適用 |
| rotateX(angle) | number | this | X軸回りに回転 |
| rotateY(angle) | number | this | Y軸回りに回転 |
| rotateZ(angle) | number | this | Z軸回りに回転 |
| translate(x, y, z) | number, number, number | this | 平行移動 |
| scale(x, y, z) | number, number, number | this | スケール |
| dispose() | - | void | GPUリソースを解放 |
| clone() | - | BufferGeometry | ジオメトリを複製 |
| copy(source) | BufferGeometry | this | ソースからコピー |
| toJSON() | - | Object | JSON形式でシリアライズ |

**使用例**

```javascript
const geometry = new THREE.BufferGeometry();

const vertices = new Float32Array([
  -1.0, -1.0, 1.0,
   1.0, -1.0, 1.0,
   1.0,  1.0, 1.0
]);

geometry.setAttribute('position', new THREE.BufferAttribute(vertices, 3));
geometry.computeVertexNormals();
```

---

### Scenes

#### 3. Scene

レンダリングするオブジェクト、ライト、背景などを配置するコンテナです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.Scene` |
| 継承元 | Object3D |
| ファイル | `src/scenes/Scene.js` |

**コンストラクタ**

```javascript
const scene = new THREE.Scene();
```

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| background | Color/Texture | null | シーン背景（色またはテクスチャ） |
| environment | Texture | null | 環境マップ |
| fog | Fog/FogExp2 | null | フォグ設定 |
| backgroundBlurriness | number | 0 | 背景のぼかし度 (0-1) |
| backgroundIntensity | number | 1 | 背景の明るさ |
| backgroundRotation | Euler | (0,0,0) | 背景の回転 |
| environmentIntensity | number | 1 | 環境マップの明るさ |
| environmentRotation | Euler | (0,0,0) | 環境マップの回転 |
| overrideMaterial | Material | null | 全オブジェクトに適用するマテリアル |

**使用例**

```javascript
const scene = new THREE.Scene();
scene.background = new THREE.Color(0x000000);
scene.fog = new THREE.Fog(0x000000, 1, 1000);

const mesh = new THREE.Mesh(geometry, material);
scene.add(mesh);
```

---

### Cameras

#### 4. PerspectiveCamera

透視投影カメラ。人間の目に近い遠近感のあるレンダリングを行います。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.PerspectiveCamera` |
| 継承元 | Camera -> Object3D |
| ファイル | `src/cameras/PerspectiveCamera.js` |

**コンストラクタ**

```javascript
const camera = new THREE.PerspectiveCamera(fov, aspect, near, far);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| fov | number | 50 | 垂直視野角（度） |
| aspect | number | 1 | アスペクト比 |
| near | number | 0.1 | ニアクリッピング面 |
| far | number | 2000 | ファークリッピング面 |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| fov | number | 50 | 垂直視野角（度） |
| aspect | number | 1 | アスペクト比 |
| near | number | 0.1 | ニアクリッピング面 |
| far | number | 2000 | ファークリッピング面 |
| zoom | number | 1 | ズーム倍率 |
| focus | number | 10 | 焦点距離（DoF用） |
| filmGauge | number | 35 | フィルムサイズ（mm） |
| filmOffset | number | 0 | 水平オフセット |
| view | Object | null | ビューオフセット設定 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| setFocalLength(focalLength) | number | void | 焦点距離からFOVを設定 |
| getFocalLength() | - | number | 焦点距離を取得 |
| getEffectiveFOV() | - | number | 有効FOVを取得（zoom考慮） |
| getFilmWidth() | - | number | フィルム幅を取得 |
| getFilmHeight() | - | number | フィルム高さを取得 |
| getViewBounds(distance, minTarget, maxTarget) | number, Vector2, Vector2 | void | ビュー境界を取得 |
| getViewSize(distance, target) | number, Vector2 | Vector2 | ビューサイズを取得 |
| setViewOffset(fullWidth, fullHeight, x, y, width, height) | number x6 | void | ビューオフセットを設定 |
| clearViewOffset() | - | void | ビューオフセットをクリア |
| updateProjectionMatrix() | - | void | 投影行列を更新 |

**使用例**

```javascript
const camera = new THREE.PerspectiveCamera(75, window.innerWidth / window.innerHeight, 0.1, 1000);
camera.position.set(0, 0, 5);
camera.lookAt(0, 0, 0);

// リサイズ対応
window.addEventListener('resize', () => {
  camera.aspect = window.innerWidth / window.innerHeight;
  camera.updateProjectionMatrix();
});
```

---

### Renderers

#### 5. WebGLRenderer

WebGL 2.0を使用してシーンをレンダリングするレンダラーです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.WebGLRenderer` |
| 継承元 | - |
| ファイル | `src/renderers/WebGLRenderer.js` |

**コンストラクタ**

```javascript
const renderer = new THREE.WebGLRenderer(parameters);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| canvas | HTMLCanvasElement | 自動作成 | 描画先Canvas要素 |
| context | WebGLRenderingContext | null | 既存のWebGLコンテキスト |
| alpha | boolean | false | 透過背景を有効化 |
| antialias | boolean | false | アンチエイリアスを有効化 |
| depth | boolean | true | 深度バッファを有効化 |
| stencil | boolean | false | ステンシルバッファを有効化 |
| premultipliedAlpha | boolean | true | プリマルチプライドアルファ |
| preserveDrawingBuffer | boolean | false | 描画バッファを保持 |
| powerPreference | string | 'default' | GPUパフォーマンス設定 |
| failIfMajorPerformanceCaveat | boolean | false | パフォーマンス問題時に失敗 |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| domElement | HTMLCanvasElement | - | 描画先Canvas要素 |
| autoClear | boolean | true | 自動クリア有効 |
| autoClearColor | boolean | true | カラーバッファ自動クリア |
| autoClearDepth | boolean | true | 深度バッファ自動クリア |
| autoClearStencil | boolean | true | ステンシルバッファ自動クリア |
| shadowMap | WebGLShadowMap | - | シャドウマップ設定 |
| toneMapping | number | NoToneMapping | トーンマッピング方式 |
| toneMappingExposure | number | 1 | トーンマッピング露出 |
| outputColorSpace | string | SRGBColorSpace | 出力カラースペース |
| info | Object | - | レンダリング統計情報 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| render(scene, camera) | Scene, Camera | void | シーンをレンダリング |
| setSize(width, height, updateStyle) | number, number, boolean | void | レンダラーサイズを設定 |
| getSize(target) | Vector2 | Vector2 | レンダラーサイズを取得 |
| setPixelRatio(value) | number | void | ピクセル比を設定 |
| getPixelRatio() | - | number | ピクセル比を取得 |
| setClearColor(color, alpha) | Color, number | void | クリアカラーを設定 |
| getClearColor(target) | Color | Color | クリアカラーを取得 |
| getClearAlpha() | - | number | クリアアルファを取得 |
| clear(color, depth, stencil) | boolean, boolean, boolean | void | バッファをクリア |
| setRenderTarget(renderTarget) | WebGLRenderTarget | void | レンダーターゲットを設定 |
| getRenderTarget() | - | WebGLRenderTarget | レンダーターゲットを取得 |
| setAnimationLoop(callback) | Function | void | アニメーションループを設定 |
| dispose() | - | void | リソースを解放 |

**使用例**

```javascript
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
renderer.setPixelRatio(window.devicePixelRatio);
renderer.shadowMap.enabled = true;
document.body.appendChild(renderer.domElement);

function animate() {
  renderer.render(scene, camera);
}
renderer.setAnimationLoop(animate);
```

---

### Objects

#### 6. Mesh

三角形ポリゴンベースの3Dオブジェクトです。ジオメトリとマテリアルの組み合わせで構成されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.Mesh` |
| 継承元 | Object3D |
| ファイル | `src/objects/Mesh.js` |

**コンストラクタ**

```javascript
const mesh = new THREE.Mesh(geometry, material);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| geometry | BufferGeometry | BufferGeometry() | メッシュジオメトリ |
| material | Material/Array<Material> | MeshBasicMaterial() | メッシュマテリアル |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| geometry | BufferGeometry | - | メッシュジオメトリ |
| material | Material/Array<Material> | - | メッシュマテリアル |
| morphTargetInfluences | Array<number> | undefined | モーフターゲット影響度 |
| morphTargetDictionary | Object | undefined | モーフターゲット辞書 |
| count | number | 1 | インスタンス数（WebGPU用） |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| updateMorphTargets() | - | void | モーフターゲットを更新 |
| getVertexPosition(index, target) | number, Vector3 | Vector3 | 頂点位置を取得 |
| raycast(raycaster, intersects) | Raycaster, Array | void | レイキャスト実行 |

**使用例**

```javascript
const geometry = new THREE.BoxGeometry(1, 1, 1);
const material = new THREE.MeshStandardMaterial({ color: 0x00ff00 });
const mesh = new THREE.Mesh(geometry, material);

mesh.position.set(0, 0, 0);
mesh.castShadow = true;
mesh.receiveShadow = true;

scene.add(mesh);
```

---

### Materials

#### 7. MeshBasicMaterial

ライティングの影響を受けない基本マテリアルです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.MeshBasicMaterial` |
| 継承元 | Material |
| ファイル | `src/materials/MeshBasicMaterial.js` |

**コンストラクタ**

```javascript
const material = new THREE.MeshBasicMaterial(parameters);
```

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| color | Color | (1,1,1) | ディフューズカラー |
| map | Texture | null | カラーマップ |
| lightMap | Texture | null | ライトマップ |
| lightMapIntensity | number | 1 | ライトマップ強度 |
| aoMap | Texture | null | アンビエントオクルージョンマップ |
| aoMapIntensity | number | 1 | AOマップ強度 |
| specularMap | Texture | null | スペキュラマップ |
| alphaMap | Texture | null | アルファマップ |
| envMap | Texture | null | 環境マップ |
| envMapRotation | Euler | (0,0,0) | 環境マップ回転 |
| combine | number | MultiplyOperation | 環境マップ合成方式 |
| reflectivity | number | 1 | 反射率 |
| refractionRatio | number | 0.98 | 屈折率 |
| wireframe | boolean | false | ワイヤーフレーム表示 |
| wireframeLinewidth | number | 1 | ワイヤーフレーム線幅 |
| fog | boolean | true | フォグ適用 |

**使用例**

```javascript
const material = new THREE.MeshBasicMaterial({
  color: 0xff0000,
  wireframe: false,
  transparent: true,
  opacity: 0.8
});
```

---

### Geometries

#### 8. BoxGeometry

直方体形状のジオメトリです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.BoxGeometry` |
| 継承元 | BufferGeometry |
| ファイル | `src/geometries/BoxGeometry.js` |

**コンストラクタ**

```javascript
const geometry = new THREE.BoxGeometry(width, height, depth, widthSegments, heightSegments, depthSegments);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| width | number | 1 | X軸方向の幅 |
| height | number | 1 | Y軸方向の高さ |
| depth | number | 1 | Z軸方向の奥行き |
| widthSegments | number | 1 | 幅方向の分割数 |
| heightSegments | number | 1 | 高さ方向の分割数 |
| depthSegments | number | 1 | 奥行き方向の分割数 |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| parameters | Object | コンストラクタパラメータのコピー |

**静的メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| fromJSON(data) | Object | BoxGeometry | JSONからインスタンスを生成 |

**使用例**

```javascript
const geometry = new THREE.BoxGeometry(1, 1, 1, 2, 2, 2);
const material = new THREE.MeshStandardMaterial({ color: 0x00ff00 });
const cube = new THREE.Mesh(geometry, material);
scene.add(cube);
```

---

### Lights

#### 9. DirectionalLight

平行光源。太陽光のシミュレーションに使用されます。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.DirectionalLight` |
| 継承元 | Light -> Object3D |
| ファイル | `src/lights/DirectionalLight.js` |

**コンストラクタ**

```javascript
const light = new THREE.DirectionalLight(color, intensity);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| color | number/Color/string | 0xffffff | ライトカラー |
| intensity | number | 1 | ライト強度 |

**プロパティ**

| プロパティ名 | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| color | Color | (1,1,1) | ライトカラー |
| intensity | number | 1 | ライト強度 |
| target | Object3D | Object3D() | ライトが向くターゲット |
| shadow | DirectionalLightShadow | - | シャドウ設定 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| dispose() | - | void | リソースを解放 |

**使用例**

```javascript
const directionalLight = new THREE.DirectionalLight(0xffffff, 1);
directionalLight.position.set(5, 10, 7.5);
directionalLight.castShadow = true;
directionalLight.shadow.mapSize.width = 2048;
directionalLight.shadow.mapSize.height = 2048;

scene.add(directionalLight);
scene.add(directionalLight.target);
```

---

### Loaders

#### 10. TextureLoader

テクスチャを読み込むローダーです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.TextureLoader` |
| 継承元 | Loader |
| ファイル | `src/loaders/TextureLoader.js` |

**コンストラクタ**

```javascript
const loader = new THREE.TextureLoader(manager);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| manager | LoadingManager | DefaultLoadingManager | ローディングマネージャー |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| load(url, onLoad, onProgress, onError) | string, Function, Function, Function | Texture | テクスチャを読み込み |
| loadAsync(url, onProgress) | string, Function | Promise<Texture> | 非同期でテクスチャを読み込み |
| setCrossOrigin(value) | string | this | CORSを設定 |
| setPath(path) | string | this | ベースパスを設定 |

**使用例**

```javascript
const loader = new THREE.TextureLoader();

// コールバック方式
loader.load(
  'texture.jpg',
  (texture) => {
    const material = new THREE.MeshBasicMaterial({ map: texture });
  },
  undefined,
  (error) => {
    console.error('An error occurred:', error);
  }
);

// async/await方式
const texture = await loader.loadAsync('texture.jpg');
```

---

### Math

#### 11. Vector3

3次元ベクトルを表すクラスです。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | `THREE.Vector3` |
| ファイル | `src/math/Vector3.js` |

**コンストラクタ**

```javascript
const vector = new THREE.Vector3(x, y, z);
```

| パラメータ | 型 | デフォルト | 説明 |
| --- | --- | --- | --- |
| x | number | 0 | X成分 |
| y | number | 0 | Y成分 |
| z | number | 0 | Z成分 |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| x | number | X成分 |
| y | number | Y成分 |
| z | number | Z成分 |

**主要メソッド**

| メソッド名 | 引数 | 戻り値 | 説明 |
| --- | --- | --- | --- |
| set(x, y, z) | number, number, number | this | 値を設定 |
| setScalar(scalar) | number | this | 全成分を同じ値に設定 |
| copy(v) | Vector3 | this | ベクトルをコピー |
| add(v) | Vector3 | this | ベクトルを加算 |
| addScalar(s) | number | this | スカラーを加算 |
| sub(v) | Vector3 | this | ベクトルを減算 |
| multiply(v) | Vector3 | this | 成分ごとに乗算 |
| multiplyScalar(s) | number | this | スカラーで乗算 |
| divide(v) | Vector3 | this | 成分ごとに除算 |
| divideScalar(s) | number | this | スカラーで除算 |
| dot(v) | Vector3 | number | 内積を計算 |
| cross(v) | Vector3 | this | 外積を計算 |
| length() | - | number | ベクトルの長さを取得 |
| lengthSq() | - | number | ベクトルの長さの2乗を取得 |
| normalize() | - | this | 正規化 |
| distanceTo(v) | Vector3 | number | 距離を計算 |
| distanceToSquared(v) | Vector3 | number | 距離の2乗を計算 |
| lerp(v, alpha) | Vector3, number | this | 線形補間 |
| equals(v) | Vector3 | boolean | 等価性を判定 |
| clone() | - | Vector3 | ベクトルを複製 |
| toArray(array, offset) | Array, number | Array | 配列に変換 |
| fromArray(array, offset) | Array, number | this | 配列から生成 |
| applyMatrix4(m) | Matrix4 | this | 4x4行列を適用 |
| applyQuaternion(q) | Quaternion | this | クォータニオンを適用 |

**使用例**

```javascript
const a = new THREE.Vector3(1, 2, 3);
const b = new THREE.Vector3(4, 5, 6);

const sum = a.clone().add(b);
const dot = a.dot(b);
const cross = a.clone().cross(b);
const distance = a.distanceTo(b);
const normalized = a.clone().normalize();
```

---

## 典型的な使用パターン

### 基本的なシーンセットアップ

```javascript
import * as THREE from 'three';

// シーン
const scene = new THREE.Scene();
scene.background = new THREE.Color(0x333333);

// カメラ
const camera = new THREE.PerspectiveCamera(
  75,
  window.innerWidth / window.innerHeight,
  0.1,
  1000
);
camera.position.z = 5;

// レンダラー
const renderer = new THREE.WebGLRenderer({ antialias: true });
renderer.setSize(window.innerWidth, window.innerHeight);
renderer.setPixelRatio(window.devicePixelRatio);
document.body.appendChild(renderer.domElement);

// メッシュ
const geometry = new THREE.BoxGeometry(1, 1, 1);
const material = new THREE.MeshStandardMaterial({ color: 0x00ff00 });
const cube = new THREE.Mesh(geometry, material);
scene.add(cube);

// ライト
const light = new THREE.DirectionalLight(0xffffff, 1);
light.position.set(5, 5, 5);
scene.add(light);

const ambientLight = new THREE.AmbientLight(0x404040);
scene.add(ambientLight);

// アニメーションループ
function animate() {
  cube.rotation.x += 0.01;
  cube.rotation.y += 0.01;
  renderer.render(scene, camera);
}
renderer.setAnimationLoop(animate);

// リサイズ対応
window.addEventListener('resize', () => {
  camera.aspect = window.innerWidth / window.innerHeight;
  camera.updateProjectionMatrix();
  renderer.setSize(window.innerWidth, window.innerHeight);
});
```

## 備考

- three.jsはWebGL 2.0以上が必須です（WebGL 1.0はr163以降サポート終了）
- WebGPU対応が進行中であり、`three/webgpu`からインポート可能
- TypeScript型定義は`@types/three`パッケージで提供
- 詳細なAPIドキュメントは公式サイト（https://threejs.org/docs/）を参照
