# three.js

## 概要

three.jsは、ブラウザ上で3Dグラフィックスを簡単に作成するための軽量で汎用的なJavaScript 3Dライブラリです。WebGLおよびWebGPUレンダラーを標準搭載し、SVGやCSS3Dレンダラーもアドオンとして利用可能です。クロスブラウザ対応で、VR/AR（WebXR）にも対応しています。

## 主な機能

| 機能カテゴリ | 説明 |
| --- | --- |
| レンダリング | WebGLRenderer、WebGPURenderer、SVG/CSS3Dレンダラー（アドオン）によるマルチレンダリング対応 |
| シーン管理 | Scene、Fog、FogExp2によるシーン構築と霧効果 |
| カメラ | PerspectiveCamera、OrthographicCamera、CubeCamera、ArrayCamera、StereoCameraなど多様なカメラタイプ |
| ジオメトリ | Box、Sphere、Plane、Cylinder、Torus、ExtrudeGeometryなど豊富なプリミティブと押出し形状 |
| マテリアル | MeshBasicMaterial、MeshStandardMaterial、MeshPhysicalMaterial、ShaderMaterialなど物理ベースレンダリング対応 |
| ライティング | AmbientLight、DirectionalLight、PointLight、SpotLight、RectAreaLight、HemisphereLightなど多様な光源 |
| テクスチャ | Texture、DataTexture、CubeTexture、VideoTexture、CanvasTextureなど多様なテクスチャタイプ |
| アニメーション | AnimationMixer、AnimationClip、KeyframeTrackによるスケルタルアニメーションとモーフターゲット |
| オーディオ | Audio、PositionalAudio、AudioAnalyserによる3D空間オーディオ対応 |
| ローダー | ObjectLoader、TextureLoader、GLTFLoader（アドオン）など多様なファイル形式の読み込み |
| 数学ユーティリティ | Vector2/3/4、Matrix3/4、Quaternion、Euler、Color、Box3、Sphere、Rayなど |
| XRサポート | WebXR APIによるVR/AR体験の構築 |
| ポストプロセス | EffectComposer、各種Passによる後処理エフェクト（アドオン） |
| 物理演算連携 | 物理エンジンとの連携用ユーティリティ（アドオン） |

## 技術スタック

### コア

- 言語: JavaScript (ES Modules)
- グラフィックスAPI: WebGL 2.0、WebGPU
- XR: WebXR API（VR/AR対応）
- オーディオ: Web Audio API

### 主要ライブラリ

- qunit: ユニットテストフレームワーク
- rollup: モジュールバンドラー
- eslint: コード品質チェック
- puppeteer: E2Eテスト
- jsdoc: ドキュメント生成

## ディレクトリ構成

```
three.js-dev/
├── build/                    # ビルド済みライブラリファイル
│   ├── three.module.js       # ESモジュール版
│   ├── three.cjs             # CommonJS版
│   ├── three.webgpu.js       # WebGPU版
│   └── three.tsl.js          # TSL（Three Shading Language）版
├── src/                      # ソースコード
│   ├── animation/            # アニメーションシステム
│   ├── audio/                # オーディオシステム
│   ├── cameras/              # カメラクラス
│   ├── core/                 # コアクラス（Object3D、BufferGeometry等）
│   ├── extras/               # ユーティリティ、カーブ、シェイプ
│   ├── geometries/           # ジオメトリクラス
│   ├── helpers/              # ヘルパークラス（可視化ツール）
│   ├── lights/               # ライトクラス
│   ├── loaders/              # ローダークラス
│   ├── materials/            # マテリアルクラス
│   ├── math/                 # 数学ユーティリティ
│   ├── nodes/                # ノードベースマテリアルシステム
│   ├── objects/              # オブジェクトクラス（Mesh、Line等）
│   ├── renderers/            # レンダラー（WebGL、WebGPU）
│   ├── scenes/               # シーン関連
│   └── textures/             # テクスチャクラス
├── examples/                 # サンプルコードとアドオン
│   ├── jsm/                  # ESモジュール形式のアドオン
│   │   ├── controls/         # カメラコントロール
│   │   ├── loaders/          # 追加ローダー（GLTF、FBX等）
│   │   ├── postprocessing/   # ポストプロセスエフェクト
│   │   ├── renderers/        # 追加レンダラー（CSS3D、SVG）
│   │   └── ...               # その他アドオン
│   └── *.html                # HTMLサンプル
├── docs/                     # APIドキュメント
├── editor/                   # three.jsエディタ
├── manual/                   # ユーザーマニュアル
├── test/                     # テストコード
│   ├── unit/                 # ユニットテスト
│   └── e2e/                  # E2Eテスト
└── utils/                    # ビルドツール等
```

## セットアップ

### 必要条件

- Node.js（npm付属）
- モダンブラウザ（WebGL 2.0またはWebGPU対応）

### インストール

**npm経由:**
```bash
npm install three
```

**CDN経由:**
```html
<script type="importmap">
{
  "imports": {
    "three": "https://cdn.jsdelivr.net/npm/three@0.182.0/build/three.module.js",
    "three/addons/": "https://cdn.jsdelivr.net/npm/three@0.182.0/examples/jsm/"
  }
}
</script>
```

### 基本的な使用例

```javascript
import * as THREE from 'three';

const width = window.innerWidth, height = window.innerHeight;

// 初期化
const camera = new THREE.PerspectiveCamera( 70, width / height, 0.01, 10 );
camera.position.z = 1;

const scene = new THREE.Scene();

const geometry = new THREE.BoxGeometry( 0.2, 0.2, 0.2 );
const material = new THREE.MeshNormalMaterial();

const mesh = new THREE.Mesh( geometry, material );
scene.add( mesh );

const renderer = new THREE.WebGLRenderer( { antialias: true } );
renderer.setSize( width, height );
renderer.setAnimationLoop( animate );
document.body.appendChild( renderer.domElement );

// アニメーション
function animate( time ) {
    mesh.rotation.x = time / 2000;
    mesh.rotation.y = time / 1000;
    renderer.render( scene, camera );
}
```

## 開発

### 開発サーバーの起動

```bash
npm install
npm start
```
開発サーバーが `http://localhost:8080` で起動します。

### ビルド

```bash
# 全ビルド
npm run build

# モジュール版のみ
npm run build-module
```

### Lint

```bash
# コアソースのlint
npm run lint

# 全てのlintと自動修正
npm run lint-fix
```

### テスト

```bash
# 全テスト実行
npm test

# ユニットテストのみ
npm run test-unit

# アドオンのユニットテスト
npm run test-unit-addons

# E2Eテスト
npm run test-e2e

# WebGPU E2Eテスト
npm run test-e2e-webgpu
```

### ドキュメント生成

```bash
npm run build-docs
```

## 関連リンク

- [公式サイト](https://threejs.org/)
- [ドキュメント](https://threejs.org/docs/)
- [サンプル](https://threejs.org/examples/)
- [マニュアル](https://threejs.org/manual/)
- [GitHub](https://github.com/mrdoob/three.js)
- [Discord](https://discord.gg/56GBJwAnUS)
- [フォーラム](https://discourse.threejs.org/)

## ライセンス

MIT License
