# アーキテクチャ設計書

## 概要

本ドキュメントは、Three.js（バージョン0.182.0）のアーキテクチャ設計を記述したものです。Three.jsは、JavaScriptで3Dグラフィックスを扱うためのオープンソースライブラリであり、WebGLおよびWebGPUをレンダリングバックエンドとして使用します。本設計書では、モジュール構成、レイヤー構造、データフロー、および横断的関心事について解説します。

## システム全体構成

### アーキテクチャ概要図

詳細なアーキテクチャ構成図は [アーキテクチャ構成図.md](./アーキテクチャ構成図.md) を参照してください。

### システム境界と外部連携

| 外部システム | 連携方式 | 用途 |
| --- | --- | --- |
| WebGL 2 API | ブラウザネイティブAPI | 3Dレンダリング（デフォルト） |
| WebGPU API | ブラウザネイティブAPI | 次世代3Dレンダリング |
| Web Audio API | ブラウザネイティブAPI | 3Dオーディオ処理 |
| WebXR API | ブラウザネイティブAPI | VR/AR体験 |
| Canvas API | ブラウザネイティブAPI | 2D描画、テクスチャ生成 |
| Fetch API | ブラウザネイティブAPI | アセットローディング |

## レイヤー構成

### アーキテクチャスタイル

Three.jsは**コンポーネントベースアーキテクチャ**と**戦略パターン**を組み合わせた設計を採用しています。

- **コンポーネントベース**: Object3D（シーングラフノード）にGeometry、Material、Textureなどのコンポーネントを組み合わせてオブジェクトを構成
- **戦略パターン**: Renderer（WebGLRenderer/WebGPURenderer）が交換可能なバックエンドとして機能
- **イベント駆動**: EventDispatcherによるカスタムイベントシステム

### レイヤー定義

| レイヤー | 責務 | 主なコンポーネント |
| --- | --- | --- |
| Application | ユーザーアプリケーションコード | カスタムシーン、アニメーションループ |
| API (Three.Core) | 3Dシーン構成の抽象化 | Scene, Camera, Object3D, Mesh, Light |
| Node System | シェーダーグラフ、マテリアル構築 | Node, NodeBuilder, NodeMaterial, TSL |
| Rendering | レンダリングパイプライン制御 | Renderer, RenderList, Pipeline, Backend |
| Backend | GPU API抽象化 | WebGLRenderer, WebGPURenderer, Backend |

### レイヤー間の依存関係ルール

```
Application
    ↓ (使用)
API (Three.Core)
    ↓ (使用)
Node System ←→ Rendering
    ↓         ↓
Backend (WebGL/WebGPU)
```

- **許可される参照**: 上位レイヤーから下位レイヤーへの参照
- **禁止される参照**: 下位レイヤーから上位レイヤーへの直接参照（イベント経由は許可）
- **同一レイヤー間**: Node SystemとRenderingは相互参照可能

## モジュール構成

### ドメイン分割

| ドメイン | 責務 | 関連モジュール |
| --- | --- | --- |
| Core | 基本オブジェクト、イベント、ジオメトリ基盤 | core/, constants.js, utils.js |
| Math | 3D数学ライブラリ（ベクトル、行列、四元数） | math/ |
| Objects | 3Dオブジェクト（メッシュ、ライン、ポイント） | objects/ |
| Cameras | カメラシステム | cameras/ |
| Lights | ライティングシステム | lights/ |
| Materials | マテリアルシステム | materials/ |
| Geometries | ジオメトリプリミティブ | geometries/ |
| Textures | テクスチャ管理 | textures/ |
| Scenes | シーングラフ管理 | scenes/ |
| Animation | アニメーションシステム | animation/ |
| Audio | 3Dオーディオシステム | audio/ |
| Nodes | ノードベースシェーダーシステム | nodes/ |
| Renderers | レンダリングエンジン | renderers/ |
| Loaders | アセットローディング | loaders/ |
| Helpers | デバッグ・可視化ヘルパー | helpers/ |
| Extras | ユーティリティ、カーブ | extras/ |
| Addons | 拡張機能（examples/jsm） | examples/jsm/ |

### パッケージ構造

```
src/
├── Three.js                 # WebGLRenderer含むメインエントリ
├── Three.Core.js            # コアモジュールエクスポート
├── Three.WebGPU.js          # WebGPURenderer含むエントリ
├── Three.TSL.js             # TSL (Three.js Shading Language)
├── constants.js             # 定数定義
├── utils.js                 # ユーティリティ関数
├── animation/               # アニメーションシステム
│   ├── AnimationAction.js
│   ├── AnimationClip.js
│   ├── AnimationMixer.js
│   └── tracks/              # キーフレームトラック
├── audio/                   # 3Dオーディオ
│   ├── Audio.js
│   ├── AudioListener.js
│   └── PositionalAudio.js
├── cameras/                 # カメラ
│   ├── Camera.js
│   ├── PerspectiveCamera.js
│   └── OrthographicCamera.js
├── core/                    # コア機能
│   ├── Object3D.js          # シーングラフノード基底クラス
│   ├── BufferGeometry.js    # ジオメトリ基底クラス
│   ├── BufferAttribute.js   # 頂点属性
│   ├── EventDispatcher.js   # イベントシステム
│   ├── Raycaster.js         # レイキャスティング
│   └── RenderTarget.js      # オフスクリーンレンダリング
├── geometries/              # ジオメトリプリミティブ
│   ├── BoxGeometry.js
│   ├── SphereGeometry.js
│   └── ...
├── helpers/                 # デバッグヘルパー
├── lights/                  # ライト
│   ├── Light.js
│   ├── DirectionalLight.js
│   ├── PointLight.js
│   └── SpotLight.js
├── loaders/                 # アセットローダー
│   ├── Loader.js            # ローダー基底クラス
│   ├── TextureLoader.js
│   ├── ObjectLoader.js
│   └── nodes/               # ノードシステム用ローダー
├── materials/               # マテリアル
│   ├── Material.js          # マテリアル基底クラス
│   ├── MeshStandardMaterial.js
│   ├── ShaderMaterial.js
│   └── nodes/               # ノードベースマテリアル
│       ├── NodeMaterial.js
│       └── manager/
├── math/                    # 3D数学
│   ├── Vector2.js
│   ├── Vector3.js
│   ├── Vector4.js
│   ├── Matrix3.js
│   ├── Matrix4.js
│   ├── Quaternion.js
│   ├── Euler.js
│   └── interpolants/        # 補間関数
├── nodes/                   # ノードシステム
│   ├── Nodes.js             # ノードエクスポート
│   ├── TSL.js               # Three.js Shading Language
│   ├── core/                # ノードコア
│   │   ├── Node.js
│   │   └── NodeBuilder.js
│   ├── accessors/           # データアクセサノード
│   ├── code/                # コード生成ノード
│   ├── display/             # ディスプレイノード
│   ├── lighting/            # ライティングノード
│   ├── math/                # 数学ノード
│   └── utils/               # ユーティリティノード
├── objects/                 # 3Dオブジェクト
│   ├── Mesh.js
│   ├── Line.js
│   ├── Points.js
│   ├── Sprite.js
│   ├── SkinnedMesh.js
│   └── InstancedMesh.js
├── renderers/               # レンダラー
│   ├── WebGLRenderer.js     # WebGL 2レンダラー
│   ├── WebGLRenderTarget.js
│   ├── common/              # 共通レンダリング基盤
│   │   ├── Renderer.js      # レンダラー基底クラス
│   │   ├── Backend.js       # バックエンド抽象化
│   │   ├── RenderList.js
│   │   ├── RenderObject.js
│   │   ├── Pipelines.js
│   │   └── nodes/           # レンダラー用ノード
│   ├── webgl/               # WebGL固有実装
│   │   ├── WebGLState.js
│   │   ├── WebGLPrograms.js
│   │   └── ...
│   ├── webgpu/              # WebGPU実装
│   │   ├── WebGPURenderer.js
│   │   └── WebGPUBackend.js
│   ├── webgl-fallback/      # WebGLフォールバック
│   ├── webxr/               # WebXRサポート
│   └── shaders/             # GLSLシェーダー
│       ├── ShaderChunk/
│       └── ShaderLib/
├── scenes/                  # シーン
│   ├── Scene.js
│   ├── Fog.js
│   └── FogExp2.js
├── textures/                # テクスチャ
│   ├── Texture.js
│   ├── DataTexture.js
│   └── CubeTexture.js
└── extras/                  # ユーティリティ
    ├── Controls.js
    ├── curves/              # カーブ
    ├── core/                # Path, Shape
    └── lib/                 # 外部ライブラリ
```

### コンポーネント依存関係

**コアコンポーネントの継承関係:**

```
EventDispatcher
    └── Object3D
            ├── Scene
            ├── Camera
            │     ├── PerspectiveCamera
            │     └── OrthographicCamera
            ├── Mesh
            ├── Line
            ├── Points
            ├── Light
            │     ├── DirectionalLight
            │     ├── PointLight
            │     └── SpotLight
            └── Group
```

**マテリアル継承関係:**

```
Material
    ├── MeshBasicMaterial
    ├── MeshStandardMaterial
    │     └── MeshPhysicalMaterial
    ├── MeshPhongMaterial
    ├── ShaderMaterial
    │     └── RawShaderMaterial
    └── NodeMaterial (nodes/materials/)
```

**レンダラー継承関係:**

```
Renderer (common/Renderer.js)
    └── WebGPURenderer (webgpu/WebGPURenderer.js)
            └── Backend抽象化
                    ├── WebGPUBackend
                    └── WebGLBackend (webgl-fallback)

WebGLRenderer (独立実装)
```

## ミドルウェア構成

### データベース

Three.jsはクライアントサイドライブラリのため、データベースミドルウェアは使用しません。

| 種類 | ミドルウェア | バージョン | 用途 |
| --- | --- | --- | --- |
| - | 該当なし | - | クライアントサイドライブラリ |

### キャッシュ

| ミドルウェア | バージョン | 用途 | TTL |
| --- | --- | --- | --- |
| Cache (内部) | - | アセットキャッシュ | セッション |
| NodeCache | - | ノードシステムキャッシュ | レンダリングフレーム |

### メッセージキュー

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 該当なし | - | クライアントサイドライブラリ |

### 検索エンジン

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| 該当なし | - | クライアントサイドライブラリ |

### その他ミドルウェア

| ミドルウェア | バージョン | 用途 |
| --- | --- | --- |
| Rollup | ^4.6.0 | ビルドシステム |
| ESLint | ^9.0.0 | コード品質管理 |
| QUnit | ^2.19.4 | ユニットテスト |
| Puppeteer | ^24.25.0 | E2Eテスト |

## データフロー

### リクエスト処理の流れ

**レンダリングパイプライン（同期処理）:**

1. `renderer.render(scene, camera)` 呼び出し
2. シーングラフのトラバース（Object3D階層の走査）
3. Frustum Culling（視錐台カリング）によるオブジェクトフィルタリング
4. RenderList作成（描画順序の決定）
5. 各オブジェクトのRenderObject生成
6. マテリアル/ジオメトリからPipeline（シェーダープログラム）取得または生成
7. Bindingsの設定（Uniform、テクスチャ）
8. GPU描画コマンドの発行
9. フレームバッファへの描画完了

### 非同期処理の流れ

**アセットローディング:**

1. Loader（TextureLoader等）にURL/パスを渡して`load()`呼び出し
2. FileLoaderがFetch APIでリソース取得
3. Cache確認（キャッシュ済みなら即座に返却）
4. 非同期でリソースダウンロード
5. パース処理（画像デコード、GLTFパース等）
6. コールバックまたはPromiseで結果返却
7. GPUへのテクスチャアップロード（遅延実行）

**アニメーションループ:**

1. `renderer.setAnimationLoop(callback)` でループ登録
2. requestAnimationFrame によるフレームコールバック
3. AnimationMixer.update() でアニメーション更新
4. シーン状態の更新
5. renderer.render() 呼び出し
6. 次フレームへ

### データ永続化の流れ

Three.jsはクライアントサイドライブラリのため、永続化は以下の形式で行われます：

1. **シリアライゼーション**: `object.toJSON()` でオブジェクトをJSON形式に変換
2. **エクスポート**: GLTFExporter、OBJExporter等でファイル形式に変換
3. **ローカルストレージ**: ブラウザのlocalStorageへの保存（アプリケーション責務）
4. **サーバー送信**: アプリケーションがサーバーにデータ送信

## 横断的関心事

### 認証・認可

Three.jsはレンダリングライブラリのため、認証・認可機能は持ちません。

| 方式 | 実装箇所 | 対象 |
| --- | --- | --- |
| 該当なし | - | アプリケーション責務 |

### ロギング・監査

| 種類 | 実装方式 | 保存先 |
| --- | --- | --- |
| アプリケーションログ | utils.js (log, warn, error) | コンソール |
| 警告ログ | utils.js (warnOnce) | コンソール（重複抑制） |
| レンダリング情報 | renderer.info | メモリ（統計情報） |

### エラーハンドリング

| エラー種別 | ハンドリング方式 | レスポンス |
| --- | --- | --- |
| WebGL/WebGPUコンテキスト取得失敗 | 例外スロー | エラーメッセージ |
| シェーダーコンパイルエラー | コンソールエラー出力 | フォールバック/停止 |
| テクスチャロード失敗 | onErrorコールバック | デフォルトテクスチャ |
| 無効なパラメータ | コンソール警告 | デフォルト値使用 |
| 非推奨API使用 | warnOnce | 警告出力（1回のみ） |

### トランザクション管理

Three.jsはクライアントサイドライブラリのため、トランザクション管理は該当しません。

| 範囲 | 管理方式 | 分離レベル |
| --- | --- | --- |
| 該当なし | - | - |

## 設計原則・コーディング規約

### 適用している設計原則

| 原則 | 適用箇所 | 実装例 |
| --- | --- | --- |
| 単一責任原則 (SRP) | 全モジュール | Object3Dは位置・回転のみ、Meshはジオメトリ/マテリアル保持 |
| オープン・クローズド原則 | Material, Geometry | 基底クラスを継承して拡張、既存コード変更なし |
| リスコフの置換原則 | Object3D継承階層 | Scene, Mesh, Lightは全てObject3Dとして扱える |
| 依存性逆転原則 | Renderer/Backend | RendererはBackendインターフェースに依存 |
| コンポジション over 継承 | Mesh | Geometry + Material の組み合わせ |
| 戦略パターン | Renderer | WebGLRenderer/WebGPURendererの切り替え |
| オブザーバーパターン | EventDispatcher | イベントリスナーによる疎結合 |
| ビジターパターン | traverse() | シーングラフの走査 |

### コーディング規約

**言語・フォーマット:**
- ECMAScript Modules (ESM) 形式
- クラスベースのオブジェクト指向
- JSDoc形式のドキュメントコメント
- タブインデント

**命名規則:**
- クラス: PascalCase (`PerspectiveCamera`)
- メソッド/プロパティ: camelCase (`setSize`, `matrixWorld`)
- 定数: UPPER_SNAKE_CASE (`BackSide`, `FrontSide`)
- プライベート: アンダースコアプレフィックス (`_listeners`)

**型チェック:**
- `isXxx` フラグによる型判定 (`object.isMesh`, `material.isMaterial`)
- `type` プロパティによる文字列型判定

**イミュータビリティ:**
- 数学オブジェクト（Vector3等）は内部状態を変更するメソッドを提供
- `clone()` メソッドで新規インスタンス作成

## 備考

### バージョン情報
- Three.js バージョン: 0.182.0
- 対応ブラウザ: "> 1%, not dead, not ie 11, not op_mini all"

### 重要な設計上の決定

1. **WebGL 1非対応**: r163以降、WebGL 2のみサポート
2. **ノードシステム**: WebGPURenderer向けに開発されたシェーダーグラフシステム
3. **TSL (Three.js Shading Language)**: JavaScriptでシェーダーを記述可能
4. **Addons分離**: コアライブラリを軽量化するため、追加機能は examples/jsm/ に分離

### 関連ドキュメント
- 公式ドキュメント: https://threejs.org/docs/
- マニュアル: https://threejs.org/manual/
- Wiki: https://github.com/mrdoob/three.js/wiki
