# three.js コードリーディングガイドライン

## はじめに

このガイドラインは、three.jsのコードベースを効率的に理解するための手引きです。
JavaScriptに精通していないエンジニアでも、段階的に学習できるよう構成されています。

**対象読者:**
- プロジェクトに新規参画するエンジニア
- 他言語からの経験者
- コードレビューを行う担当者

---

## 1. 言語基礎

> このセクションでは、JavaScriptの基本構文とthree.jsで使用される概念を解説します。

### 1.1 プログラム構造

three.jsはES Modules形式を採用しています。各ファイルはモジュールとして独立し、`export`/`import`で連携します。

**実装例:** `src/Three.js:1-10`
```javascript
export * from './Three.Core.js';

export { WebGLRenderer } from './renderers/WebGLRenderer.js';
export { ShaderLib } from './renderers/shaders/ShaderLib.js';
export { UniformsLib } from './renderers/shaders/UniformsLib.js';
export { UniformsUtils } from './renderers/shaders/UniformsUtils.js';
export { ShaderChunk } from './renderers/shaders/ShaderChunk.js';
export { PMREMGenerator } from './extras/PMREMGenerator.js';
export { WebGLUtils } from './renderers/webgl/WebGLUtils.js';
```

**解説:**
- `export * from` - 別モジュールの全エクスポートを再エクスポート
- `export { X } from` - 特定のクラス/関数を名前付きエクスポート
- ファイルパスは相対パスで指定（`./`から始まる）

### 1.2 データ型と変数

three.jsでは、主に以下の変数宣言を使用します:

**実装例:** `src/math/Vector3.js:38-70`
```javascript
constructor( x = 0, y = 0, z = 0 ) {

    /**
     * This flag can be used for type testing.
     *
     * @type {boolean}
     * @readonly
     * @default true
     */
    Vector3.prototype.isVector3 = true;

    /**
     * The x value of this vector.
     *
     * @type {number}
     */
    this.x = x;
    this.y = y;
    this.z = z;

}
```

**解説:**
- `const` - 再代入不可の定数（three.jsで最も使用）
- `let` - 再代入可能な変数
- デフォルト引数 `x = 0` - 引数が省略された場合のデフォルト値
- JSDocコメント `@type {boolean}` - 型情報の記述

### 1.3 制御構造

**実装例:** `src/objects/Mesh.js:287-343`
```javascript
if ( index !== null ) {

    // indexed buffer geometry

    if ( Array.isArray( material ) ) {

        for ( let i = 0, il = groups.length; i < il; i ++ ) {

            const group = groups[ i ];
            const groupMaterial = material[ group.materialIndex ];

            const start = Math.max( group.start, drawRange.start );
            const end = Math.min( index.count, Math.min( ( group.start + group.count ), ( drawRange.start + drawRange.count ) ) );

            for ( let j = start, jl = end; j < jl; j += 3 ) {
                // ...処理
            }

        }

    }

}
```

**解説:**
- `if/else` - 条件分岐
- `for (let i = 0, il = length; i < il; i++)` - 最適化されたforループ（lengthをキャッシュ）
- `Array.isArray()` - 配列かどうかの判定

### 1.4 関数/メソッド定義

three.jsはクラスベースのオブジェクト指向を採用しています。

**実装例:** `src/math/Vector3.js:80-90`
```javascript
/**
 * Sets the vector components.
 *
 * @param {number} x - The value of the x component.
 * @param {number} y - The value of the y component.
 * @param {number} z - The value of the z component.
 * @return {Vector3} A reference to this vector.
 */
set( x, y, z ) {

    if ( z === undefined ) z = this.z; // sprite.scale.set(x,y)

    this.x = x;
    this.y = y;
    this.z = z;

    return this;

}
```

**解説:**
- メソッドは `methodName() { }` 形式で定義
- `this` - 現在のインスタンスを参照
- `return this` - メソッドチェーンを可能にするパターン

### 1.5 モジュール/インポート

**実装例:** `src/core/Object3D.js:1-9`
```javascript
import { Quaternion } from '../math/Quaternion.js';
import { Vector3 } from '../math/Vector3.js';
import { Matrix4 } from '../math/Matrix4.js';
import { EventDispatcher } from './EventDispatcher.js';
import { Euler } from '../math/Euler.js';
import { Layers } from './Layers.js';
import { Matrix3 } from '../math/Matrix3.js';
import { generateUUID } from '../math/MathUtils.js';
import { error } from '../utils.js';
```

**解説:**
- `import { X } from 'path'` - 名前付きインポート
- 相対パス `../` - 親ディレクトリへの参照
- 相対パス `./` - 同階層ディレクトリへの参照

---

## 2. プロジェクト固有の概念

> このセクションでは、three.js特有の概念を解説します。

### 2.1 フレームワーク固有の概念

#### シーングラフ
three.jsは**シーングラフ**構造を採用しています。3Dオブジェクトは親子関係を持ち、階層構造で管理されます。

**実装例:** `src/core/Object3D.js:64-128`
```javascript
class Object3D extends EventDispatcher {

    constructor() {

        super();

        this.isObject3D = true;
        Object.defineProperty( this, 'id', { value: _object3DId ++ } );
        this.uuid = generateUUID();
        this.name = '';
        this.type = 'Object3D';
        this.parent = null;
        this.children = [];
        this.up = Object3D.DEFAULT_UP.clone();
        // ...
    }
}
```

**解説:**
- `parent` - 親オブジェクトへの参照
- `children` - 子オブジェクトの配列
- `up` - オブジェクトの上方向ベクトル

#### 型判定フラグパターン
three.jsでは、`instanceof`の代わりに型判定用のフラグプロパティを使用します。

**実装例:** `src/objects/Mesh.js:58`
```javascript
this.isMesh = true;
```

**解説:**
- `isXxx` フラグで型を判定（パフォーマンス最適化のため）
- 例: `object.isMesh`, `material.isMaterial`, `vector.isVector3`

### 2.2 プロジェクト独自のパターン

#### メソッドチェーンパターン
多くのメソッドは`this`を返し、メソッドチェーンを可能にします。

**実装例:** `src/math/Vector3.js:224-232`
```javascript
add( v ) {

    this.x += v.x;
    this.y += v.y;
    this.z += v.z;

    return this;

}
```

**使用例:**
```javascript
vector.add(v1).multiplyScalar(2).normalize();
```

#### Pure関数によるメモリ最適化
グローバルスコープで再利用可能なオブジェクトを宣言し、メモリ割り当てを最小化します。

**実装例:** `src/objects/Mesh.js:12-25`
```javascript
const _inverseMatrix = /*@__PURE__*/ new Matrix4();
const _ray = /*@__PURE__*/ new Ray();
const _sphere = /*@__PURE__*/ new Sphere();
const _sphereHitAt = /*@__PURE__*/ new Vector3();

const _vA = /*@__PURE__*/ new Vector3();
const _vB = /*@__PURE__*/ new Vector3();
const _vC = /*@__PURE__*/ new Vector3();
```

**解説:**
- `/*@__PURE__*/` - ツリーシェイキング用のヒント
- アンダースコア接頭辞 `_` - モジュール内部の一時オブジェクト

---

## 3. 命名規則

> このセクションでは、プロジェクト全体で使用される命名規則を解説します。

### 3.1 ファイル・ディレクトリ命名

| パターン | 意味 | 例 |
|---------|------|-----|
| `PascalCase.js` | クラス定義ファイル | `Vector3.js`, `Object3D.js`, `Mesh.js` |
| `camelCase.js` | ユーティリティ/定数ファイル | `constants.js`, `utils.js` |
| 複数形ディレクトリ | 同種のクラスを格納 | `materials/`, `geometries/`, `loaders/` |

### 3.2 クラス・関数・変数命名

| プレフィックス/サフィックス | 意味 | 例 |
|---------------------------|------|-----|
| `is` 接頭辞 | 型判定フラグ | `isMesh`, `isVector3`, `isMaterial` |
| `_` 接頭辞 | プライベート/モジュール内部変数 | `_object3DId`, `_inverseMatrix` |
| `DEFAULT_` 接頭辞 | 静的デフォルト値 | `DEFAULT_UP`, `DEFAULT_MATRIX_AUTO_UPDATE` |
| `Loader` 接尾辞 | ローダークラス | `TextureLoader`, `GLTFLoader` |
| `Helper` 接尾辞 | デバッグ/可視化ヘルパー | `AxesHelper`, `BoxHelper` |
| `Material` 接尾辞 | マテリアルクラス | `MeshBasicMaterial`, `MeshStandardMaterial` |
| `Geometry` 接尾辞 | ジオメトリクラス | `BoxGeometry`, `SphereGeometry` |
| `Controls` 接尾辞 | インタラクション制御 | `OrbitControls`, `FlyControls` |

### 3.3 プログラム分類一覧

| 分類 | ディレクトリ | 説明 |
|-----|------------|------|
| コア | `src/core/` | Object3D, BufferGeometry等の基盤クラス |
| 数学 | `src/math/` | Vector, Matrix, Quaternion等 |
| オブジェクト | `src/objects/` | Mesh, Line, Points等の描画オブジェクト |
| マテリアル | `src/materials/` | 各種マテリアルクラス |
| ジオメトリ | `src/geometries/` | 各種ジオメトリクラス |
| ローダー | `src/loaders/` | ファイル読み込みクラス |
| レンダラー | `src/renderers/` | WebGL/WebGPUレンダラー |
| シーン | `src/scenes/` | Scene, Fog等 |
| カメラ | `src/cameras/` | 各種カメラクラス |
| ライト | `src/lights/` | 各種ライトクラス |
| テクスチャ | `src/textures/` | 各種テクスチャクラス |
| アニメーション | `src/animation/` | アニメーション関連クラス |
| アドオン | `examples/jsm/` | 追加機能（Controls, Loaders等） |

---

## 4. ディレクトリ構造

> このセクションでは、プロジェクトのディレクトリ構造を解説します。

```
three.js-dev/
├── src/                    # ライブラリのソースコード
│   ├── Three.js            # メインエントリーポイント
│   ├── Three.Core.js       # コアエクスポート
│   ├── Three.WebGPU.js     # WebGPU版エントリー
│   ├── constants.js        # 定数定義
│   ├── utils.js            # ユーティリティ関数
│   ├── animation/          # アニメーション関連
│   ├── audio/              # オーディオ関連
│   ├── cameras/            # カメラクラス
│   ├── core/               # コア基盤クラス
│   ├── extras/             # ユーティリティとカーブ
│   ├── geometries/         # ジオメトリクラス
│   ├── helpers/            # デバッグヘルパー
│   ├── lights/             # ライトクラス
│   ├── loaders/            # ローダークラス
│   ├── materials/          # マテリアルクラス
│   ├── math/               # 数学クラス
│   ├── nodes/              # ノードシステム (TSL)
│   ├── objects/            # 3Dオブジェクトクラス
│   ├── renderers/          # レンダラー
│   ├── scenes/             # シーン関連
│   └── textures/           # テクスチャクラス
├── examples/               # サンプルとアドオン
│   ├── jsm/                # ESモジュール形式のアドオン
│   │   ├── controls/       # カメラコントロール
│   │   ├── loaders/        # 追加ローダー (GLTF等)
│   │   ├── postprocessing/ # ポストプロセッシング
│   │   └── ...
│   └── fonts/              # フォントデータ
├── build/                  # ビルド済みファイル
├── docs/                   # ドキュメント
├── editor/                 # three.jsエディター
├── test/                   # テストコード
└── utils/                  # ビルドスクリプト等
```

### 各ディレクトリの役割

| ディレクトリ | 役割 | 主要ファイル |
|-------------|------|-------------|
| `src/` | ライブラリ本体のソース | `Three.js`, `Three.Core.js` |
| `src/core/` | 基盤クラス（全オブジェクトの親） | `Object3D.js`, `BufferGeometry.js`, `EventDispatcher.js` |
| `src/math/` | 数学演算クラス | `Vector3.js`, `Matrix4.js`, `Quaternion.js` |
| `src/objects/` | 描画可能なオブジェクト | `Mesh.js`, `Line.js`, `Points.js` |
| `src/materials/` | 外観を定義するマテリアル | `Material.js`, `MeshBasicMaterial.js`, `MeshStandardMaterial.js` |
| `src/renderers/` | 描画エンジン | `WebGLRenderer.js`, `webgpu/WebGPURenderer.js` |
| `examples/jsm/` | 追加機能（非コア） | `controls/OrbitControls.js`, `loaders/GLTFLoader.js` |

---

## 5. アーキテクチャ

> このセクションでは、プロジェクトのアーキテクチャパターンを解説します。

### 5.1 全体アーキテクチャ

three.jsは**コンポーネントベースのシーングラフアーキテクチャ**を採用しています。

```
┌─────────────────────────────────────────────────────────────┐
│                      アプリケーション                        │
├─────────────────────────────────────────────────────────────┤
│  ┌─────────┐  ┌─────────┐  ┌──────────┐  ┌───────────────┐ │
│  │ Scene   │  │ Camera  │  │ Renderer │  │ Controls      │ │
│  │(シーン) │  │(カメラ) │  │(レンダラ)│  │(インタラクション)│ │
│  └────┬────┘  └─────────┘  └──────────┘  └───────────────┘ │
│       │                                                     │
│  ┌────┴─────────────────────────────────────────────┐      │
│  │              Scene Graph (シーングラフ)           │      │
│  │  ┌─────────┐  ┌─────────┐  ┌─────────┐          │      │
│  │  │ Object3D│──│ Object3D│──│ Object3D│          │      │
│  │  │ (Mesh)  │  │ (Light) │  │ (Group) │          │      │
│  │  └────┬────┘  └─────────┘  └────┬────┘          │      │
│  │       │                         │                │      │
│  │  ┌────┴────┐               ┌────┴────┐          │      │
│  │  │Geometry │               │ children│          │      │
│  │  │Material │               └─────────┘          │      │
│  │  └─────────┘                                    │      │
│  └─────────────────────────────────────────────────┘      │
├─────────────────────────────────────────────────────────────┤
│                    three.js コアライブラリ                   │
│  ┌──────┐  ┌──────┐  ┌──────────┐  ┌──────────┐          │
│  │ Math │  │ Core │  │ Renderers│  │ Loaders  │          │
│  └──────┘  └──────┘  └──────────┘  └──────────┘          │
└─────────────────────────────────────────────────────────────┘
```

### 5.2 レイヤー構成

| レイヤー | 責務 | 代表的なファイル |
|---------|------|-----------------|
| アプリケーション | ユーザーアプリケーションコード | - |
| シーン管理 | 3Dオブジェクトの階層管理 | `src/scenes/Scene.js` |
| オブジェクト | 描画可能なエンティティ | `src/objects/Mesh.js`, `src/lights/PointLight.js` |
| コンポーネント | Geometry, Material | `src/core/BufferGeometry.js`, `src/materials/Material.js` |
| 数学基盤 | ベクトル、行列演算 | `src/math/Vector3.js`, `src/math/Matrix4.js` |
| レンダリング | GPU描画処理 | `src/renderers/WebGLRenderer.js` |
| ユーティリティ | ローダー、ヘルパー | `src/loaders/TextureLoader.js`, `examples/jsm/controls/OrbitControls.js` |

### 5.3 データフロー

1. **初期化フェーズ**
   - Scene, Camera, Rendererを作成
   - Mesh (Geometry + Material) を作成しSceneに追加

2. **レンダリングループ**
   ```
   アニメーションループ開始
     ↓
   オブジェクト状態更新（位置、回転等）
     ↓
   カメラマトリクス更新
     ↓
   シーングラフ走査
     ↓
   描画コマンド発行（WebGL/WebGPU）
     ↓
   次フレームへ
   ```

3. **マトリクス更新フロー**
   ```
   Object3D.updateMatrix()        # ローカルマトリクス更新
     ↓
   Object3D.updateMatrixWorld()   # ワールドマトリクス更新
     ↓
   子オブジェクトへ再帰的に伝播
   ```

---

## 6. 主要コンポーネント

> このセクションでは、主要なコンポーネントとその連携を解説します。

### 6.1 エントリーポイント

three.jsのエントリーポイントは`src/Three.js`で、全てのパブリッククラスを再エクスポートします。

**実装例:** `src/Three.js:1-9`
```javascript
export * from './Three.Core.js';

export { WebGLRenderer } from './renderers/WebGLRenderer.js';
export { ShaderLib } from './renderers/shaders/ShaderLib.js';
export { UniformsLib } from './renderers/shaders/UniformsLib.js';
export { UniformsUtils } from './renderers/shaders/UniformsUtils.js';
export { ShaderChunk } from './renderers/shaders/ShaderChunk.js';
export { PMREMGenerator } from './extras/PMREMGenerator.js';
export { WebGLUtils } from './renderers/webgl/WebGLUtils.js';
```

### 6.2 ビジネスロジック

three.jsの主要なビジネスロジックは以下のクラスに実装されています:

#### Object3D - 全3Dオブジェクトの基底クラス

**実装例:** `src/core/Object3D.js:64-128`
```javascript
class Object3D extends EventDispatcher {

    constructor() {
        super();

        this.isObject3D = true;
        this.uuid = generateUUID();
        this.name = '';
        this.type = 'Object3D';
        this.parent = null;
        this.children = [];
        // ...
    }

    add( object ) {
        // 子オブジェクトの追加
        if ( object === this ) {
            error( 'Object3D.add: object can\'t be added as a child of itself.' );
            return this;
        }

        if ( object && object.isObject3D ) {
            object.removeFromParent();
            object.parent = this;
            this.children.push( object );
            // ...
        }
        return this;
    }
}
```

#### Mesh - ポリゴンメッシュ

**実装例:** `src/objects/Mesh.js:39-108`
```javascript
class Mesh extends Object3D {

    constructor( geometry = new BufferGeometry(), material = new MeshBasicMaterial() ) {
        super();

        this.isMesh = true;
        this.type = 'Mesh';
        this.geometry = geometry;
        this.material = material;
        // ...
    }
}
```

### 6.3 データアクセス

three.jsでは、外部リソースの読み込みにLoaderクラスを使用します。

**実装例:** `src/loaders/Loader.js`
```javascript
class Loader {

    constructor( manager ) {
        this.manager = ( manager !== undefined ) ? manager : DefaultLoadingManager;
        this.crossOrigin = 'anonymous';
        this.withCredentials = false;
        this.path = '';
        this.resourcePath = '';
        this.requestHeader = {};
    }

    load( /* url, onLoad, onProgress, onError */ ) {}
    loadAsync( url, onProgress ) {
        return new Promise( ( resolve, reject ) => {
            this.load( url, resolve, onProgress, reject );
        });
    }
}
```

### 6.4 ユーティリティ/共通機能

#### MathUtils - 数学ユーティリティ
**ファイル:** `src/math/MathUtils.js`

```javascript
export function generateUUID() {
    // UUID生成
}

export function clamp( value, min, max ) {
    return Math.max( min, Math.min( max, value ) );
}

export function lerp( x, y, t ) {
    return ( 1 - t ) * x + t * y;
}
```

#### EventDispatcher - イベント管理
**実装例:** `src/core/EventDispatcher.js`

```javascript
class EventDispatcher {

    addEventListener( type, listener ) { /* ... */ }
    hasEventListener( type, listener ) { /* ... */ }
    removeEventListener( type, listener ) { /* ... */ }
    dispatchEvent( event ) { /* ... */ }

}
```

---

## 7. よく使われるパターン

> このセクションでは、コード内で頻出するパターンを解説します。

### パターン一覧

| パターン | 説明 | 出現頻度 | 代表的なファイル |
|---------|------|---------|-----------------|
| 型判定フラグ | `isXxx`プロパティによる型判定 | 高 | 全クラス |
| メソッドチェーン | `return this`による連鎖呼び出し | 高 | `Vector3.js`, `Matrix4.js` |
| クローン/コピー | `clone()`/`copy()`メソッド | 高 | 全クラス |
| toJSON/fromJSON | シリアライズ/デシリアライズ | 中 | `Object3D.js`, `Material.js` |
| dispose | リソース解放 | 中 | `Material.js`, `Texture.js` |
| コールバック | `onBeforeRender`等のフック | 中 | `Object3D.js`, `Material.js` |

### 各パターンの詳細

#### パターン1: 型判定フラグ

**目的:** `instanceof`よりも高速な型判定を実現

**実装例:**
```javascript
// ファイル: src/objects/Mesh.js:58
this.isMesh = true;
```

**使用例:**
```javascript
if ( object.isMesh ) {
    // Meshとしての処理
}
```

**解説:** `instanceof`はプロトタイプチェーンを辿るため遅い。フラグ判定は単純なプロパティアクセスで高速。

#### パターン2: メソッドチェーン

**目的:** 複数の操作を1行で連続実行

**実装例:**
```javascript
// ファイル: src/math/Vector3.js:224-232
add( v ) {
    this.x += v.x;
    this.y += v.y;
    this.z += v.z;
    return this;
}
```

**使用例:**
```javascript
vector.add(v1).multiplyScalar(2).normalize();
```

#### パターン3: クローン/コピー

**目的:** オブジェクトの複製と値のコピー

**実装例:**
```javascript
// ファイル: src/math/Vector3.js:196-216
clone() {
    return new this.constructor( this.x, this.y, this.z );
}

copy( v ) {
    this.x = v.x;
    this.y = v.y;
    this.z = v.z;
    return this;
}
```

**解説:**
- `clone()` - 新しいインスタンスを作成
- `copy()` - 既存インスタンスに値をコピー（メモリ効率が良い）

#### パターン4: dispose（リソース解放）

**目的:** GPUリソースの明示的解放

**実装例:**
```javascript
// ファイル: src/materials/Material.js:989-997
dispose() {
    this.dispatchEvent( { type: 'dispose' } );
}
```

**解説:** three.jsはGPUリソースを自動解放しないため、不要になったオブジェクトは明示的に`dispose()`を呼ぶ必要がある。

---

## 8. 業務フロー追跡の実践例

> このセクションでは、実際の業務フローをコードで追跡する方法を解説します。

### 8.1 フロー追跡の基本手順

1. エントリーポイントを特定
2. 処理の流れを追跡（呼び出し関係を追う）
3. データの変換を確認
4. 最終的な出力を確認

### 8.2 フロー追跡の実例

#### 例1: Meshのレンダリングフロー

**概要:** Meshオブジェクトがどのようにレンダリングされるかを追跡

**処理フロー:**
```
renderer.render(scene, camera)
  → scene.traverse() でオブジェクト収集
  → Mesh.raycast() でカリング判定
  → WebGLRenderer内部でドローコール発行
```

**詳細な追跡:**

1. **シーントラバース** (`src/core/Object3D.js:1068-1080`)
   ```javascript
   traverse( callback ) {
       callback( this );
       const children = this.children;
       for ( let i = 0, l = children.length; i < l; i ++ ) {
           children[ i ].traverse( callback );
       }
   }
   ```

2. **マトリクス更新** (`src/core/Object3D.js:1163-1201`)
   ```javascript
   updateMatrixWorld( force ) {
       if ( this.matrixAutoUpdate ) this.updateMatrix();

       if ( this.matrixWorldNeedsUpdate || force ) {
           if ( this.matrixWorldAutoUpdate === true ) {
               if ( this.parent === null ) {
                   this.matrixWorld.copy( this.matrix );
               } else {
                   this.matrixWorld.multiplyMatrices( this.parent.matrixWorld, this.matrix );
               }
           }
           // ...
       }
   }
   ```

3. **レイキャスト（衝突判定）** (`src/objects/Mesh.js:226-270`)
   ```javascript
   raycast( raycaster, intersects ) {
       const geometry = this.geometry;
       const material = this.material;

       // バウンディングスフィアでの粗い判定
       if ( geometry.boundingSphere === null ) geometry.computeBoundingSphere();
       _sphere.copy( geometry.boundingSphere );
       _sphere.applyMatrix4( matrixWorld );

       // 詳細な交差判定へ
       this._computeIntersections( raycaster, intersects, _ray );
   }
   ```

### 8.3 フロー追跡チェックリスト

- [ ] エントリーポイントを特定したか
- [ ] 呼び出し関係を把握したか
- [ ] データの変換ポイントを確認したか
- [ ] エラーハンドリングを確認したか
- [ ] 最終的な出力を確認したか

---

## 9. 設計書の参照順序

> このセクションでは、プロジェクト理解のための設計書参照順序を案内します。

### 9.1 目的別ロードマップ

#### 全体像を把握したい場合
1. README.md - プロジェクト概要
2. 公式マニュアル（https://threejs.org/manual/）
3. `src/Three.js` - エクスポート一覧を確認
4. `src/core/Object3D.js` - 基盤クラスの理解

#### 特定機能を理解したい場合
1. 該当クラスのソースファイル
2. JSDocコメントを確認
3. `examples/`内の使用例を確認
4. 公式ドキュメント

#### 改修作業を行う場合
1. 対象クラスのソースコードを読む
2. 継承関係を確認（Object3D → Mesh等）
3. 関連するテストコード（`test/`）を確認
4. `examples/jsm/`で類似実装を参照

### 9.2 ドキュメント一覧

| ドキュメント | 概要 | 参照タイミング |
|-------------|------|---------------|
| README.md | プロジェクト概要、基本使用法 | 最初に確認 |
| 公式ドキュメント | APIリファレンス | クラス詳細確認時 |
| 公式マニュアル | チュートリアル | 学習開始時 |
| examples/ | 実装例 | 実装参考時 |
| Wiki | マイグレーションガイド等 | バージョンアップ時 |

---

## 10. トラブルシューティング

> このセクションでは、コードリーディング時によくある問題と解決法を解説します。

### よくある疑問と回答

#### Q: `/*@__PURE__*/` コメントの意味は？
A: ツリーシェイキング（未使用コード削除）のためのヒントです。この式が副作用を持たないことをバンドラーに伝えます。

#### Q: なぜ`instanceof`ではなく`isXxx`フラグを使うのか？
A: パフォーマンス最適化のためです。`instanceof`はプロトタイプチェーンを辿るため、単純なプロパティアクセスより遅くなります。

#### Q: `examples/jsm/`と`src/`の違いは？
A: `src/`はコアライブラリで、`examples/jsm/`はオプションのアドオンです。アドオンはコアに依存しますが、コアはアドオンに依存しません。

#### Q: WebGLRendererとWebGPURendererの違いは？
A: WebGLRendererはWebGL APIベース、WebGPURendererはWebGPU APIベースです。WebGPUはより新しいAPIで、将来的にWebGLを置き換える可能性があります。

### デバッグのヒント

1. **ブラウザの開発者ツール**
   - Sourcesパネルでブレークポイント設定
   - Consoleでオブジェクトのプロパティを確認

2. **three.js開発者ツール**
   - Chrome拡張「three.js Developer Tools」を使用
   - シーングラフの可視化が可能

3. **ヘルパーオブジェクト**
   ```javascript
   scene.add( new THREE.AxesHelper( 5 ) );     // 軸の可視化
   scene.add( new THREE.BoxHelper( mesh ) );   // バウンディングボックス
   scene.add( new THREE.CameraHelper( camera ) ); // カメラ視錐台
   ```

---

## 付録

### A. 用語集

| 用語 | 説明 |
|-----|------|
| Scene | 3Dオブジェクトを配置するコンテナ |
| Camera | シーンを見る視点 |
| Renderer | シーンを画面に描画するエンジン |
| Mesh | Geometry + Materialで構成される3Dオブジェクト |
| Geometry | 頂点データ（位置、法線、UV等） |
| Material | 表面の見た目（色、テクスチャ、シェーダー等） |
| Object3D | 全3Dオブジェクトの基底クラス |
| BufferGeometry | GPUバッファベースのジオメトリ |
| Texture | 画像データをGPUにアップロードしたもの |
| Shader | GPUで実行されるプログラム |
| Raycaster | 光線との交差判定を行うユーティリティ |
| Scene Graph | オブジェクトの親子関係を表す木構造 |
| TSL | Three.js Shading Language（ノードベースシェーダー） |

### B. ファイル一覧

| ファイル/ディレクトリ | 説明 | 主な内容 |
|---------------------|------|---------|
| `src/Three.js` | メインエントリー | 全エクスポートの集約 |
| `src/Three.Core.js` | コアエクスポート | WebGLRenderer以外の全クラス |
| `src/constants.js` | 定数定義 | REVISION、各種定数値 |
| `src/core/Object3D.js` | 基盤クラス | 位置、回転、スケール、親子関係 |
| `src/math/Vector3.js` | 3Dベクトル | 位置、方向の表現 |
| `src/math/Matrix4.js` | 4x4行列 | 変換行列 |
| `src/objects/Mesh.js` | メッシュ | ポリゴンモデル |
| `src/materials/Material.js` | マテリアル基底 | 描画設定 |
| `src/scenes/Scene.js` | シーン | オブジェクトコンテナ |
| `examples/jsm/controls/OrbitControls.js` | カメラ制御 | マウス/タッチ操作 |
| `examples/jsm/loaders/GLTFLoader.js` | GLTFローダー | 3Dモデル読み込み |

### C. 参考資料

- [three.js公式ドキュメント](https://threejs.org/docs/)
- [three.js公式マニュアル](https://threejs.org/manual/)
- [three.js GitHubリポジトリ](https://github.com/mrdoob/three.js)
- [three.js Wiki](https://github.com/mrdoob/three.js/wiki)
- [three.js Discord](https://discord.gg/56GBJwAnUS)
- [MDN WebGL Tutorial](https://developer.mozilla.org/en-US/docs/Web/API/WebGL_API/Tutorial)
- [WebGPU Specification](https://www.w3.org/TR/webgpu/)
