# セキュリティ設計書

## 概要

本ドキュメントはThree.js (v0.182.0) のセキュリティ設計に関する文書です。Three.jsはクライアントサイドで動作するJavaScript 3Dグラフィックスライブラリであり、WebGL/WebGPUを活用した3Dレンダリング機能を提供します。サーバーサイドの認証・認可機能は本ライブラリの範囲外ですが、クライアントサイドセキュリティに関連する設計要素を本書で定義します。

## 認証設計

### 認証方式

Three.jsは純粋なクライアントサイドレンダリングライブラリであり、ユーザー認証機能は提供していません。認証が必要な場合は、Three.jsを使用するアプリケーション側で実装する必要があります。

ただし、外部リソース（3Dモデル、テクスチャなど）の読み込み時に認証情報を付与する機能が提供されています。

| 機能 | 実装クラス | 説明 |
| --- | --- | --- |
| withCredentials | Loader | XMLHttpRequest/Fetchにおける認証情報（Cookie等）の送信設定 |
| requestHeader | Loader | カスタムHTTPヘッダー（Authorizationヘッダー等）の設定 |

### セッション管理

Three.js自体はセッション管理機能を持ちません。

| 項目 | 設定値 | 備考 |
| --- | --- | --- |
| セッション有効期限 | N/A（ライブラリ範囲外） | アプリケーション側で管理 |
| セッション固定化対策 | N/A（ライブラリ範囲外） | アプリケーション側で管理 |

## 認可設計

### 権限体系

Three.jsはクライアントサイドライブラリのため、権限体系は定義されていません。

| ロール | 権限 | 説明 |
| --- | --- | --- |
| N/A | N/A | アプリケーション側で定義 |

### アクセス制御

Three.jsは直接的なアクセス制御機能を持ちませんが、以下の機能がリソースアクセスに関連します。

- **LoaderクラスのwithCredentials設定**: 認証が必要なリソースへのアクセス時に認証情報を含めるかどうかを制御
- **crossOrigin設定**: CORSポリシーに基づくクロスオリジンリソースアクセスの制御

## 通信セキュリティ

| 項目 | 対策 |
| --- | --- |
| HTTPS | ライブラリ自体はプロトコルに依存しない。HTTPSの使用はデプロイ環境で設定する |
| HSTS | アプリケーション/サーバー側で設定 |
| CORS | crossOrigin属性による制御をサポート（デフォルト: 'anonymous'） |
| Credentials | withCredentials設定による認証情報送信の制御（デフォルト: false） |
| Fetch API | AbortSignalによるリクエストキャンセル機能を実装 |

### CORSの実装詳細

Three.jsのLoaderクラスは以下のCORS関連設定を提供します。

```javascript
// src/loaders/Loader.js
this.crossOrigin = 'anonymous';
this.withCredentials = false;
```

| 設定項目 | デフォルト値 | 説明 |
| --- | --- | --- |
| crossOrigin | 'anonymous' | CORS認証モード |
| withCredentials | false | 認証情報（Cookie、TLS証明書等）の送信 |

## データセキュリティ

### 暗号化

| 対象 | 暗号化方式 |
| --- | --- |
| 通信 | TLS（アプリケーション/サーバー側で設定） |
| 保存データ | N/A（ライブラリ範囲外） |

### 機密情報管理

Three.jsライブラリ自体は機密情報を保持しません。リソース読み込み時に設定されるカスタムヘッダー（requestHeader）はアプリケーション側の責任で管理する必要があります。

#### キャッシュ機能

Three.jsはリソースキャッシュ機能（`Cache`クラス）を提供しますが、デフォルトでは無効化されています。

```javascript
// src/loaders/Cache.js
const Cache = {
  enabled: false,  // デフォルトで無効
  files: {},
  // ...
};
```

キャッシュを有効にする場合は、機密性の高いリソースがキャッシュされないよう注意が必要です。

## 入出力対策

| 脅威 | 対策 |
| --- | --- |
| XSS | innerHTML/outerHTML/eval等の危険な関数の使用を最小限に抑制 |
| SQLインジェクション | N/A（データベース操作機能なし） |
| CSRF | N/A（状態変更APIなし） |
| コードインジェクション | ScriptableNodeでFunction()コンストラクタを使用（信頼できるコードのみ実行すべき） |
| DOMParserインジェクション | FileLoaderでDOMParserを使用（documentレスポンス処理時） |

### 注意すべき実装

#### ScriptableNode

`ScriptableNode`クラス（`src/nodes/code/ScriptableNode.js`）では、動的コード実行のために`Function()`コンストラクタが使用されています。

```javascript
// src/nodes/code/ScriptableNode.js:613
this._method = new Function( ...parametersProps, code );
```

この機能を使用する場合、信頼できないコードが実行されないよう、アプリケーション側での入力検証が必要です。

#### DOMParser

`FileLoader`（`src/loaders/FileLoader.js`）では、XMLやHTMLドキュメントの解析にDOMParserが使用されています。

```javascript
// src/loaders/FileLoader.js:243-244
const parser = new DOMParser();
return parser.parseFromString( text, mimeType );
```

これはresponseType='document'の場合にのみ使用され、XSS攻撃のベクトルとなる可能性は低いですが、信頼できないソースからのドキュメント読み込みには注意が必要です。

### 入力検証

Three.jsは以下の入力検証を行います。

| 検証箇所 | 検証内容 |
| --- | --- |
| JSONパース | ObjectLoaderでのJSON.parse例外処理 |
| URL解析 | LoadingManager.resolveURL()によるURL正規化 |
| 型変換 | getTypedArray()による型安全な配列変換 |
| メタデータ検証 | ObjectLoaderでのmetadata.type検証 |

## 監査ログ

| ログ種別 | 記録内容 | 保持期間 |
| --- | --- | --- |
| N/A | N/A（ライブラリ機能として監査ログは提供されていない） | N/A |

Three.jsはコンソール出力（`warn()`, `error()`, `log()`）機能を持ちますが、これらは開発者向けのデバッグ目的であり、監査ログではありません。

## 静的解析・脆弱性検知

### CodeQL

プロジェクトではGitHub CodeQLによる静的コード解析が設定されています。

| 項目 | 設定 |
| --- | --- |
| 解析対象 | JavaScript |
| 実行タイミング | dev ブランチへのpush/PR、週次スケジュール |
| クエリセット | security-and-quality |

### ESLint

ESLintによるコード品質チェックが実施されています。

| ルール | 設定 |
| --- | --- |
| no-throw-literal | error |
| no-duplicate-imports | error |
| no-irregular-whitespace | error |

## 依存関係セキュリティ

### 本番依存関係

Three.jsはランタイム依存関係を持ちません（devDependenciesのみ）。

### 開発依存関係

| パッケージ | バージョン | 用途 |
| --- | --- | --- |
| eslint | ^9.0.0 | コード品質チェック |
| rollup | ^4.6.0 | ビルドツール |
| puppeteer | ^24.25.0 | E2Eテスト |

## 保護フォルダー

プロジェクトでは`build`および`docs`フォルダーがプルリクエストで保護されています。オーナー・コラボレーター以外はこれらのフォルダーを変更できません。

## 脆弱性報告

セキュリティ脆弱性は非公開で報告する必要があります。

- **連絡先**: hello@mrdoob.com
- **対応期間**: 90日間

## 備考

- Three.jsはクライアントサイドライブラリであり、サーバーサイドセキュリティ機能は範囲外です
- 本番環境での使用時は、アプリケーション全体のセキュリティ設計を別途行う必要があります
- ScriptableNode等の動的コード実行機能を使用する場合は、信頼できない入力を許可しないよう注意が必要です
- 外部リソースの読み込みにはCORS設定の適切な構成が必要です
