# 機能設計書 10-Uniform

## 概要

本ドキュメントは、Three.jsライブラリにおけるシェーダーユニフォーム変数を管理するUniformクラスの機能設計について記述する。Uniformは、ShaderMaterialで使用されるグローバルシェーダー変数をラップし、シェーダープログラムへの値の受け渡しを容易にする。

### 本機能の処理概要

Uniformクラスは、シェーダーに渡すユニフォーム変数を表現するシンプルなラッパークラスである。任意の値（数値、ベクトル、行列、テクスチャなど）を保持し、シェーダープログラムに転送される。

**業務上の目的・背景**：WebGL/WebGPUシェーダーでは、JavaScript側からシェーダーにデータを渡すためにユニフォーム変数を使用する。Uniformクラスは、この転送を型安全かつ一貫性のある方法で行うための抽象化を提供する。特にShaderMaterialでカスタムシェーダーを使用する際に、ユニフォーム変数の管理を簡素化する。

**機能の利用シーン**：
- ShaderMaterialでのカスタムユニフォーム定義
- アニメーションで変化するパラメータのシェーダーへの受け渡し
- テクスチャのシェーダーへのバインド
- 時間、解像度などの共通パラメータの管理

**主要な処理内容**：
1. **値の保持**: valueプロパティによるユニフォーム値の保持
2. **クローン**: clone()メソッドによるユニフォームの複製

**関連システム・外部連携**：
- ShaderMaterialのuniformsプロパティ
- WebGLRenderer/WebGPURendererによるシェーダーへの値転送
- GLSLのuniform変数宣言

**権限による制御**：特になし（ライブラリレベルの機能のため、アプリケーション側での権限管理は行わない）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 23 | WebGL上級サンプル | 補助機能 | カスタムシェーダーでのユニフォーム使用 |

## 機能種別

データ管理 / シェーダー連携

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| value | any | Yes | ユニフォーム値 | 任意の型（数値、Vector、Matrix、Texture等） |

### 入力データソース

- コンストラクタ引数
- valueプロパティへの直接代入
- ShaderMaterialのuniformsオブジェクトリテラル

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| value | any | 保持されているユニフォーム値 |

### 出力先

- WebGLRenderer/WebGPURendererによるシェーダーへの転送
- clone()の戻り値

## 処理フロー

### 処理シーケンス

```
1. Uniformインスタンス生成
   └─ コンストラクタ(value)
       └─ this.value = value

2. 値の使用
   └─ ShaderMaterial.uniforms設定
   └─ レンダラーがシェーダーに転送

3. 値の更新
   └─ uniform.value = newValue
   └─ 次回レンダリング時に自動反映

4. 複製
   └─ clone()
       └─ valueがclone()を持つ場合はvalue.clone()
       └─ 持たない場合はそのまま使用
       └─ 新しいUniformインスタンスを返す
```

### フローチャート

```mermaid
flowchart TD
    A[Uniform生成] --> B[value設定]
    B --> C{ShaderMaterialで使用?}
    C -->|Yes| D[uniformsオブジェクトに登録]
    D --> E[レンダリング]
    E --> F[シェーダーに転送]
    F --> G{値更新?}
    G -->|Yes| H[uniform.value = newValue]
    H --> E
    G -->|No| I{clone?}
    I -->|Yes| J{value.clone存在?}
    J -->|Yes| K[value.clone()使用]
    J -->|No| L[valueをそのまま使用]
    K --> M[新Uniformを返す]
    L --> M
    I -->|No| E
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 任意型対応 | valueには任意の型を格納可能 | 常時 |
| BR-02 | clone対応 | valueがclone()メソッドを持つ場合、clone()で値もクローンされる | clone()呼び出し時 |
| BR-03 | 参照共有 | valueがclone()を持たない場合、clone()後も参照を共有 | clone()呼び出し時 |
| BR-04 | 宣言形式 | ShaderMaterialでは{value: x}形式またはnew Uniform(x)形式で宣言可能 | ShaderMaterial使用時 |

### 計算ロジック

**clone()の実装:**
```javascript
clone() {
    return new Uniform(this.value.clone === undefined ? this.value : this.value.clone());
}
```

## データベース操作仕様

### 操作別データベース影響一覧

該当なし（Uniformはデータベースを使用しない）

## エラー処理

### エラーケース一覧

該当なし（Uniformは明示的なエラー処理を行わない）

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（メモリ上のオブジェクト操作のみ）

## パフォーマンス要件

- Uniformの値更新は軽量な操作
- 頻繁に更新されるユニフォーム（時間など）でもオーバーヘッドは最小限
- レンダラーは変更されたユニフォームのみを効率的にGPUに転送

## セキュリティ考慮事項

該当なし（クライアントサイドのグラフィックスライブラリ）

## 備考

- UniformはShaderMaterialのuniformsオプションで使用
- `{ value: x }` 形式でも `new Uniform(x)` 形式でも同等に動作
- WebGLRendererでのみサポート（WebGPURendererではNode.jsベースのマテリアルシステムを使用）
- 配列ユニフォームや構造体ユニフォームにも対応

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: クラス構造を理解する

Uniformの基本構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Uniform.js | `src/core/Uniform.js` | クラス全体の構造（非常にシンプル） |

**読解のコツ**: Uniformは最もシンプルなThree.jsクラスの一つ。valueプロパティとclone()メソッドのみを持つ。

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Uniform.js | `src/core/Uniform.js` | コンストラクタでのvalue設定 |

**主要処理フロー**:
- **21-30行目**: コンストラクタ
- **28行目**: this.value = value

#### Step 3: クローンを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Uniform.js | `src/core/Uniform.js` | clone()メソッド |

**主要処理フロー**:
- **38-42行目**: clone()
- **40行目**: value.clone存在チェックと条件分岐

### プログラム呼び出し階層図

```
Uniform
    │
    ├─ constructor(value)
    │      └─ this.value = value
    │
    └─ clone()
           ├─ [if value.clone exists]
           │      └─ return new Uniform(value.clone())
           │
           └─ [if value.clone does not exist]
                  └─ return new Uniform(value)
```

### データフロー図

```
[入力]                      [処理]                        [出力]

value ─────────────→ constructor() ────→ Uniform
(number/Vector/                              │
 Matrix/Texture/                             │
 etc.)                                       │
                                             ↓
                                      ┌────────────┐
                                      │  value     │
                                      └────────────┘
                                             │
                                             ↓
                    ┌─────────────────────────────────────────┐
                    │         ShaderMaterial                  │
                    │                                         │
                    │ uniforms: {                             │
                    │   time: { value: 0.0 },                │
                    │   resolution: new Uniform(new Vector2())│
                    │ }                                       │
                    └─────────────────────────────────────────┘
                                             │
                                             ↓
                    ┌─────────────────────────────────────────┐
                    │         WebGLRenderer                   │
                    │                                         │
                    │ シェーダーにユニフォーム値を転送        │
                    └─────────────────────────────────────────┘
                                             │
                                             ↓
                    ┌─────────────────────────────────────────┐
                    │         GLSL Shader                     │
                    │                                         │
                    │ uniform float time;                     │
                    │ uniform vec2 resolution;                │
                    └─────────────────────────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Uniform.js | `src/core/Uniform.js` | ソース | Uniformクラス本体 |
| ShaderMaterial.js | `src/materials/ShaderMaterial.js` | ソース | ユニフォームの使用元 |
| WebGLUniforms.js | `src/renderers/webgl/WebGLUniforms.js` | ソース | シェーダーへの転送処理 |

### 使用例

```javascript
// 方法1: オブジェクトリテラル形式
const material = new THREE.ShaderMaterial({
    uniforms: {
        time: { value: 0.0 },
        resolution: { value: new THREE.Vector2() }
    },
    vertexShader: vertexShaderCode,
    fragmentShader: fragmentShaderCode
});

// 方法2: Uniformクラス使用
const material = new THREE.ShaderMaterial({
    uniforms: {
        time: new THREE.Uniform(0.0),
        resolution: new THREE.Uniform(new THREE.Vector2())
    },
    vertexShader: vertexShaderCode,
    fragmentShader: fragmentShaderCode
});

// アニメーションループでの更新
function animate() {
    material.uniforms.time.value = performance.now() / 1000;
    renderer.render(scene, camera);
    requestAnimationFrame(animate);
}
```
