# 機能設計書 102-TextureLoader

## 概要

本ドキュメントは、Three.jsライブラリにおけるテクスチャ読み込み機能「TextureLoader」の設計について記述する。TextureLoaderは、画像ファイルを読み込んでThree.jsのTextureオブジェクトとして返すローダークラスである。

### 本機能の処理概要

**業務上の目的・背景**：3Dグラフィックスにおいて、オブジェクトの表面にテクスチャ（画像）を貼り付けることは基本的な表現手法である。TextureLoaderは、画像ファイルを読み込み、マテリアルに適用可能なTextureオブジェクトに変換する機能を提供する。これにより、開発者は簡潔なコードでテクスチャマッピングを実現できる。

**機能の利用シーン**：
- 3Dオブジェクトの表面テクスチャの適用
- 法線マップ、バンプマップなどの詳細マップの読み込み
- 環境マップ（等距円筒図法）の読み込み
- UIスプライトやパーティクル用テクスチャの読み込み
- 地形やキャラクターのディフューズマップの適用

**主要な処理内容**：
1. 新規Textureオブジェクトの作成
2. 内部でImageLoaderを使用して画像ファイルを読み込み
3. 読み込んだ画像をTextureのimageプロパティに設定
4. texture.needsUpdate = trueを設定してGPUへのアップロードをトリガー
5. コールバック関数へのTexture渡し

**関連システム・外部連携**：
- ImageLoader: 内部で使用される画像読み込みローダー
- Texture: 出力されるテクスチャオブジェクト
- LoadingManager: 読み込み進捗管理
- Cache: ImageLoader経由でのキャッシュ利用

**権限による制御**：特になし（ブラウザのセキュリティポリシーに依存）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | Sidebar - Scene | 補助機能 | 背景テクスチャの設定 |
| 6 | Sidebar - Material | 補助機能 | テクスチャマップの設定 |

## 機能種別

データ読み込み / リソースローダー / テクスチャ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| url | string | Yes | 読み込む画像のパス/URL（Data URIも可） | なし |
| onLoad | function(Texture) | No | 読み込み完了時のコールバック | なし |
| onProgress | function | No | 進捗コールバック（このローダーでは未サポート） | なし |
| onError | function | No | エラー発生時のコールバック | なし |
| manager | LoadingManager | No | ローディングマネージャー | なし |

### 入力データソース

- ローカルファイルシステム上の画像ファイル
- リモートサーバー上の画像ファイル（URL指定）
- Data URI形式の埋め込み画像データ
- サポート形式: JPEG, PNG, GIF, WebP等（ブラウザサポートに依存）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| texture | Texture | 読み込まれた画像を含むテクスチャオブジェクト |

### 出力先

- onLoadコールバック関数への引数として渡される
- load()メソッドの戻り値として即座に返却（この時点ではimageは未設定）

## 処理フロー

### 処理シーケンス

```
1. load()メソッド呼び出し
   └─ 新規Textureオブジェクト作成
2. ImageLoaderインスタンス作成
   └─ crossOrigin設定の継承
   └─ path設定の継承
3. ImageLoader.load()呼び出し
   └─ 画像の非同期読み込み開始
4. 画像読み込み完了時
   └─ texture.image = image
   └─ texture.needsUpdate = true
   └─ onLoadコールバック実行（定義されている場合）
5. Textureオブジェクトを即座に返却
   └─ 画像読み込み前でも使用可能（プレースホルダー）
```

### フローチャート

```mermaid
flowchart TD
    A[load開始] --> B[new Texture作成]
    B --> C[ImageLoader作成]
    C --> D[crossOrigin/path設定継承]
    D --> E[ImageLoader.load呼び出し]
    E --> F[texture返却]
    E --> G{画像読み込み}
    G -->|成功| H[texture.image = image]
    H --> I[texture.needsUpdate = true]
    I --> J{onLoad定義?}
    J -->|Yes| K[onLoad実行]
    J -->|No| L[終了]
    K --> L
    G -->|失敗| M[onError実行]
    M --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-102-1 | 即時返却 | Textureオブジェクトは画像読み込み前に即座に返却される | 常時 |
| BR-102-2 | needsUpdate設定 | 画像読み込み完了時にneedsUpdate = trueを設定 | 読み込み成功時 |
| BR-102-3 | 設定継承 | ImageLoaderにcrossOriginとpathを継承 | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

該当なし（ファイル読み込み機能のため）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 画像読み込みエラー | ネットワークエラー、ファイル不存在、CORS違反等 | onErrorコールバック実行 |

### リトライ仕様

自動リトライ機能なし（アプリケーション側で実装が必要）

## トランザクション仕様

該当なし

## パフォーマンス要件

- ImageLoaderのキャッシュ機構を活用
- 非同期読み込みによるメインスレッドのブロック回避
- Textureの即時返却により、シーン構築を先行可能

## セキュリティ考慮事項

- ImageLoaderを通じてCORS対応
- crossOrigin属性の設定が可能

## 備考

- r84以降、プログレスイベントのサポートは廃止されている
- loadAsync()メソッドによるPromiseベースの読み込みも可能（Loaderクラスから継承）
- 返却されたTextureを使用する際、画像読み込み完了前は空のテクスチャとして表示される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

TextureLoaderが出力するTextureクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Texture.js | `src/textures/Texture.js` | imageプロパティ、needsUpdateフラグ、各種テクスチャ設定 |

**読解のコツ**: Textureクラスはimageプロパティに画像を格納し、needsUpdateフラグでGPUへの転送をトリガーする。

#### Step 2: 依存するImageLoaderを理解する

TextureLoaderが内部で使用するImageLoaderの仕組みを把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ImageLoader.js | `src/loaders/ImageLoader.js` | load()メソッドの実装、キャッシュ機構 |

**主要処理フロー**:
- **45-163行目**: load()メソッドの実装全体

#### Step 3: TextureLoaderの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | TextureLoader.js | `src/loaders/TextureLoader.js` | シンプルなラッパー実装、Texture生成、ImageLoader呼び出し |

**主要処理フロー**:
- **21行目**: class TextureLoader extends Loader - 基底クラス継承
- **28-31行目**: コンストラクタ - LoadingManager受け取り
- **46-68行目**: load()メソッドの実装
- **48行目**: new Texture() - 空のテクスチャ作成
- **50-52行目**: ImageLoader設定（crossOrigin, path継承）
- **54-65行目**: ImageLoader.load()呼び出しとコールバック設定
- **56行目**: texture.image = image - 画像設定
- **57行目**: texture.needsUpdate = true - 更新フラグ設定
- **67行目**: return texture - 即時返却

### プログラム呼び出し階層図

```
TextureLoader.load(url, onLoad, onProgress, onError)
    │
    ├─ new Texture()
    │
    ├─ new ImageLoader(this.manager)
    │      ├─ loader.setCrossOrigin(this.crossOrigin)
    │      └─ loader.setPath(this.path)
    │
    └─ loader.load(url, callback, onProgress, onError)
           │
           └─ callback(image)
                  ├─ texture.image = image
                  ├─ texture.needsUpdate = true
                  └─ onLoad(texture) [if defined]
```

### データフロー図

```
[入力]               [処理]                    [出力]

URL/Data URI ──▶ TextureLoader.load() ──▶ Texture
                       │
                       ├── new Texture() (空のテクスチャ生成)
                       │
                       └── ImageLoader.load()
                                │
                                └── image読み込み完了
                                        │
                                        ├── texture.image = image
                                        └── texture.needsUpdate = true
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| TextureLoader.js | `src/loaders/TextureLoader.js` | ソース | テクスチャローダー本体（75行） |
| ImageLoader.js | `src/loaders/ImageLoader.js` | ソース | 内部で使用する画像ローダー |
| Texture.js | `src/textures/Texture.js` | ソース | 出力されるテクスチャクラス |
| Loader.js | `src/loaders/Loader.js` | ソース | ローダー基底クラス |
| LoadingManager.js | `src/loaders/LoadingManager.js` | ソース | 読み込み進捗管理 |
