# 機能設計書 103-CubeTextureLoader

## 概要

本ドキュメントは、Three.jsライブラリにおけるキューブマップテクスチャ読み込み機能「CubeTextureLoader」の設計について記述する。CubeTextureLoaderは、6枚の画像ファイルを読み込んでCubeTextureオブジェクトを生成するローダークラスである。

### 本機能の処理概要

**業務上の目的・背景**：3Dグラフィックスにおいて、環境マッピング（反射や屈折の表現）やスカイボックス（背景の360度パノラマ表現）には、キューブマップテクスチャが必要となる。CubeTextureLoaderは、立方体の6面に対応する6枚の画像を読み込み、CubeTextureとして統合する機能を提供する。

**機能の利用シーン**：
- スカイボックス/スカイスフィアの背景設定
- 環境マッピングによる金属・ガラスの反射表現
- 屈折マッピングによる透明物体の屈折表現
- 環境光の間接照明シミュレーション
- IBL（Image-Based Lighting）のための環境マップ

**主要な処理内容**：
1. 新規CubeTextureオブジェクトの作成（sRGBカラースペース設定）
2. 内部でImageLoaderを使用して6枚の画像を並列読み込み
3. 各画像をCubeTextureのimages配列の対応位置に設定
4. 全6枚の読み込み完了時にneedsUpdate = trueを設定
5. コールバック関数へのCubeTexture渡し

**関連システム・外部連携**：
- ImageLoader: 内部で使用される画像読み込みローダー
- CubeTexture: 出力されるキューブマップテクスチャオブジェクト
- LoadingManager: 読み込み進捗管理
- Scene: 背景として使用可能

**権限による制御**：特になし（ブラウザのセキュリティポリシーに依存）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし（プログラムからの利用が主） |

## 機能種別

データ読み込み / リソースローダー / キューブマップ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| urls | Array<string> | Yes | 6枚の画像URL配列（順序: +X, -X, +Y, -Y, +Z, -Z） | 配列長6 |
| onLoad | function(CubeTexture) | No | 全画像読み込み完了時のコールバック | なし |
| onProgress | function | No | 進捗コールバック（このローダーでは未サポート） | なし |
| onError | function | No | エラー発生時のコールバック | なし |
| manager | LoadingManager | No | ローディングマネージャー | なし |

### 入力データソース

6枚の画像ファイル（以下の順序で指定）：
1. pos-x (右面: +X方向)
2. neg-x (左面: -X方向)
3. pos-y (上面: +Y方向)
4. neg-y (下面: -Y方向)
5. pos-z (前面: +Z方向)
6. neg-z (後面: -Z方向)

**注意**: Three.jsは右手座標系を使用するため、環境マップでは pos-x と neg-x が入れ替わる。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| texture | CubeTexture | 6枚の画像を含むキューブマップテクスチャ |

### 出力先

- onLoadコールバック関数への引数として渡される
- load()メソッドの戻り値として即座に返却

## 処理フロー

### 処理シーケンス

```
1. load()メソッド呼び出し
   └─ 新規CubeTextureオブジェクト作成
   └─ colorSpace = SRGBColorSpace設定
2. ImageLoaderインスタンス作成
   └─ crossOrigin設定の継承
   └─ path設定の継承
3. 6枚の画像を並列で読み込み開始
   └─ loadTexture(i)を0〜5で呼び出し
4. 各画像読み込み完了時
   └─ texture.images[i] = image
   └─ loadedカウンタをインクリメント
5. 全6枚読み込み完了時（loaded === 6）
   └─ texture.needsUpdate = true
   └─ onLoadコールバック実行
6. CubeTextureオブジェクトを即座に返却
```

### フローチャート

```mermaid
flowchart TD
    A[load開始] --> B[new CubeTexture作成]
    B --> C[colorSpace = SRGBColorSpace]
    C --> D[ImageLoader作成・設定継承]
    D --> E[for i = 0 to 5]
    E --> F[loadTexture i 呼び出し]
    F --> G[ImageLoader.load urls i]
    G --> H{読み込み成功?}
    H -->|Yes| I[texture.images i = image]
    I --> J[loaded++]
    J --> K{loaded === 6?}
    K -->|Yes| L[texture.needsUpdate = true]
    L --> M[onLoad実行]
    K -->|No| N[待機]
    H -->|No| O[onError実行]
    E --> P[texture返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-103-1 | sRGBカラースペース | CubeTextureはデフォルトでSRGBColorSpaceに設定 | 常時 |
| BR-103-2 | 6枚同時読み込み | 全6枚の画像を並列で読み込み開始 | 常時 |
| BR-103-3 | 完了判定 | loaded === 6 で全画像読み込み完了を判定 | 常時 |
| BR-103-4 | 座標系変換 | Three.jsは右手座標系のため、環境マップのpos-x/neg-xが入れ替わる | 環境マップ使用時 |

### 計算ロジック

読み込み完了判定：`loaded === 6`

## データベース操作仕様

該当なし（ファイル読み込み機能のため）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 画像読み込みエラー | いずれかの画像でネットワークエラー、ファイル不存在、CORS違反等 | onErrorコールバック実行 |

### リトライ仕様

自動リトライ機能なし（アプリケーション側で実装が必要）

## トランザクション仕様

該当なし

## パフォーマンス要件

- 6枚の画像を並列で読み込み、読み込み時間を最適化
- ImageLoaderのキャッシュ機構を活用

## セキュリティ考慮事項

- ImageLoaderを通じてCORS対応
- crossOrigin属性の設定が可能

## 備考

- r84以降、プログレスイベントのサポートは廃止
- 画像の順序は pos-x, neg-x, pos-y, neg-y, pos-z, neg-z（左手座標系規約）
- Three.jsは右手座標系を使用するため、環境マップ使用時はpos-xとneg-xが入れ替わることに注意

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

CubeTextureLoaderが出力するCubeTextureクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CubeTexture.js | `src/textures/CubeTexture.js` | imagesプロパティ（6要素配列）、Textureからの継承 |

**読解のコツ**: CubeTextureはimagesプロパティに6枚の画像を配列で格納する。配列のインデックスが立方体の面に対応。

#### Step 2: 依存するImageLoaderを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ImageLoader.js | `src/loaders/ImageLoader.js` | load()メソッドの実装、非同期読み込み |

#### Step 3: CubeTextureLoaderの実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CubeTextureLoader.js | `src/loaders/CubeTextureLoader.js` | 6枚の並列読み込み、完了判定ロジック |

**主要処理フロー**:
- **32行目**: class CubeTextureLoader extends Loader
- **39-43行目**: コンストラクタ
- **59-98行目**: load()メソッドの実装
- **61行目**: new CubeTexture() - 空のキューブテクスチャ作成
- **62行目**: texture.colorSpace = SRGBColorSpace - カラースペース設定
- **64-66行目**: ImageLoader設定
- **68行目**: let loaded = 0 - 読み込み完了カウンタ
- **70-88行目**: loadTexture(i)関数 - 個別画像読み込み
- **74行目**: texture.images[i] = image - 画像設定
- **76行目**: loaded++ - カウンタインクリメント
- **78-82行目**: loaded === 6 での完了判定
- **90-94行目**: 6枚分のloadTexture呼び出し
- **96行目**: return texture - 即時返却

### プログラム呼び出し階層図

```
CubeTextureLoader.load(urls, onLoad, onProgress, onError)
    │
    ├─ new CubeTexture()
    │      └─ texture.colorSpace = SRGBColorSpace
    │
    ├─ new ImageLoader(this.manager)
    │      ├─ loader.setCrossOrigin(this.crossOrigin)
    │      └─ loader.setPath(this.path)
    │
    └─ for (i = 0; i < urls.length; i++)
           │
           └─ loadTexture(i)
                  │
                  └─ loader.load(urls[i], callback, undefined, onError)
                         │
                         └─ callback(image)
                                ├─ texture.images[i] = image
                                ├─ loaded++
                                └─ if (loaded === 6)
                                       ├─ texture.needsUpdate = true
                                       └─ onLoad(texture)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

URL[0] (pos-x) ──┐
URL[1] (neg-x) ──┤
URL[2] (pos-y) ──┼──▶ CubeTextureLoader.load() ──▶ CubeTexture
URL[3] (neg-y) ──┤           │
URL[4] (pos-z) ──┤           ├── ImageLoader.load() x 6 (並列)
URL[5] (neg-z) ──┘           │
                             └── loaded === 6 で完了判定
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CubeTextureLoader.js | `src/loaders/CubeTextureLoader.js` | ソース | キューブマップローダー本体（104行） |
| ImageLoader.js | `src/loaders/ImageLoader.js` | ソース | 内部で使用する画像ローダー |
| CubeTexture.js | `src/textures/CubeTexture.js` | ソース | 出力されるキューブテクスチャクラス |
| Loader.js | `src/loaders/Loader.js` | ソース | ローダー基底クラス |
| constants.js | `src/constants.js` | ソース | SRGBColorSpace定数 |
