# 機能設計書 109-Cache

## 概要

本ドキュメントは、Three.jsライブラリにおけるリソースキャッシュ管理機能「Cache」の設計について記述する。CacheはFileLoaderで内部的に使用されるシンプルなキャッシュシステムであり、読み込み済みリソースを保持することで重複読み込みを防止する。

### 本機能の処理概要

**業務上の目的・背景**：3Dアプリケーションでは、同じリソース（テクスチャ、モデル、音声等）が複数回参照されることが多い。Cacheモジュールは、一度読み込んだリソースをメモリ上に保持し、再度同じURLが要求された際にネットワークアクセスを省略することで、パフォーマンスとユーザー体験を向上させる。

**機能の利用シーン**：
- 同一テクスチャを複数オブジェクトで共有する場合
- シーン再読み込み時の高速化
- 複数のローダーが同じリソースを参照する場合
- 大量のリソースを使用するアプリケーションの最適化

**主要な処理内容**：
1. enabled: キャッシュ機能の有効/無効切り替え
2. add(): キーとファイルのペアでキャッシュに追加
3. get(): キーを指定してキャッシュからファイルを取得
4. remove(): 指定キーのキャッシュエントリを削除
5. clear(): 全キャッシュをクリア

**関連システム・外部連携**：
- FileLoader: Cacheを内部で使用する主要ローダー
- ImageLoader: 画像キャッシュで使用
- 各種ローダー: FileLoaderを通じて間接的に利用

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 直接関連する画面なし（内部ユーティリティ） |

## 機能種別

ユーティリティ / キャッシュ管理 / シングルトン

## 入力仕様

### プロパティ

| プロパティ名 | 型 | デフォルト | 説明 |
|-------------|-----|----------|------|
| enabled | boolean | false | キャッシュ機能の有効/無効 |
| files | Object | {} | キャッシュされたファイルを保持する辞書 |

### メソッド

| メソッド名 | 引数 | 戻り値 | 説明 |
|-----------|------|--------|------|
| add | (key: string, file: Object) | void | キャッシュにエントリを追加 |
| get | (key: string) | Object\|undefined | キャッシュからエントリを取得 |
| remove | (key: string) | void | 指定キーのエントリを削除 |
| clear | () | void | 全エントリを削除 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| file | Object\|undefined | get()で取得されるキャッシュ済みファイル |

## 処理フロー

### 処理シーケンス（FileLoaderでの使用例）

```
1. FileLoader.load()呼び出し
2. Cache.get()でキャッシュ確認
   └─ enabled === falseの場合: undefined返却
   └─ enabled === trueの場合: files[key]を返却
3-a. キャッシュヒット時
   └─ キャッシュ済みデータを使用（ネットワークアクセス不要）
3-b. キャッシュミス時
   └─ ネットワークからファイル取得
   └─ Cache.add()でキャッシュに追加
```

### フローチャート

```mermaid
flowchart TD
    A[リソース要求] --> B{Cache.enabled?}
    B -->|No| C[ネットワーク取得]
    B -->|Yes| D[Cache.get]
    D --> E{キャッシュヒット?}
    E -->|Yes| F[キャッシュデータ返却]
    E -->|No| C
    C --> G[ファイル取得完了]
    G --> H{Cache.enabled?}
    H -->|Yes| I[Cache.add]
    H -->|No| J[処理終了]
    I --> J
    F --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-109-1 | デフォルト無効 | enabledはデフォルトでfalse | 常時 |
| BR-109-2 | 条件付き追加 | enabled === falseの場合、add()は何もしない | enabled === false |
| BR-109-3 | 条件付き取得 | enabled === falseの場合、get()はundefinedを返す | enabled === false |
| BR-109-4 | 無条件削除 | remove()/clear()はenabledに関わらず動作 | 常時 |
| BR-109-5 | キー上書き | 同じキーでadd()を呼ぶと上書きされる | 常時 |

### 計算ロジック

特になし

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

特になし（エラーを発生させない設計）

## トランザクション仕様

該当なし

## パフォーマンス要件

- シンプルなオブジェクトリテラルによる高速なアクセス
- メモリ使用量はキャッシュサイズに依存

## セキュリティ考慮事項

- キャッシュされたデータはメモリ上に保持される
- 機密データのキャッシュには注意が必要

## 備考

- Three.jsではデフォルトでキャッシュは無効
- 有効化するには `THREE.Cache.enabled = true` を設定
- FileLoaderはキーとして `file:${url}` を使用
- ImageLoaderはキーとして `image:${url}` を使用
- キャッシュクリアはメモリ解放に有効だが、参照が残っている場合はGCされない

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: Cacheオブジェクトの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Cache.js | `src/loaders/Cache.js` | 全体構造（88行と非常にシンプル） |

**読解のコツ**: Cacheはclassではなくオブジェクトリテラルとして定義されたシングルトンである。

**主要処理フロー**:
- **7行目**: const Cache = { ... } - オブジェクトリテラル定義
- **16行目**: enabled: false - デフォルト無効
- **24行目**: files: {} - キャッシュ辞書
- **34-42行目**: add() - 条件付きエントリ追加
- **51-59行目**: get() - 条件付きエントリ取得
- **67-71行目**: remove() - エントリ削除
- **78-82行目**: clear() - 全クリア

#### Step 2: FileLoaderでの使用方法を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FileLoader.js | `src/loaders/FileLoader.js` | 87行目: Cache.get(), 276行目: Cache.add() |

**主要処理フロー**:
- **87行目**: Cache.get(`file:${url}`) - キャッシュ確認
- **89-102行目**: キャッシュヒット時の処理
- **276行目**: Cache.add(`file:${url}`, data) - キャッシュ追加

#### Step 3: ImageLoaderでの使用方法を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ImageLoader.js | `src/loaders/ImageLoader.js` | 53行目: Cache.get(), 156行目: Cache.add() |

**主要処理フロー**:
- **53行目**: Cache.get(`image:${url}`) - 画像キャッシュ確認
- **156行目**: Cache.add(`image:${url}`, image) - 画像キャッシュ追加
- **119行目**: Cache.remove(`image:${url}`) - エラー時のキャッシュ削除

### プログラム呼び出し階層図

```
アプリケーションコード
    │
    └─ THREE.Cache.enabled = true  [有効化]
           │
FileLoader.load(url)
    │
    ├─ Cache.get(`file:${url}`)
    │      ├─ enabled === false → undefined
    │      └─ enabled === true → files[key] or undefined
    │
    ├─ [ネットワーク取得]
    │
    └─ Cache.add(`file:${url}`, data)
           ├─ enabled === false → return (何もしない)
           └─ enabled === true → files[key] = file
```

### データフロー図

```
[Cache構造]

Cache (シングルトン)
    │
    ├── enabled: boolean (デフォルト: false)
    │
    └── files: Object
              │
              ├── "file:url1" → data1
              ├── "file:url2" → data2
              ├── "image:url3" → image3
              └── ...

[操作フロー]

add(key, file)          get(key)            remove(key)      clear()
     │                       │                    │              │
     ▼                       ▼                    ▼              ▼
enabled チェック        enabled チェック      files[key]    files = {}
     │                       │                 削除
     ▼                       ▼
files[key] = file      return files[key]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Cache.js | `src/loaders/Cache.js` | ソース | キャッシュ管理本体（88行） |
| FileLoader.js | `src/loaders/FileLoader.js` | ソース | Cacheを使用する主要ローダー |
| ImageLoader.js | `src/loaders/ImageLoader.js` | ソース | 画像キャッシュで使用 |
