# 機能設計書 131-Box2

## 概要

本ドキュメントは、Three.jsの数学ライブラリにおける2D軸平行境界ボックス（AABB: Axis-Aligned Bounding Box）を表現するBox2クラスの機能設計書である。

### 本機能の処理概要

Box2クラスは、2D空間における軸平行境界ボックスを表現するためのクラスである。最小点（min）と最大点（max）の2つのVector2によってボックスの範囲を定義し、2Dオブジェクトの境界判定や衝突検出などに使用される。

**業務上の目的・背景**：3Dグラフィックスにおいて、2D空間での境界判定は、UIエレメントの配置、スクリーン座標でのマウスピッキング、2Dテクスチャ座標の範囲チェック、スプライトの衝突検出など様々な場面で必要とされる。Box2クラスは、これらの処理を効率的に実現するための基本的な幾何学プリミティブを提供する。

**機能の利用シーン**：
- 2DUIエレメントの境界判定
- テクスチャ座標（UV座標）の範囲計算
- 2Dスプライトの衝突検出
- スクリーン座標でのオブジェクト選択範囲の計算
- 2Dビューポートのクリッピング領域の定義

**主要な処理内容**：
1. 2D境界ボックスの生成と初期化（min/max座標の設定）
2. 点群やサイズからの境界ボックス設定
3. 点の包含判定・ボックスの包含判定
4. ボックス同士の交差判定と交差領域の計算
5. ボックスの拡張・縮小・移動
6. 点とボックスの距離計算
7. 複数ボックスの合成（union）

**関連システム・外部連携**：Three.jsの2D関連処理、特にVector2クラスと密接に連携する。3D処理においてはBox3クラスが対応する。

**権限による制御**：なし（純粋な数学ユーティリティクラス）

## 関連画面

本機能は数学ユーティリティであり、直接的な画面との関連はない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 内部計算処理として使用 |

## 機能種別

計算処理 / ジオメトリ演算

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| min | Vector2 | No | ボックスの下限座標 | デフォルト: (+Infinity, +Infinity) |
| max | Vector2 | No | ボックスの上限座標 | デフォルト: (-Infinity, -Infinity) |

### 入力データソース

コンストラクタ引数またはメソッド呼び出しによる直接設定

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isBox2 | boolean | Box2インスタンス識別フラグ（常にtrue） |
| min | Vector2 | ボックスの下限座標 |
| max | Vector2 | ボックスの上限座標 |

### 出力先

メモリ上のオブジェクトとして保持

## 処理フロー

### 処理シーケンス

```
1. 境界ボックスの初期化
   └─ min/maxをInfinityで初期化（空のボックス状態）
2. 点群からの境界設定（setFromPoints）
   └─ 各点でexpandByPointを呼び出し、min/maxを更新
3. 包含判定（containsPoint/containsBox）
   └─ 座標比較による判定
4. 交差判定（intersectsBox）
   └─ 分離軸定理に基づく判定
5. ボックス操作
   └─ 拡張、縮小、移動、合成などの幾何学演算
```

### フローチャート

```mermaid
flowchart TD
    A[Box2インスタンス生成] --> B{初期化方法}
    B -->|コンストラクタ| C[min/maxを直接設定]
    B -->|setFromPoints| D[点群から境界計算]
    B -->|setFromCenterAndSize| E[中心とサイズから計算]
    C --> F[境界ボックス確定]
    D --> F
    E --> F
    F --> G{操作種別}
    G -->|包含判定| H[containsPoint/containsBox]
    G -->|交差判定| I[intersectsBox]
    G -->|変形| J[expand/translate]
    G -->|合成| K[union/intersect]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 空ボックス表現 | min > maxの場合、ボックスは空とみなされる | isEmpty()判定時 |
| BR-002 | 等価境界 | min == maxの場合、1点を含むボックスとみなされる | containsPoint判定時 |
| BR-003 | 座標系 | 2D直交座標系を前提とする | 全操作 |

### 計算ロジック

**中心点の計算**:
```javascript
center = (min + max) * 0.5
```

**サイズの計算**:
```javascript
size = max - min
```

**点との距離計算**:
```javascript
clampedPoint = clamp(point, min, max)
distance = length(point - clampedPoint)
```

## データベース操作仕様

該当なし（純粋な計算処理クラス）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 空ボックス | min > maxの状態で演算 | isEmpty()で事前チェック |
| - | ゼロ除算 | サイズ0のボックスでgetParameter | 呼び出し側で事前チェック必要 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 全メソッドO(1)の計算量（setFromPointsを除く）
- setFromPointsはO(n)（nは点の数）
- メモリ使用量: Vector2 x 2 = 32バイト程度

## セキュリティ考慮事項

該当なし（純粋な数学計算）

## 備考

- Box3クラスは3D版の対応クラス
- 境界ボックスは衝突検出の高速化に広く使用される
- 空のボックスはInfinityを使用して表現される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Box2クラスは非常にシンプルな構造を持ち、2つのVector2（min, max）で境界を表現する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Vector2.js | `src/math/Vector2.js` | Box2が依存する2Dベクトルクラスの基本操作を理解する |
| 1-2 | Box2.js | `src/math/Box2.js` | クラスのプロパティ定義（min, max, isBox2）を確認 |

**読解のコツ**: Box2はmin/maxの2点で矩形領域を定義する。Infinityの使用は「空のボックス」を表現するための慣用的パターン。

#### Step 2: エントリーポイントを理解する

Box2クラスのコンストラクタとプロパティ初期化を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Box2.js | `src/math/Box2.js` | コンストラクタでのデフォルト値（Infinity）の意味を理解 |

**主要処理フロー**:
1. **16-41行目**: コンストラクタでmin/maxを初期化。デフォルトは空のボックス状態
2. **25行目**: isBox2フラグで型判定を可能にする

#### Step 3: 設定メソッドを理解する

ボックスの境界を設定する各種メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Box2.js | `src/math/Box2.js` | set(), setFromPoints(), setFromCenterAndSize()の実装 |

**主要処理フロー**:
- **51-58行目**: set() - 直接min/maxを設定
- **67-79行目**: setFromPoints() - 点群からmakeEmpty後にexpandByPointで拡張
- **89-97行目**: setFromCenterAndSize() - 中心とサイズから計算

#### Step 4: 判定メソッドを理解する

包含判定・交差判定のロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Box2.js | `src/math/Box2.js` | containsPoint(), containsBox(), intersectsBox()の実装 |

**主要処理フロー**:
- **146-152行目**: isEmpty() - max < minでの判定
- **233-238行目**: containsPoint() - 各軸で範囲内かを判定
- **247-252行目**: containsBox() - 他のボックスが完全に内包されているか判定
- **279-286行目**: intersectsBox() - 分離軸定理による交差判定

### プログラム呼び出し階層図

```
Box2
    │
    ├─ set()
    │      └─ Vector2.copy()
    │
    ├─ setFromPoints()
    │      ├─ makeEmpty()
    │      └─ expandByPoint() [loop]
    │             └─ Vector2.min() / Vector2.max()
    │
    ├─ setFromCenterAndSize()
    │      └─ Vector2.copy() / Vector2.sub() / Vector2.add()
    │
    ├─ containsPoint()
    │      └─ 座標比較
    │
    ├─ intersectsBox()
    │      └─ 座標比較（分離軸定理）
    │
    └─ distanceToPoint()
           └─ clampPoint()
                  └─ Vector2.clamp()
```

### データフロー図

```
[入力]               [処理]                    [出力]

Vector2[] ───▶ setFromPoints() ───▶ Box2 {min, max}
(点群)

Vector2 ───▶ containsPoint() ───▶ boolean
(テスト点)

Box2 ───────▶ intersectsBox() ───▶ boolean
(他のボックス)

Vector2 ───▶ distanceToPoint() ───▶ number
(テスト点)                          (距離)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Box2.js | `src/math/Box2.js` | ソース | 2D境界ボックスクラス本体 |
| Vector2.js | `src/math/Vector2.js` | ソース | 2Dベクトルクラス（依存） |
| Box3.js | `src/math/Box3.js` | ソース | 3D版の対応クラス（参考） |
