# 機能設計書 14-WebGPURenderer

## 概要

本ドキュメントは、Three.jsライブラリにおけるWebGPURenderer機能の設計を詳細に記述するものである。WebGPURendererは、次世代グラフィックスAPI「WebGPU」を使用した高性能レンダリングエンジンであり、WebGLRendererの後継として位置づけられている。

### 本機能の処理概要

WebGPURendererクラスは、WebGPU APIをプライマリバックエンドとして使用し、ブラウザがWebGPUをサポートしない場合はWebGL 2にフォールバックする次世代レンダリングエンジンである。ノードベースのシェーディング言語（TSL: Three.js Shading Language）と統合されており、より柔軟で高性能なレンダリングを実現する。

**業務上の目的・背景**：WebGLは2011年に登場したAPIであり、現代のGPUハードウェアの能力を十分に活用できない制約がある。WebGPUは、Vulkan/Metal/DirectX 12の設計思想を取り入れた次世代APIであり、マルチスレッド対応、コンピュートシェーダー、より効率的なリソース管理などを提供する。WebGPURendererはこれらの新機能を活用し、より高性能なレンダリングを実現する。

**機能の利用シーン**：高度な3Dグラフィックスアプリケーション、大規模なパーティクルシステム、GPGPUコンピューティング、複雑なポストプロセッシングエフェクト、VR/ARアプリケーションなど、高いパフォーマンスが求められるシーンで使用される。

**主要な処理内容**：
1. WebGPU/WebGLバックエンドの自動選択とフォールバック
2. ノードベースマテリアルシステム（TSL）との統合
3. コンピュートシェーダーのサポート
4. 非同期レンダリングパイプライン
5. マルチビューレンダリング（WebXR対応）
6. 高精度カラーバッファ（HalfFloatType）のデフォルト使用

**関連システム・外部連携**：WebGPU API、WebGL 2 API（フォールバック）、WebXR Device API、ノードシステム（TSL）。

**権限による制御**：特になし。ただしWebXR使用時はユーザーの明示的な許可が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 1 | エディタメイン画面 | 主機能 | WebGPU APIを使用したレンダリング処理 |
| 2 | Viewport | 主機能 | WebGPU使用時のプレビューレンダリング |
| 8 | Sidebar - Project | 主機能 | WebGPUレンダラー設定の管理 |
| 24 | WebGPUサンプル | 主機能 | WebGPU APIを使用したレンダリング |

## 機能種別

レンダリング処理 / GPU描画

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| logarithmicDepthBuffer | boolean | No | 対数深度バッファの有効化 | デフォルト: false |
| alpha | boolean | No | アルファチャンネルの有効化 | デフォルト: true |
| depth | boolean | No | 深度バッファの有効化 | デフォルト: true |
| stencil | boolean | No | ステンシルバッファの有効化 | デフォルト: false |
| antialias | boolean | No | アンチエイリアスの有効化 | デフォルト: false |
| samples | number | No | MSAAサンプル数 | デフォルト: 0（antialias時は4） |
| forceWebGL | boolean | No | WebGLバックエンドを強制使用 | デフォルト: false |
| multiview | boolean | No | WebXRマルチビューの有効化 | デフォルト: false |
| outputType | number | No | 出力テクスチャタイプ | デフォルト: デバイス推奨形式 |
| outputBufferType | number | No | 出力バッファタイプ | デフォルト: HalfFloatType |

### 入力データソース

- Scene: レンダリング対象のシーングラフ
- Camera: 視点を定義するカメラオブジェクト
- RenderTarget: オフスクリーンレンダリング先（オプション）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| domElement | HTMLCanvasElement | レンダリング結果が描画されるキャンバス |
| info | Info | レンダリング統計情報 |
| backend | Backend | 現在使用中のバックエンド（WebGPU/WebGL） |
| library | StandardNodeLibrary | ノードライブラリ |

### 出力先

- HTML5 Canvas（画面表示）
- RenderTarget（オフスクリーン）

## 処理フロー

### 処理シーケンス

```
1. WebGPURendererインスタンスの生成
   └─ バックエンド選択（WebGPU優先、WebGLフォールバック）

2. 初期化（非同期）
   ├─ init(): バックエンドの初期化
   └─ StandardNodeLibraryの設定

3. レンダリングループ
   ├─ setSize(): キャンバスサイズ設定
   ├─ setRenderTarget(): レンダー先設定（オプション）
   └─ render(scene, camera): シーン描画
       ├─ ノードマテリアルの構築
       ├─ コマンドエンコーダーの作成
       ├─ レンダーパスの実行
       └─ コマンドバッファのサブミット

4. コンピュート処理（オプション）
   └─ compute(): コンピュートシェーダー実行

5. リソース管理
   └─ dispose(): リソース解放
```

### フローチャート

```mermaid
flowchart TD
    A[WebGPURenderer生成] --> B{forceWebGL?}
    B -->|Yes| C[WebGLBackend使用]
    B -->|No| D{WebGPU利用可能?}
    D -->|Yes| E[WebGPUBackend使用]
    D -->|No| F[WebGLBackendにフォールバック]
    C --> G[Renderer基底クラス初期化]
    E --> G
    F --> G
    G --> H[StandardNodeLibrary設定]
    H --> I{init()完了?}
    I -->|Yes| J[render()可能]
    I -->|No| K[待機]
    K --> I
    J --> L[シーン描画]
    L --> M{compute要求?}
    M -->|Yes| N[コンピュートシェーダー実行]
    M -->|No| O[フレーム完了]
    N --> O
    O --> J
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | WebGPU優先 | WebGPU利用可能時はWebGPUを使用 | forceWebGL=false時 |
| BR-02 | 自動フォールバック | WebGPU未対応時はWebGL 2にフォールバック | WebGPU未対応ブラウザ |
| BR-03 | 高精度バッファ | outputBufferTypeのデフォルトはHalfFloatType | インスタンス生成時 |
| BR-04 | 非同期初期化 | init()でバックエンドを非同期初期化 | 初回レンダリング前 |
| BR-05 | ノード統合 | StandardNodeLibraryでノードマテリアルをサポート | マテリアル処理時 |

### 計算ロジック

- MSAAサンプル数 = antialias ? (samples || 4) : samples

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | WebGPU/WebGL未対応 | 両方がサポートされない | エラーメッセージ表示 |
| - | フォールバック警告 | WebGPU未対応でWebGLにフォールバック | コンソール警告 |
| - | コンテキスト喪失 | GPUリソース不足等 | 復旧処理 |

### リトライ仕様

- WebGPU初期化失敗時はWebGLに自動フォールバック

## トランザクション仕様

該当なし

## パフォーマンス要件

- WebGPUではWebGLより効率的なGPUリソース管理
- コンピュートシェーダーによるGPGPU処理
- HalfFloatTypeバッファで高品質レンダリング
- マルチビューレンダリングでVRパフォーマンス向上

## セキュリティ考慮事項

- WebGPU APIはブラウザサンドボックス内で動作
- クロスオリジンリソースはCORS設定が必要

## 備考

- Rendererクラスを継承
- isWebGPURendererフラグでタイプ判定可能
- THREE_DEVTOOLSとの統合サポート

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

WebGPURendererの構造とその依存関係を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | WebGPURenderer.js | `src/renderers/webgpu/WebGPURenderer.js` | クラス定義と継承関係 |
| 1-2 | Renderer.js | `src/renderers/common/Renderer.js` | 基底クラスの機能 |
| 1-3 | WebGPUBackend.js | `src/renderers/webgpu/WebGPUBackend.js` | WebGPUバックエンド |
| 1-4 | WebGLBackend.js | `src/renderers/webgl-fallback/WebGLBackend.js` | WebGLフォールバック |

**読解のコツ**: WebGPURendererはRendererクラスを継承し、バックエンド（WebGPUBackend/WebGLBackend）を注入する設計。まずRendererの共通機能を理解し、次にWebGPURenderer固有の初期化処理を読み解く。

#### Step 2: エントリーポイントを理解する

WebGPURendererの初期化とバックエンド選択を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | WebGPURenderer.js | `src/renderers/webgpu/WebGPURenderer.js` | コンストラクタ |

**主要処理フロー**:
1. **52行目**: コンストラクタ開始、parametersを受け取る
2. **54-58行目**: forceWebGLパラメータでバックエンドクラスを決定
3. **60-72行目**: WebGPU未対応時のフォールバック関数を設定
4. **74行目**: バックエンドインスタンス生成
5. **77行目**: super()でRenderer基底クラスを初期化
6. **85行目**: StandardNodeLibraryを設定
7. **94行目**: isWebGPURendererフラグをtrueに設定
8. **96-100行目**: THREE_DEVTOOLSへのイベント発行

#### Step 3: Renderer基底クラスを理解する

共通レンダリング機能を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Renderer.js | `src/renderers/common/Renderer.js` | 基底クラス |
| 3-2 | NodeManager.js | `src/renderers/common/nodes/NodeManager.js` | ノード管理 |
| 3-3 | RenderObjects.js | `src/renderers/common/RenderObjects.js` | レンダーオブジェクト管理 |

**主要処理フロー**:
- **Renderer**: autoClear、toneMapping、sortObjectsなどの共通プロパティ
- **NodeManager**: ノードマテリアルのビルドと更新
- **RenderObjects**: レンダリング対象オブジェクトの管理

### プログラム呼び出し階層図

```
WebGPURenderer
    │
    ├─ extends Renderer
    │       │
    │       ├─ init() ─────────────▶ Backend.init()
    │       │
    │       ├─ render() ───────────▶ renderScene()
    │       │       │
    │       │       ├─ NodeManager.updateBefore()
    │       │       │
    │       │       ├─ projectObject()
    │       │       │
    │       │       ├─ renderObjects()
    │       │       │
    │       │       └─ NodeManager.updateAfter()
    │       │
    │       └─ compute() ──────────▶ Backend.compute()
    │
    ├─ WebGPUBackend (primary)
    │       │
    │       ├─ createRenderPipeline()
    │       │
    │       ├─ createComputePipeline()
    │       │
    │       └─ submit()
    │
    └─ WebGLBackend (fallback)
            │
            └─ (WebGL 2 API calls)
```

### データフロー図

```
[入力]                        [処理]                            [出力]

parameters
    │
    ▼
WebGPURenderer() ────────────────────────────────────────────▶ バックエンド選択
    │
    ▼
Renderer.init() ─────────────▶ Backend.init() ───────────────▶ GPU初期化完了
    │
    ▼
Scene + Camera ──────────────▶ Renderer.render() ────────────▶ Canvas描画
    │                               │
    │                               ▼
    │                         NodeManager ───────────────────▶ シェーダー生成
    │                               │
    │                               ▼
    │                         Backend.submit() ──────────────▶ GPUコマンド実行
    │
    ▼
ComputeNode ─────────────────▶ Renderer.compute() ───────────▶ 計算結果
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| WebGPURenderer.js | `src/renderers/webgpu/WebGPURenderer.js` | ソース | メインレンダラークラス |
| Renderer.js | `src/renderers/common/Renderer.js` | ソース | 基底レンダラークラス |
| WebGPUBackend.js | `src/renderers/webgpu/WebGPUBackend.js` | ソース | WebGPUバックエンド |
| WebGLBackend.js | `src/renderers/webgl-fallback/WebGLBackend.js` | ソース | WebGLフォールバック |
| StandardNodeLibrary.js | `src/renderers/webgpu/nodes/StandardNodeLibrary.js` | ソース | 標準ノードライブラリ |
| NodeManager.js | `src/renderers/common/nodes/NodeManager.js` | ソース | ノード管理 |
| RenderObjects.js | `src/renderers/common/RenderObjects.js` | ソース | レンダーオブジェクト |
| Pipelines.js | `src/renderers/common/Pipelines.js` | ソース | パイプライン管理 |
| Bindings.js | `src/renderers/common/Bindings.js` | ソース | バインディング管理 |
| Info.js | `src/renderers/common/Info.js` | ソース | 統計情報 |
| TSL.js | `src/nodes/TSL.js` | ソース | Three.js Shading Language |
