# 機能設計書 141-MathUtils

## 概要

本ドキュメントは、Three.jsライブラリにおけるMathUtils（数学ユーティリティ）機能の設計について記述する。MathUtilsは3Dグラフィックス処理において頻繁に使用される数学的計算を効率的に行うためのユーティリティ関数群を提供する。

### 本機能の処理概要

MathUtilsは、3Dグラフィックスアプリケーション開発において必要となる様々な数学的計算を簡潔に実行できるユーティリティ関数の集合体である。

**業務上の目的・背景**：3Dグラフィックス開発では、角度変換、値のクランプ、線形補間、乱数生成など基本的な数学計算が頻繁に必要となる。これらの計算をライブラリとして標準化することで、開発者は複雑な数学的実装を意識することなく、直感的なAPIを通じて高品質な3Dアプリケーションを構築できる。また、UUID生成やパワーオブツー計算など、WebGL/WebGPUにおける特有の要件にも対応している。

**機能の利用シーン**：
- アニメーション制御における値の補間（lerp、damp、smoothstep）
- オブジェクトの回転処理での角度変換（degToRad、radToDeg）
- シェーダーやバッファ操作での値の正規化/非正規化
- オブジェクトID生成のためのUUID生成
- テクスチャサイズのパワーオブツー計算（WebGL制約対応）
- 物理シミュレーションでの乱数生成

**主要な処理内容**：
1. 角度変換（度数法/弧度法の相互変換）
2. 値のクランプ処理（最小値・最大値の範囲制限）
3. 線形補間およびスムーズ補間
4. 乱数生成（ランダム整数、浮動小数点、シード付き乱数）
5. UUID生成
6. パワーオブツー計算
7. 四元数のProper Euler角からの変換
8. 型付き配列に応じた値の正規化/非正規化

**関連システム・外部連携**：本機能は他のThree.js数学クラス（Vector3、Quaternion、Euler等）と連携して使用される。特にQuaternionクラスとの連携でsetQuaternionFromProperEuler関数が使用される。

**権限による制御**：本機能は純粋なユーティリティ関数群であり、権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUI画面に直接関連せず、内部計算処理として使用される |

## 機能種別

計算処理 / ユーティリティ関数群

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| value | number | Yes | 処理対象の数値 | なし（数値として扱われる） |
| min | number | Yes | 最小値（clamp等） | なし |
| max | number | Yes | 最大値（clamp等） | なし |
| t | number | Yes | 補間係数（lerp等） | 通常0-1の範囲 |
| degrees | number | Yes | 度数法の角度 | なし |
| radians | number | Yes | 弧度法の角度 | なし |
| array | TypedArray | Yes | 型付き配列（normalize/denormalize） | 対応型の配列 |

### 入力データソース

直接の関数呼び出しによる引数入力。ユーザーコードまたは他のThree.jsクラスからの呼び出し。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| 計算結果 | number | 各関数の計算結果 |
| UUID | string | generateUUID()によるUUID文字列 |
| boolean | boolean | isPowerOfTwo()の判定結果 |

### 出力先

関数の戻り値として呼び出し元に返却。

## 処理フロー

### 処理シーケンス

```
1. 関数呼び出し
   └─ 各ユーティリティ関数が引数を受け取る
2. 入力値の処理
   └─ 必要に応じて入力値の変換や検証
3. 数学計算の実行
   └─ 各関数固有のアルゴリズムによる計算
4. 結果の返却
   └─ 計算結果を戻り値として返却
```

### フローチャート

```mermaid
flowchart TD
    A[関数呼び出し] --> B{関数種別}
    B -->|角度変換| C[DEG2RAD/RAD2DEG係数を乗算]
    B -->|クランプ| D[Math.max/minで範囲制限]
    B -->|補間| E[補間計算式を適用]
    B -->|乱数| F[Math.randomまたはシード乱数生成]
    B -->|UUID| G[ランダム値から文字列生成]
    B -->|POT| H[対数計算でパワーオブツー算出]
    C --> I[結果返却]
    D --> I
    E --> I
    F --> I
    G --> I
    H --> I
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-141-01 | 角度変換定数 | DEG2RAD = PI/180、RAD2DEG = 180/PI | 角度変換関数使用時 |
| BR-141-02 | クランプ範囲 | min <= result <= max を保証 | clamp関数使用時 |
| BR-141-03 | 補間範囲 | t=0でx、t=1でyを返却 | lerp関数使用時 |
| BR-141-04 | UUID形式 | 8-4-4-4-12形式の小文字UUID | generateUUID使用時 |

### 計算ロジック

**線形補間（lerp）**：
```
result = (1 - t) * x + t * y
```

**スムーズステップ（smoothstep）**：
```
x = (x - min) / (max - min)
result = x * x * (3 - 2 * x)
```

**ダンプ補間（damp）**：
```
result = lerp(x, y, 1 - exp(-lambda * dt))
```

**ピンポン（pingpong）**：
```
result = length - abs(euclideanModulo(x, length * 2) - length)
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Error | normalize/denormalizeで非対応型配列 | "Invalid component type."エラー |
| - | Warning | setQuaternionFromProperEulerで未知のorder | コンソール警告を出力 |

### リトライ仕様

該当なし（同期的な計算処理のため）

## トランザクション仕様

該当なし（データベース操作を行わないため）

## パフォーマンス要件

- 各関数は定数時間 O(1) で完了
- ルックアップテーブル（_lut）を使用したUUID生成の高速化
- インライン可能な軽量関数設計

## セキュリティ考慮事項

- seededRandom関数は暗号学的に安全ではない（Mulberry32アルゴリズム使用）
- セキュリティ用途には使用不可

## 備考

- DEG2RAD、RAD2DEG定数はモジュール外からも直接インポート可能
- 個別関数も名前付きエクスポートとして利用可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 定数とルックアップテーブルを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | MathUtils.js | `src/math/MathUtils.js` | 定数DEG2RAD、RAD2DEG、_lutテーブルの定義（1-9行目） |

**読解のコツ**: _lutは0x00-0xffの16進数文字列のルックアップテーブルで、UUID生成時の高速化に使用される。

#### Step 2: 基本関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MathUtils.js | `src/math/MathUtils.js` | clamp関数（43-47行目）：Math.max/minの組み合わせ |
| 2-2 | MathUtils.js | `src/math/MathUtils.js` | lerp関数（116-120行目）：基本的な線形補間 |
| 2-3 | MathUtils.js | `src/math/MathUtils.js` | degToRad/radToDeg関数（264-280行目）：角度変換 |

**主要処理フロー**:
- **43-47行目**: clamp - Math.max(min, Math.min(max, value))で範囲制限
- **116-120行目**: lerp - (1-t)*x + t*y で線形補間
- **134-138行目**: damp - lerp + 指数減衰でフレームレート非依存の補間

#### Step 3: 高度な関数を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | MathUtils.js | `src/math/MathUtils.js` | smoothstep/smootherstep（167-196行目）：S字カーブ補間 |
| 3-2 | MathUtils.js | `src/math/MathUtils.js` | setQuaternionFromProperEuler（331-379行目）：四元数変換 |
| 3-3 | MathUtils.js | `src/math/MathUtils.js` | normalize/denormalize（388-473行目）：型付き配列の正規化 |

**主要処理フロー**:
- **167-176行目**: smoothstep - 3次多項式 x*x*(3-2*x) でS字カーブ
- **187-196行目**: smootherstep - 5次多項式で1次・2次微分が境界で0
- **242-256行目**: seededRandom - Mulberry32アルゴリズムで決定的乱数

### プログラム呼び出し階層図

```
MathUtils (オブジェクト)
    │
    ├─ 定数
    │      ├─ DEG2RAD
    │      └─ RAD2DEG
    │
    ├─ 基本関数
    │      ├─ clamp()
    │      ├─ euclideanModulo()
    │      ├─ lerp()
    │      └─ mapLinear()
    │
    ├─ 補間関数
    │      ├─ damp()
    │      ├─ pingpong()
    │      ├─ smoothstep()
    │      └─ smootherstep()
    │
    ├─ 乱数関数
    │      ├─ randInt()
    │      ├─ randFloat()
    │      ├─ randFloatSpread()
    │      └─ seededRandom()
    │
    ├─ 変換関数
    │      ├─ degToRad()
    │      ├─ radToDeg()
    │      ├─ normalize()
    │      └─ denormalize()
    │
    └─ その他
           ├─ generateUUID()
           ├─ isPowerOfTwo()
           ├─ ceilPowerOfTwo()
           ├─ floorPowerOfTwo()
           └─ setQuaternionFromProperEuler()
```

### データフロー図

```
[入力]               [処理]                    [出力]

数値 ───────────▶ clamp/lerp等 ───────────▶ 数値
                  基本演算

角度(度) ────────▶ degToRad ──────────────▶ 角度(rad)

角度(rad) ───────▶ radToDeg ──────────────▶ 角度(度)

TypedArray値 ────▶ normalize/denormalize ──▶ 正規化値

なし ────────────▶ generateUUID ───────────▶ UUID文字列
                  (Math.random使用)

Quaternion,
角度,order ──────▶ setQuaternionFromProperEuler ──▶ なし(q更新)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| MathUtils.js | `src/math/MathUtils.js` | ソース | MathUtilsの本体実装 |
| utils.js | `src/utils.js` | ソース | warn関数の提供元 |
| Quaternion.js | `src/math/Quaternion.js` | ソース | setQuaternionFromProperEulerの対象 |
