# 機能設計書 142-ColorManagement

## 概要

本ドキュメントは、Three.jsライブラリにおけるColorManagement（カラースペース管理）機能の設計について記述する。ColorManagementは、異なるカラースペース間での色変換を管理し、PBR（物理ベースレンダリング）における正確な色再現を実現する。

### 本機能の処理概要

ColorManagementは、3DグラフィックスにおけるカラースペースをLinear sRGBで統一的に管理し、入出力時に適切なカラースペース変換を行うことで、物理的に正確なレンダリングを実現するシステムである。

**業務上の目的・背景**：従来のWebGL開発では、テクスチャやマテリアルの色がsRGBで指定されているにもかかわらず、シェーダー内でリニア計算が必要な場面で変換が行われないことがあり、不正確な色再現となっていた。ColorManagementは、作業用カラースペースをLinear sRGBに統一し、入力時にsRGBからの変換、出力時にsRGBへの変換を自動的に行うことで、PBRワークフローにおける正確な色計算を保証する。これにより、HDR環境マップやライティング計算が物理的に正確になり、現実的なレンダリング結果が得られる。

**機能の利用シーン**：
- テクスチャ読み込み時のカラースペース変換
- マテリアル色指定時のリニア変換
- レンダリング出力時のsRGB変換
- 環境マップ処理でのカラースペース管理
- カスタムカラースペース定義（P3、ACEScc等）

**主要な処理内容**：
1. カラースペース間の色変換（sRGB ⇔ Linear sRGB）
2. 作業用カラースペースの管理
3. 転送関数（Transfer Function）の適用（SRGBTransfer、LinearTransfer）
4. 原色点（Primaries）と白色点（White Point）の管理
5. XYZ色空間を経由した色域変換
6. 輝度係数（Luminance Coefficients）の取得

**関連システム・外部連携**：
- WebGLRenderer/WebGPURendererの出力カラースペース設定
- Textureクラスのcolorspace属性
- Colorクラスの色変換メソッド
- Matrix3クラス（XYZ変換行列）

**権限による制御**：本機能は内部計算処理であり、権限による制御は存在しない。ColorManagement.enabledフラグで機能全体の有効/無効を切り替え可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUI画面に直接関連せず、内部のカラースペース変換処理として使用される |

## 機能種別

計算処理 / カラースペース変換

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | Color | Yes | 変換対象の色オブジェクト | Colorインスタンス |
| sourceColorSpace | string | Yes | 変換元カラースペース | 定義済みカラースペース |
| targetColorSpace | string | Yes | 変換先カラースペース | 定義済みカラースペース |

### 入力データソース

- Colorクラスからの色オブジェクト
- TextureLoaderからのテクスチャカラースペース情報

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| color | Color | カラースペース変換後の色オブジェクト |
| primaries | Array | カラースペースの原色点座標 |
| transfer | string | 転送関数の種類 |
| luminanceCoefficients | Vector3 | RGB輝度係数 |

### 出力先

- 変換されたColorオブジェクト（ミュータブル変換）
- レンダラーの描画バッファ設定

## 処理フロー

### 処理シーケンス

```
1. カラースペース変換リクエスト
   └─ convert(color, source, target)呼び出し
2. 有効性チェック
   └─ enabledフラグと同一カラースペースのチェック
3. 転送関数適用（ソース）
   └─ sRGBの場合、SRGBToLinear変換
4. 原色点変換（必要な場合）
   └─ XYZ空間を経由した色域マッピング
5. 転送関数適用（ターゲット）
   └─ sRGBの場合、LinearToSRGB変換
6. 結果返却
   └─ 変換されたcolorオブジェクトを返却
```

### フローチャート

```mermaid
flowchart TD
    A[convert呼び出し] --> B{enabled?}
    B -->|No| C[変換なしで返却]
    B -->|Yes| D{source == target?}
    D -->|Yes| C
    D -->|No| E{source.transfer == sRGB?}
    E -->|Yes| F[SRGBToLinear適用]
    E -->|No| G{原色点が異なる?}
    F --> G
    G -->|Yes| H[toXYZ変換]
    H --> I[fromXYZ変換]
    G -->|No| J{target.transfer == sRGB?}
    I --> J
    J -->|Yes| K[LinearToSRGB適用]
    J -->|No| L[結果返却]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-142-01 | 作業カラースペース | デフォルトはLinearSRGBColorSpace | 常時 |
| BR-142-02 | sRGB変換閾値 | 0.04045でリニア/ガンマ切替 | SRGBToLinear時 |
| BR-142-03 | Linear変換閾値 | 0.0031308でリニア/ガンマ切替 | LinearToSRGB時 |
| BR-142-04 | REC709輝度係数 | [0.2126, 0.7152, 0.0722] | 輝度計算時 |

### 計算ロジック

**SRGBToLinear変換**：
```javascript
(c < 0.04045) ? c * 0.0773993808 : Math.pow(c * 0.9478672986 + 0.0521327014, 2.4)
```

**LinearToSRGB変換**：
```javascript
(c < 0.0031308) ? c * 12.92 : 1.055 * Math.pow(c, 0.41666) - 0.055
```

**XYZ変換行列（REC709）**：
```
LINEAR_REC709_TO_XYZ:
| 0.4123908  0.3575843  0.1804808 |
| 0.2126390  0.7151687  0.0721923 |
| 0.0193308  0.1191948  0.9505322 |
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | Warning | 非推奨メソッド使用 | warnOnceでコンソール警告 |

### リトライ仕様

該当なし（同期的な計算処理のため）

## トランザクション仕様

該当なし（データベース操作を行わないため）

## パフォーマンス要件

- 色変換は定数時間 O(1) で完了
- 行列演算による色域変換も効率的
- ルックアップテーブルではなく数式による動的計算

## セキュリティ考慮事項

特になし（純粋な計算処理）

## 備考

- `fromWorkingColorSpace`と`toWorkingColorSpace`はr177で非推奨化
- 新しいメソッド名は`workingToColorSpace`と`colorSpaceToWorking`
- カスタムカラースペースはdefine()メソッドで追加可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 変換行列とカラースペース定義を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ColorManagement.js | `src/math/ColorManagement.js` | XYZ変換行列の定義（5-15行目） |
| 1-2 | ColorManagement.js | `src/math/ColorManagement.js` | REC709原色点・輝度係数・D65白色点（170-172行目） |

**読解のコツ**: XYZ色空間は人間の視覚に基づく装置非依存の色空間。REC709はsRGBの基礎となる規格。

#### Step 2: ColorManagementオブジェクトの構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ColorManagement.js | `src/math/ColorManagement.js` | createColorManagement関数とspaces定義（17-44行目） |
| 2-2 | ColorManagement.js | `src/math/ColorManagement.js` | LinearSRGBColorSpace定義（176-185行目） |
| 2-3 | ColorManagement.js | `src/math/ColorManagement.js` | SRGBColorSpace定義（187-195行目） |

**主要処理フロー**:
- **21行目**: enabled - カラーマネジメント有効/無効フラグ
- **23行目**: workingColorSpace - 作業用カラースペース（デフォルトLinear sRGB）
- **43行目**: spaces - カラースペース定義を格納するオブジェクト

#### Step 3: 変換処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | ColorManagement.js | `src/math/ColorManagement.js` | convert関数（45-78行目） |
| 3-2 | ColorManagement.js | `src/math/ColorManagement.js` | SRGBToLinear関数（205-209行目） |
| 3-3 | ColorManagement.js | `src/math/ColorManagement.js` | LinearToSRGB関数（211-215行目） |

**主要処理フロー**:
- **47-51行目**: 無効時または同一カラースペースは変換スキップ
- **53-59行目**: sRGB転送関数の場合、リニア変換を適用
- **61-66行目**: 原色点が異なる場合、XYZ経由で変換
- **68-74行目**: ターゲットがsRGBの場合、ガンマ変換を適用

### プログラム呼び出し階層図

```
ColorManagement (シングルトンオブジェクト)
    │
    ├─ プロパティ
    │      ├─ enabled
    │      ├─ workingColorSpace
    │      └─ spaces
    │
    ├─ 公開メソッド
    │      ├─ convert()
    │      │      ├─ SRGBToLinear()
    │      │      ├─ color.applyMatrix3()
    │      │      └─ LinearToSRGB()
    │      │
    │      ├─ workingToColorSpace()
    │      │      └─ convert()
    │      │
    │      ├─ colorSpaceToWorking()
    │      │      └─ convert()
    │      │
    │      ├─ getPrimaries()
    │      ├─ getTransfer()
    │      ├─ getToneMappingMode()
    │      ├─ getLuminanceCoefficients()
    │      └─ define()
    │
    └─ 内部メソッド
           ├─ _getMatrix()
           ├─ _getDrawingBufferColorSpace()
           └─ _getUnpackColorSpace()
```

### データフロー図

```
[入力]               [処理]                    [出力]

sRGB Color ─────▶ SRGBToLinear ─────────▶ Linear Color
                  (ガンマ展開)

Linear Color ───▶ LinearToSRGB ─────────▶ sRGB Color
                  (ガンマ圧縮)

Color +
sourceCS + ─────▶ convert ──────────────▶ Color
targetCS          (カラースペース変換)

colorSpace ─────▶ getPrimaries ─────────▶ primaries[]
                  (原色点取得)

colorSpace ─────▶ getTransfer ──────────▶ transfer
                  (転送関数取得)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ColorManagement.js | `src/math/ColorManagement.js` | ソース | カラーマネジメント本体実装 |
| Matrix3.js | `src/math/Matrix3.js` | ソース | XYZ変換行列クラス |
| constants.js | `src/constants.js` | ソース | カラースペース定数定義 |
| utils.js | `src/utils.js` | ソース | warnOnce関数の提供元 |
| Color.js | `src/math/Color.js` | ソース | 色オブジェクト（applyMatrix3使用） |
