# 機能設計書 145-LinearInterpolant

## 概要

本ドキュメントは、Three.jsライブラリにおけるLinearInterpolant（線形補間）機能の設計について記述する。LinearInterpolantは、Interpolant基底クラスを継承し、2つのサンプル値間を直線的に補間する最も基本的な補間器である。

### 本機能の処理概要

LinearInterpolantは、キーフレーム間の値を線形（直線的）に補間するクラスである。開始点と終了点を結ぶ直線上の任意の点を計算することで、滑らかなアニメーション遷移を実現する。

**業務上の目的・背景**：線形補間（Linear Interpolation、通称lerp）は、2点間の補間として最も直感的で計算効率が高い方式である。位置の移動、色の変化、透明度の遷移など、多くのアニメーションパラメータで使用される。3次補間ほど滑らかではないが、計算コストが低く、予測可能な動きを生成するため、多くのユースケースで適している。特に短い区間での補間や、高速なアニメーション更新が必要な場合に有効である。

**機能の利用シーン**：
- 位置アニメーション（VectorKeyframeTrack）
- 色のアニメーション（ColorKeyframeTrack）
- 数値プロパティのアニメーション（NumberKeyframeTrack）
- スケールアニメーション
- カスタムプロパティの遷移

**主要な処理内容**：
1. 基底クラスInterpolantから区間探索機能を継承
2. 2つのサンプル値間の重み付き平均を計算
3. 多次元データ（Vector3、Colorなど）の各成分を個別に補間
4. 結果をresultBufferに格納して返却

**関連システム・外部連携**：
- Interpolant基底クラス（区間探索機能を継承）
- KeyframeTrackクラス（補間器として使用）
- VectorKeyframeTrack、ColorKeyframeTrack、NumberKeyframeTrack

**権限による制御**：本機能は純粋な数学計算クラスであり、権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUI画面に直接関連せず、内部のアニメーション補間として使用される |

## 機能種別

計算処理 / 線形補間

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| parameterPositions | TypedArray | Yes | 補間因子（時間）の配列 | ソート済みを想定 |
| sampleValues | TypedArray | Yes | サンプル値の配列 | parameterPositionsと対応 |
| sampleSize | number | Yes | 1サンプルあたりの値の数 | 正の整数 |
| resultBuffer | TypedArray | No | 結果格納バッファ | 省略時は自動生成 |
| i1 | number | Yes | 現在の区間インデックス | 基底クラスから渡される |
| t0 | number | Yes | 区間開始時間 | 基底クラスから渡される |
| t | number | Yes | 評価時間 | 基底クラスから渡される |
| t1 | number | Yes | 区間終了時間 | 基底クラスから渡される |

### 入力データソース

- Interpolant.evaluate()からの呼び出し
- KeyframeTrackからのキーフレームデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | TypedArray | 線形補間された値 |

### 出力先

- resultBuffer（内部バッファ）
- AnimationMixerを経由してオブジェクトプロパティに適用

## 処理フロー

### 処理シーケンス

```
1. interpolate_(i1, t0, t, t1)呼び出し
   └─ 基底クラスevaluate()からの呼び出し
2. オフセット計算
   └─ offset1 = i1 * stride、offset0 = offset1 - stride
3. 重み計算
   └─ weight1 = (t - t0) / (t1 - t0)、weight0 = 1 - weight1
4. 各成分の補間
   └─ result[i] = values[offset0 + i] * weight0 + values[offset1 + i] * weight1
5. 結果返却
   └─ resultBufferを返却
```

### フローチャート

```mermaid
flowchart TD
    A[interpolate_呼び出し] --> B[オフセット計算]
    B --> C[重み計算]
    C --> D{各成分をループ}
    D -->|i < stride| E[線形補間計算]
    E --> F[result配列に格納]
    F --> D
    D -->|完了| G[resultBuffer返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-145-01 | 線形補間式 | result = v0 * (1-t) + v1 * t | 常時 |
| BR-145-02 | 重み範囲 | weight1は[0, 1]の範囲 | t0 <= t <= t1の場合 |
| BR-145-03 | 成分独立 | 各成分は独立に補間 | 多次元データの場合 |

### 計算ロジック

**線形補間**：
```javascript
weight1 = (t - t0) / (t1 - t0)
weight0 = 1 - weight1

for (i = 0; i < stride; i++) {
    result[i] = values[offset0 + i] * weight0 + values[offset1 + i] * weight1
}
```

**重み計算の性質**：
- t = t0 の場合: weight1 = 0、結果は前のサンプル値
- t = t1 の場合: weight1 = 1、結果は次のサンプル値
- 中間: 両サンプルの加重平均

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | - | 本クラスは特別なエラー処理を行わない |

### リトライ仕様

該当なし（同期的な計算処理のため）

## トランザクション仕様

該当なし（データベース操作を行わないため）

## パフォーマンス要件

- 補間計算はO(stride)の線形時間
- 単純な乗算と加算のみで構成
- ループ内に分岐なし

## セキュリティ考慮事項

特になし（純粋な数学計算クラス）

## 備考

- 最もシンプルで高速な補間方式
- 速度変化が不連続（加速度が無限大）になる点に注意
- 滑らかな動きが必要な場合はCubicInterpolantを検討

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: クラス構造と継承関係を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | LinearInterpolant.js | `src/math/interpolants/LinearInterpolant.js` | クラス定義とInterpolant継承（1-8行目） |
| 1-2 | LinearInterpolant.js | `src/math/interpolants/LinearInterpolant.js` | コンストラクタ（18-22行目） |

**読解のコツ**: LinearInterpolantはInterpolantを継承し、コンストラクタでsuper()を呼び出すのみ。

#### Step 2: interpolate_()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | LinearInterpolant.js | `src/math/interpolants/LinearInterpolant.js` | interpolate_メソッド（24-46行目） |

**主要処理フロー**:
- **26-28行目**: resultBuffer、sampleValues、strideの取得
- **30-31行目**: オフセット計算（offset1 = i1 * stride）
- **33-34行目**: 重み計算（weight1 = (t - t0) / (t1 - t0)）
- **36-41行目**: 各成分の線形補間ループ

### プログラム呼び出し階層図

```
Interpolant (基底クラス)
    │
    └─ LinearInterpolant
           │
           ├─ constructor()
           │      └─ super() 呼び出し
           │
           └─ interpolate_(i1, t0, t, t1)
                  │
                  ├─ オフセット計算
                  │      ├─ offset1 = i1 * stride
                  │      └─ offset0 = offset1 - stride
                  │
                  ├─ 重み計算
                  │      ├─ weight1 = (t - t0) / (t1 - t0)
                  │      └─ weight0 = 1 - weight1
                  │
                  └─ ループ（各成分）
                         └─ result[i] = v0 * w0 + v1 * w1
```

### データフロー図

```
[入力]               [処理]                    [出力]

i1, t0, t, t1 ─────▶ 重み計算 ─────────────────┐
                     weight1 = (t-t0)/(t1-t0)  │
                     weight0 = 1 - weight1     │
                                               │
sampleValues ──────▶ オフセット計算 ──────────┤
                     offset0, offset1          │
                                               ▼
                     各成分ループ ────────────▶ resultBuffer
                     v0*w0 + v1*w1             (補間結果)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| LinearInterpolant.js | `src/math/interpolants/LinearInterpolant.js` | ソース | 線形補間クラス本体 |
| Interpolant.js | `src/math/Interpolant.js` | ソース | 補間基底クラス |
| KeyframeTrack.js | `src/animation/KeyframeTrack.js` | ソース | 補間器を使用するトラック |
| VectorKeyframeTrack.js | `src/animation/tracks/VectorKeyframeTrack.js` | ソース | ベクトル値のトラック |
| ColorKeyframeTrack.js | `src/animation/tracks/ColorKeyframeTrack.js` | ソース | 色値のトラック |
| NumberKeyframeTrack.js | `src/animation/tracks/NumberKeyframeTrack.js` | ソース | 数値のトラック |
