# 機能設計書 148-DiscreteInterpolant

## 概要

本ドキュメントは、Three.jsライブラリにおけるDiscreteInterpolant（離散補間）機能の設計について記述する。DiscreteInterpolantは、Interpolant基底クラスを継承し、補間を行わずに現在位置の直前のサンプル値をそのまま返すステップ補間を提供する。

### 本機能の処理概要

DiscreteInterpolantは、キーフレーム間で補間を行わず、パラメータ位置の直前にあるキーフレームの値をそのまま使用する「ステップ」または「ホールド」補間器である。値が次のキーフレームに達するまで一定に保持される。

**業務上の目的・背景**：一部のアニメーションでは、滑らかな補間ではなく、特定の瞬間に値が離散的に変化する必要がある。例えば、スプライトアニメーションでのフレーム切り替え、状態遷移（表示/非表示）、モーフターゲットの切り替えなどでは、中間値を持たない離散的な変化が必要となる。DiscreteInterpolantはこのような「ホールド」または「ステップ」動作を実現し、値が瞬時に切り替わるアニメーションを可能にする。

**機能の利用シーン**：
- スプライトアニメーション（フレーム番号の切り替え）
- 可視性のオン/オフ切り替え
- テクスチャの切り替え
- 状態マシンの状態遷移
- 離散的なプロパティ値の変更

**主要な処理内容**：
1. 基底クラスInterpolantから区間探索機能を継承
2. 現在の区間の開始点（前のキーフレーム）の値をコピー
3. copySampleValue_()を使用してサンプル値をそのまま返却

**関連システム・外部連携**：
- Interpolant基底クラス（区間探索機能を継承）
- KeyframeTrackクラス（補間器として使用）
- BooleanKeyframeTrack、StringKeyframeTrack

**権限による制御**：本機能は純粋な数学計算クラスであり、権限による制御は存在しない。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はUI画面に直接関連せず、内部のアニメーション補間として使用される |

## 機能種別

計算処理 / 離散補間（ステップ補間）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| parameterPositions | TypedArray | Yes | 補間因子（時間）の配列 | ソート済みを想定 |
| sampleValues | TypedArray | Yes | サンプル値の配列 | parameterPositionsと対応 |
| sampleSize | number | Yes | 1サンプルあたりの値の数 | 正の整数 |
| resultBuffer | TypedArray | No | 結果格納バッファ | 省略時は自動生成 |
| i1 | number | Yes | 現在の区間インデックス | 基底クラスから渡される |

### 入力データソース

- Interpolant.evaluate()からの呼び出し
- KeyframeTrackからのキーフレームデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | TypedArray | 前のキーフレームの値（そのままコピー） |

### 出力先

- resultBuffer（内部バッファ）
- AnimationMixerを経由してオブジェクトプロパティに適用

## 処理フロー

### 処理シーケンス

```
1. interpolate_(i1)呼び出し
   └─ 基底クラスevaluate()からの呼び出し
2. 前のサンプルインデックス計算
   └─ i1 - 1 を使用
3. サンプル値のコピー
   └─ copySampleValue_(i1 - 1)を呼び出し
4. 結果返却
   └─ resultBufferを返却
```

### フローチャート

```mermaid
flowchart TD
    A[interpolate_呼び出し] --> B[前のインデックス i1-1 を計算]
    B --> C[copySampleValue_ 呼び出し]
    C --> D[サンプル値をresultBufferにコピー]
    D --> E[resultBuffer返却]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-148-01 | 前サンプル使用 | 常に前のキーフレーム（i1-1）の値を使用 | 常時 |
| BR-148-02 | 補間なし | 中間値を計算しない | 常時 |
| BR-148-03 | ステップ動作 | 値は次のキーフレームまで一定 | 常時 |

### 計算ロジック

**離散補間**：
```javascript
interpolate_(i1 /*, t0, t, t1 */) {
    return this.copySampleValue_(i1 - 1);
}
```

**copySampleValue_（基底クラス）**：
```javascript
copySampleValue_(index) {
    const result = this.resultBuffer,
          values = this.sampleValues,
          stride = this.valueSize,
          offset = index * stride;

    for (let i = 0; i !== stride; ++i) {
        result[i] = values[offset + i];
    }

    return result;
}
```

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | - | 本クラスは特別なエラー処理を行わない |

### リトライ仕様

該当なし（同期的な計算処理のため）

## トランザクション仕様

該当なし（データベース操作を行わないため）

## パフォーマンス要件

- 補間計算なしのためO(stride)のコピー時間のみ
- 最もシンプルで最速の補間器
- 分岐や数学計算を含まない

## セキュリティ考慮事項

特になし（純粋な数学計算クラス）

## 備考

- 「ステップ」「ホールド」「コンスタント」補間とも呼ばれる
- t0, t, t1パラメータは使用されない（インターフェース互換性のため存在）
- BooleanKeyframeTrackやStringKeyframeTrackで使用される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: クラス構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | DiscreteInterpolant.js | `src/math/interpolants/DiscreteInterpolant.js` | クラス定義とドキュメント（3-8行目） |
| 1-2 | DiscreteInterpolant.js | `src/math/interpolants/DiscreteInterpolant.js` | コンストラクタ（19-23行目） |

**読解のコツ**: DiscreteInterpolantは非常にシンプルで、interpolate_メソッドのみをオーバーライドする。

#### Step 2: interpolate_()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DiscreteInterpolant.js | `src/math/interpolants/DiscreteInterpolant.js` | interpolate_メソッド（25-29行目） |

**主要処理フロー**:
- **25行目**: t0, t, t1パラメータを無視（コメントアウト）
- **27行目**: copySampleValue_(i1 - 1)で前のサンプルをコピー

### プログラム呼び出し階層図

```
Interpolant (基底クラス)
    │
    └─ DiscreteInterpolant
           │
           ├─ constructor()
           │      └─ super() 呼び出し
           │
           └─ interpolate_(i1)
                  │
                  └─ copySampleValue_(i1 - 1)
                         │
                         └─ sampleValues[offset] → resultBuffer
```

### データフロー図

```
[入力]               [処理]                    [出力]

i1 ─────────────────▶ i1 - 1 計算
                            │
                            ▼
sampleValues ───────▶ copySampleValue_ ───────▶ resultBuffer
                     (前のサンプルをコピー)    (前のサンプル値)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| DiscreteInterpolant.js | `src/math/interpolants/DiscreteInterpolant.js` | ソース | 離散補間クラス本体 |
| Interpolant.js | `src/math/Interpolant.js` | ソース | 補間基底クラス（copySampleValue_提供） |
| KeyframeTrack.js | `src/animation/KeyframeTrack.js` | ソース | 補間器を使用するトラック |
| BooleanKeyframeTrack.js | `src/animation/tracks/BooleanKeyframeTrack.js` | ソース | 真偽値トラック（離散補間使用） |
| StringKeyframeTrack.js | `src/animation/tracks/StringKeyframeTrack.js` | ソース | 文字列トラック（離散補間使用） |
