# 機能設計書 154-GridHelper

## 概要

本ドキュメントは、Three.jsライブラリにおけるGridHelper機能の設計仕様を記載する。GridHelperは3D空間に2次元グリッドを表示するためのヘルパーオブジェクトである。

### 本機能の処理概要

GridHelperは、XZ平面上に等間隔の格子線を描画し、3D空間での位置関係や距離の把握を支援するデバッグ・開発支援機能である。中心線と外側の線で異なる色を使用でき、視覚的な参照グリッドとして機能する。

**業務上の目的・背景**：3Dエディタやビジュアライゼーションにおいて、オブジェクトの配置位置や距離感を把握することは重要である。GridHelperは、地面や作業平面を表現するグリッドを提供することで、空間認識を向上させ、精密な配置作業を支援する。

**機能の利用シーン**：
- 3Dエディタの作業平面表示
- オブジェクト配置時の位置ガイド
- スケール感の視覚的確認
- シーン構成時の基準面表示

**主要な処理内容**：
1. サイズと分割数に基づく格子線頂点の計算
2. 中心線と外側線の色分け
3. 頂点カラーによる色設定
4. LineSegmentsとしてのレンダリング

**関連システム・外部連携**：WebGLRenderer/WebGPURendererによるレンダリングパイプラインと連携。

**権限による制御**：本機能に権限制御はない。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 2 | Viewport | 補助機能 | 編集用グリッドの表示 |
| 14 | Menubar - View | 主機能 | グリッド表示のオン/オフ切替 |

## 機能種別

可視化処理 / デバッグ支援 / UI補助

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| size | number | No | グリッドの総サイズ（デフォルト: 10） | 正の数値 |
| divisions | number | No | グリッドの分割数（デフォルト: 10） | 正の整数 |
| color1 | number\|Color\|string | No | 中心線の色（デフォルト: 0x444444） | 有効なカラー値 |
| color2 | number\|Color\|string | No | 外側線の色（デフォルト: 0x888888） | 有効なカラー値 |

### 入力データソース

コンストラクタ引数として直接指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| GridHelper | LineSegments | グリッドを表す線分オブジェクト |

### 出力先

シーングラフへの追加（scene.add()）によりレンダリング対象となる。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ size, divisions, color1, color2パラメータの受け取り
2. Colorインスタンス生成
   └─ color1, color2をColorオブジェクトに変換
3. 格子線の計算
   └─ center = divisions / 2
   └─ step = size / divisions
   └─ halfSize = size / 2
4. 頂点と色の配列生成
   └─ divisions + 1本の水平線
   └─ divisions + 1本の垂直線
   └─ 中心線はcolor1、他はcolor2
5. BufferGeometryの構築
   └─ position属性とcolor属性を設定
6. LineBasicMaterialの生成
   └─ vertexColors: true, toneMapped: false
7. LineSegmentsとして初期化
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new GridHelper] --> B[パラメータ受け取り]
    B --> C[Colorインスタンス生成]
    C --> D[center, step, halfSize計算]
    D --> E[ループ開始: i = 0 to divisions]
    E --> F[水平線頂点追加]
    F --> G[垂直線頂点追加]
    G --> H{i == center?}
    H -->|Yes| I[color1を使用]
    H -->|No| J[color2を使用]
    I --> K[色配列に追加]
    J --> K
    K --> L{i <= divisions?}
    L -->|Yes| E
    L -->|No| M[BufferGeometry作成]
    M --> N[LineBasicMaterial作成]
    N --> O[LineSegments初期化]
    O --> P[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-154-01 | デフォルトサイズ | size未指定時は10を使用 | size未指定時 |
| BR-154-02 | デフォルト分割数 | divisions未指定時は10を使用 | divisions未指定時 |
| BR-154-03 | 中心線色 | 中央の線はcolor1を使用 | i == center時 |
| BR-154-04 | 外側線色 | 中央以外の線はcolor2を使用 | i != center時 |
| BR-154-05 | XZ平面配置 | グリッドはY=0のXZ平面に配置 | 常時 |

### 計算ロジック

グリッド生成パラメータ:
```javascript
center = divisions / 2       // 中心線のインデックス
step = size / divisions      // 線間の距離
halfSize = size / 2          // 中心からの距離
```

線分の座標:
```javascript
// 水平線（Z方向に平行）
vertices.push(-halfSize, 0, k, halfSize, 0, k)
// 垂直線（X方向に平行）
vertices.push(k, 0, -halfSize, k, 0, halfSize)
// k = -halfSize + i * step (i = 0 to divisions)
```

## データベース操作仕様

該当なし（本機能はデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 暗黙的エラー | divisionsが0 | ゼロ除算によるエラー |
| - | 暗黙的エラー | sizeが0 | グリッドが表示されない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 頂点数: (divisions + 1) × 4（各線2頂点 × 2方向）
- デフォルト(divisions=10)時: 44頂点
- ドローコール: 1
- 静的ジオメトリ（生成後は変更なし）

## セキュリティ考慮事項

該当なし（クライアントサイドの可視化機能）

## 備考

- dispose()メソッドを呼び出すことでGPUリソースを解放可能
- Y座標は常に0（XZ平面）
- 回転や移動で別の平面に配置可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

GridHelperは頂点カラーを使用したLineSegmentsである。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点データの管理方法 |
| 1-2 | BufferAttribute.js | `src/core/BufferAttribute.js` | Float32BufferAttributeによる属性データ |
| 1-3 | Color.js | `src/math/Color.js` | 色の表現とtoArray()メソッド |

**読解のコツ**: 頂点カラーを使用する場合、各頂点に対応するRGB値をcolor属性に設定する。toArray()でColorオブジェクトの値を配列に展開する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | GridHelper.js | `src/helpers/GridHelper.js` | 本体実装 |

**主要処理フロー**:
1. **31行目**: コンストラクタ開始、デフォルト値設定
2. **33-34行目**: color1, color2をColorインスタンスに変換
3. **36-38行目**: center, step, halfSize計算
4. **40行目**: vertices, colors配列初期化
5. **42-54行目**: ループでグリッド線を生成
6. **44行目**: 水平線（-halfSize, 0, k）→（halfSize, 0, k）
7. **45行目**: 垂直線（k, 0, -halfSize）→（k, 0, halfSize）
8. **47行目**: 中心線判定（i === center）
9. **49-52行目**: 各頂点に色を設定
10. **56-58行目**: BufferGeometryに属性設定
11. **60行目**: LineBasicMaterial生成（vertexColors有効）
12. **62行目**: LineSegments初期化

#### Step 3: disposeを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | GridHelper.js | `src/helpers/GridHelper.js` | 72-76行目のdispose |

**主要処理フロー**:
- **74行目**: geometry.dispose()
- **75行目**: material.dispose()

### プログラム呼び出し階層図

```
GridHelper (src/helpers/GridHelper.js)
    │
    ├─ extends LineSegments (src/objects/LineSegments.js)
    │      └─ extends Line (src/objects/Line.js)
    │             └─ extends Object3D (src/core/Object3D.js)
    │
    ├─ uses BufferGeometry (src/core/BufferGeometry.js)
    │      └─ setAttribute('position', ...)
    │      └─ setAttribute('color', ...)
    │
    ├─ uses LineBasicMaterial (src/materials/LineBasicMaterial.js)
    │      └─ vertexColors: true
    │
    └─ uses Color (src/math/Color.js)
           └─ toArray() → 色データを配列に展開
```

### データフロー図

```
[入力]                          [処理]                              [出力]

size, divisions        ───▶  GridHelper Constructor      ───▶  LineSegmentsオブジェクト
color1, color2                    │
                                  ├─▶ グリッドパラメータ計算
                                  │     └─▶ center, step, halfSize
                                  │
                                  ├─▶ ループ処理 (i = 0 to divisions)
                                  │     ├─▶ 水平線頂点追加
                                  │     ├─▶ 垂直線頂点追加
                                  │     └─▶ 色配列追加 (i == centerならcolor1)
                                  │
                                  └─▶ BufferGeometry構築
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| GridHelper.js | `src/helpers/GridHelper.js` | ソース | メイン実装 |
| LineSegments.js | `src/objects/LineSegments.js` | ソース | 親クラス（線分描画） |
| Line.js | `src/objects/Line.js` | ソース | 祖父クラス（線オブジェクト） |
| Object3D.js | `src/core/Object3D.js` | ソース | 基底クラス（3Dオブジェクト） |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | ジオメトリデータ管理 |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | 頂点属性データ |
| LineBasicMaterial.js | `src/materials/LineBasicMaterial.js` | ソース | 線描画マテリアル |
| Color.js | `src/math/Color.js` | ソース | 色管理ユーティリティ |
