# 機能設計書 155-PolarGridHelper

## 概要

本ドキュメントは、Three.jsライブラリにおけるPolarGridHelper機能の設計仕様を記載する。PolarGridHelperは3D空間に極座標系のグリッド（同心円と放射線）を表示するためのヘルパーオブジェクトである。

### 本機能の処理概要

PolarGridHelperは、中心から放射状に伸びるセクターラインと同心円状のリングラインを組み合わせた極座標グリッドを描画する。円形や回転対称のオブジェクト配置、角度ベースの作業に適したデバッグ・開発支援機能である。

**業務上の目的・背景**：回転体のモデリングや円形配置が必要なシーンでは、直交座標グリッドよりも極座標グリッドの方が作業効率が高い。PolarGridHelperは、角度や半径に基づく配置を視覚的にサポートし、円形パターンのレイアウト作業を支援する。

**機能の利用シーン**：
- 回転対称オブジェクトの配置
- 円形アレイ配置のガイド
- ターンテーブルやレコード盤のような円形UI
- 角度分割作業の視覚的補助

**主要な処理内容**：
1. セクター（放射線）の生成 - 中心から外周への直線
2. リング（同心円）の生成 - 各半径での円周
3. 交互カラーリングによる視認性向上
4. LineSegmentsとしてのレンダリング

**関連システム・外部連携**：WebGLRenderer/WebGPURendererによるレンダリングパイプラインと連携。

**権限による制御**：本機能に権限制御はない。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接的な関連なし。デバッグ・特殊用途で利用 |

## 機能種別

可視化処理 / デバッグ支援 / UI補助

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| radius | number | No | グリッドの半径（デフォルト: 10） | 正の数値 |
| sectors | number | No | セクター（放射線）の数（デフォルト: 16） | 正の整数 |
| rings | number | No | リング（同心円）の数（デフォルト: 8） | 正の整数 |
| divisions | number | No | 各リングの分割数（デフォルト: 64） | 正の整数 |
| color1 | number\|Color\|string | No | 第1色（デフォルト: 0x444444） | 有効なカラー値 |
| color2 | number\|Color\|string | No | 第2色（デフォルト: 0x888888） | 有効なカラー値 |

### 入力データソース

コンストラクタ引数として直接指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PolarGridHelper | LineSegments | 極座標グリッドを表す線分オブジェクト |

### 出力先

シーングラフへの追加（scene.add()）によりレンダリング対象となる。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ radius, sectors, rings, divisions, color1, color2パラメータの受け取り
2. Colorインスタンス生成
   └─ color1, color2をColorオブジェクトに変換
3. セクター（放射線）の生成
   └─ 中心(0,0,0)から外周(r*sin,0,r*cos)への線分
   └─ sectors本の放射線
   └─ インデックスに応じてcolor1/color2を交互に適用
4. リング（同心円）の生成
   └─ rings個の同心円
   └─ 各円はdivisions個の線分で構成
   └─ リングインデックスに応じてcolor1/color2を交互に適用
5. BufferGeometryの構築
   └─ position属性とcolor属性を設定
6. LineBasicMaterialの生成
   └─ vertexColors: true, toneMapped: false
7. LineSegmentsとして初期化
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new PolarGridHelper] --> B[パラメータ受け取り]
    B --> C[Colorインスタンス生成]
    C --> D{sectors > 1?}
    D -->|Yes| E[セクター生成ループ]
    D -->|No| F[リング生成へ]
    E --> G[角度計算: v = i/sectors * 2PI]
    G --> H[外周座標計算: sin*r, cos*r]
    H --> I[頂点追加: 0,0,0 → x,0,z]
    I --> J[色追加: i & 1 ? color1 : color2]
    J --> K{i < sectors?}
    K -->|Yes| E
    K -->|No| F
    F --> L[リング生成ループ]
    L --> M[半径計算: r - r/rings * i]
    M --> N[円周分割ループ]
    N --> O[2点の角度計算]
    O --> P[線分頂点追加]
    P --> Q[色追加: i & 1 ? color1 : color2]
    Q --> R{j < divisions?}
    R -->|Yes| N
    R -->|No| S{i < rings?}
    S -->|Yes| L
    S -->|No| T[BufferGeometry作成]
    T --> U[LineBasicMaterial作成]
    U --> V[LineSegments初期化]
    V --> W[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-155-01 | デフォルト半径 | radius未指定時は10を使用 | radius未指定時 |
| BR-155-02 | デフォルトセクター | sectors未指定時は16を使用 | sectors未指定時 |
| BR-155-03 | デフォルトリング | rings未指定時は8を使用 | rings未指定時 |
| BR-155-04 | デフォルト分割 | divisions未指定時は64を使用 | divisions未指定時 |
| BR-155-05 | セクター条件 | sectors > 1の場合のみ放射線を描画 | sectors判定時 |
| BR-155-06 | 交互カラー | ビット演算(i & 1)で色を交互に切替 | 常時 |
| BR-155-07 | XZ平面配置 | グリッドはY=0のXZ平面に配置 | 常時 |

### 計算ロジック

セクター角度計算:
```javascript
v = (i / sectors) * (Math.PI * 2)  // 各セクターの角度
x = Math.sin(v) * radius           // X座標
z = Math.cos(v) * radius           // Z座標
```

リング半径計算:
```javascript
r = radius - (radius / rings * i)  // 外側から内側へ
```

リング円周の線分:
```javascript
v1 = (j / divisions) * (Math.PI * 2)      // 開始角度
v2 = ((j + 1) / divisions) * (Math.PI * 2) // 終了角度
// 各線分: (sin(v1)*r, 0, cos(v1)*r) → (sin(v2)*r, 0, cos(v2)*r)
```

## データベース操作仕様

該当なし（本機能はデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 暗黙的処理 | sectors <= 1 | 放射線が描画されない |
| - | 暗黙的エラー | rings が 0 | リングが描画されない |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- セクター頂点数: sectors × 2（sectors > 1の場合）
- リング頂点数: rings × divisions × 2
- デフォルト時: 16×2 + 8×64×2 = 1056頂点
- ドローコール: 1
- 静的ジオメトリ（生成後は変更なし）

## セキュリティ考慮事項

該当なし（クライアントサイドの可視化機能）

## 備考

- dispose()メソッドを呼び出すことでGPUリソースを解放可能
- Y座標は常に0（XZ平面）
- GridHelperとの違い: 極座標系（円形）vs直交座標系（格子）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

PolarGridHelperはセクターとリングの2種類の線分で構成される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点データの管理方法 |
| 1-2 | BufferAttribute.js | `src/core/BufferAttribute.js` | Float32BufferAttributeによる属性データ |
| 1-3 | Color.js | `src/math/Color.js` | 色の表現とr, g, bプロパティ |

**読解のコツ**: 三角関数を使用した極座標計算を理解すること。sin/cosの値域と角度の関係を把握する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PolarGridHelper.js | `src/helpers/PolarGridHelper.js` | 本体実装 |

**主要処理フロー**:
1. **35行目**: コンストラクタ開始、デフォルト値設定
2. **37-38行目**: color1, color2をColorインスタンスに変換
3. **40-41行目**: vertices, colors配列初期化
4. **45-63行目**: セクター（放射線）生成ループ
   - **49行目**: 角度計算 `v = (i / sectors) * (Math.PI * 2)`
   - **51-52行目**: 外周座標 `x = sin(v)*r, z = cos(v)*r`
   - **54-55行目**: 頂点追加（原点→外周）
   - **57行目**: ビット演算で色選択 `(i & 1) ? color1 : color2`
5. **68-97行目**: リング（同心円）生成ループ
   - **70行目**: 色選択（リングインデックス基準）
   - **72行目**: リング半径計算
   - **74-95行目**: 円周分割ループ
6. **100-102行目**: BufferGeometryに属性設定
7. **104行目**: LineBasicMaterial生成
8. **106行目**: LineSegments初期化

#### Step 3: disposeを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PolarGridHelper.js | `src/helpers/PolarGridHelper.js` | 116-120行目のdispose |

**主要処理フロー**:
- **118行目**: geometry.dispose()
- **119行目**: material.dispose()

### プログラム呼び出し階層図

```
PolarGridHelper (src/helpers/PolarGridHelper.js)
    │
    ├─ extends LineSegments (src/objects/LineSegments.js)
    │      └─ extends Line (src/objects/Line.js)
    │             └─ extends Object3D (src/core/Object3D.js)
    │
    ├─ uses BufferGeometry (src/core/BufferGeometry.js)
    │      └─ setAttribute('position', ...)
    │      └─ setAttribute('color', ...)
    │
    ├─ uses LineBasicMaterial (src/materials/LineBasicMaterial.js)
    │      └─ vertexColors: true
    │
    └─ uses Color (src/math/Color.js)
           └─ r, g, b プロパティ参照
```

### データフロー図

```
[入力]                              [処理]                              [出力]

radius, sectors,          ───▶  PolarGridHelper Constructor  ───▶  LineSegmentsオブジェクト
rings, divisions,                     │
color1, color2                        │
                                      ├─▶ セクター生成 (sectors > 1の場合)
                                      │     └─▶ 原点から外周への放射線
                                      │
                                      └─▶ リング生成
                                            └─▶ 同心円（外側から内側へ）
                                                  └─▶ 各円をdivisions分割
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PolarGridHelper.js | `src/helpers/PolarGridHelper.js` | ソース | メイン実装 |
| LineSegments.js | `src/objects/LineSegments.js` | ソース | 親クラス（線分描画） |
| Line.js | `src/objects/Line.js` | ソース | 祖父クラス（線オブジェクト） |
| Object3D.js | `src/core/Object3D.js` | ソース | 基底クラス（3Dオブジェクト） |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | ジオメトリデータ管理 |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | 頂点属性データ |
| LineBasicMaterial.js | `src/materials/LineBasicMaterial.js` | ソース | 線描画マテリアル |
| Color.js | `src/math/Color.js` | ソース | 色管理ユーティリティ |
