# 機能設計書 160-PointLightHelper

## 概要

本ドキュメントは、Three.jsライブラリにおけるPointLightHelper機能の設計仕様を記載する。PointLightHelperは点光源（PointLight）の位置を球体メッシュで可視化するためのヘルパーオブジェクトである。

### 本機能の処理概要

PointLightHelperは、点光源の位置をワイヤーフレームの球体で表現する。光源の色に追従して球体の色も更新され、点光源の位置を直感的に把握できるデバッグ・開発支援機能である。

**業務上の目的・背景**：点光源は全方向に光を放射するため、その位置を把握することが重要である。PointLightHelperは、シンプルな球体で光源位置を可視化することで、ライティング設定のデバッグを支援する。他のライトヘルパーと異なり、Meshを継承しているため、ワイヤーフレーム球体として描画される。

**機能の利用シーン**：
- 点光源（電球等）の配置確認
- 室内照明のデバッグ
- 複数点光源の位置識別
- ライティングデバッグ

**主要な処理内容**：
1. SphereGeometryによる球体生成
2. MeshBasicMaterialのワイヤーフレームモード
3. 光源のmatrixWorldを共有して位置同期
4. update()による色の動的更新

**関連システム・外部連携**：PointLightと連携。WebGLRenderer/WebGPURendererによるレンダリングパイプラインと連携。

**権限による制御**：本機能に権限制御はない。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接的な関連なし。デバッグ用途で利用 |

## 機能種別

可視化処理 / デバッグ支援

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| light | PointLight | Yes | 可視化する点光源 | PointLightインスタンス |
| sphereSize | number | No | 球体のサイズ（デフォルト: 1） | 正の数値 |
| color | number\|Color\|string | No | ヘルパーの色（未指定時はライトの色を使用） | 有効なカラー値 |

### 入力データソース

コンストラクタ引数として直接指定。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| PointLightHelper | Mesh | 球体で点光源を表すメッシュオブジェクト |

### 出力先

シーングラフへの追加（scene.add()）によりレンダリング対象となる。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ light, sphereSize, colorパラメータの受け取り
2. SphereGeometryの生成
   └─ 半径: sphereSize
   └─ セグメント: 4（水平）, 2（垂直）
3. MeshBasicMaterialの生成
   └─ wireframe: true
   └─ fog: false
   └─ toneMapped: false
4. Meshとして初期化
5. lightプロパティ設定
6. colorプロパティ設定
7. 光源のmatrixWorldを共有
   └─ matrixAutoUpdate = false
8. update()呼び出し
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new PointLightHelper] --> B[light, sphereSize, color引数受け取り]
    B --> C[SphereGeometry作成]
    C --> D[MeshBasicMaterial作成]
    D --> E[Mesh初期化]
    E --> F[lightプロパティ設定]
    F --> G[colorプロパティ設定]
    G --> H[matrixWorld共有]
    H --> I[matrixAutoUpdate = false]
    I --> J[update呼び出し]
    J --> K[終了]

    L[update呼び出し] --> M[light.updateWorldMatrix]
    M --> N{color指定あり?}
    N -->|Yes| O[指定色を設定]
    N -->|No| P[lightの色をコピー]
    O --> Q[終了]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-160-01 | デフォルトサイズ | sphereSize未指定時は1を使用（暗黙的） | sphereSize未指定時 |
| BR-160-02 | 自動カラー | color未指定時はlightの色を使用 | color未指定時 |
| BR-160-03 | ワイヤーフレーム | 球体は常にワイヤーフレームで描画 | 常時 |
| BR-160-04 | 簡易ジオメトリ | セグメント数は最小限（4×2） | 常時 |
| BR-160-05 | フォグ無効 | fog: falseでフォグの影響を受けない | 常時 |

### 計算ロジック

球体ジオメトリの生成:
```javascript
new SphereGeometry(sphereSize, 4, 2)
// sphereSize: 半径
// 4: 水平セグメント数（経度方向）
// 2: 垂直セグメント数（緯度方向）
```

位置同期:
```javascript
this.matrix = this.light.matrixWorld  // matrixWorldを共有
this.matrixAutoUpdate = false          // 自動更新無効
```

## データベース操作仕様

該当なし（本機能はデータベースを使用しない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 暗黙的エラー | lightがnull/undefined | エラー発生 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 頂点数: 最小限（4×2セグメントの球体）
- ドローコール: 1
- Meshを継承（LineSegmentsではない）
- update()は光源の色変更時に手動呼び出し

## セキュリティ考慮事項

該当なし（クライアントサイドの可視化機能）

## 備考

- dispose()メソッドを呼び出すことでGPUリソースを解放可能
- Meshを継承（他のヘルパーとは異なる）
- ワイヤーフレームモードで描画されるため、塗りつぶしではない
- コメントアウトされたコードに距離表示機能の残骸あり（将来的な機能拡張の可能性）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

PointLightHelperはMeshを継承し、SphereGeometryとMeshBasicMaterialで構成される。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PointLight.js | `src/lights/PointLight.js` | distance, color属性 |
| 1-2 | Mesh.js | `src/objects/Mesh.js` | Meshの基本構造 |
| 1-3 | SphereGeometry.js | `src/geometries/SphereGeometry.js` | 球体ジオメトリ生成 |
| 1-4 | MeshBasicMaterial.js | `src/materials/MeshBasicMaterial.js` | wireframeオプション |

**読解のコツ**: PointLightHelperは他のヘルパーと異なり、Meshを継承している。ワイヤーフレームモードで球体を描画するため、ライト周囲に球形のガイドが表示される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PointLightHelper.js | `src/helpers/PointLightHelper.js` | 本体実装 |

**主要処理フロー**:
1. **31行目**: コンストラクタ開始
2. **33行目**: SphereGeometry生成（半径sphereSize、4×2セグメント）
3. **34行目**: MeshBasicMaterial生成（wireframe: true）
4. **36行目**: super()でMesh初期化
5. **43行目**: lightプロパティ設定
6. **51行目**: colorプロパティ設定
7. **53行目**: type = 'PointLightHelper'
8. **55行目**: matrixWorld共有
9. **56行目**: matrixAutoUpdate = false
10. **58行目**: update()呼び出し

#### Step 3: updateを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PointLightHelper.js | `src/helpers/PointLightHelper.js` | 101-113行目のupdate |

**主要処理フロー**:
- **103行目**: light.updateWorldMatrix()
- **105-111行目**: 色設定
  - color指定あり: material.color.set(color)
  - color指定なし: material.color.copy(light.color)

#### Step 4: disposeを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | PointLightHelper.js | `src/helpers/PointLightHelper.js` | 90-94行目のdispose |

**主要処理フロー**:
- **92行目**: this.geometry.dispose()
- **93行目**: this.material.dispose()

#### 補足: コメントアウトコードについて

コンストラクタとupdateメソッドにコメントアウトされたコードがある（61-82行目、115-128行目）。これは距離に応じた球体表示機能の実装途中のコードと思われる。

### プログラム呼び出し階層図

```
PointLightHelper (src/helpers/PointLightHelper.js)
    │
    ├─ extends Mesh (src/objects/Mesh.js)
    │      └─ extends Object3D (src/core/Object3D.js)
    │
    ├─ uses SphereGeometry (src/geometries/SphereGeometry.js)
    │      └─ 半径sphereSize、4×2セグメント
    │
    ├─ uses MeshBasicMaterial (src/materials/MeshBasicMaterial.js)
    │      └─ wireframe: true
    │      └─ fog: false
    │      └─ toneMapped: false
    │
    └─ references PointLight (src/lights/PointLight.js)
           ├─ matrixWorld
           └─ color
```

### データフロー図

```
[入力]                          [処理]                              [出力]

PointLight            ───▶  PointLightHelper Constructor  ───▶  Meshオブジェクト
sphereSize                       │
color                            │
                                 ├─▶ SphereGeometry生成
                                 │        └─▶ 簡易球体（4×2セグメント）
                                 │
                                 └─▶ MeshBasicMaterial生成
                                          └─▶ wireframe: true
    │
    └── [ライト変更時] ──▶  update()
                                  │
                                  └─▶ 色更新（指定色 or ライト色）
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PointLightHelper.js | `src/helpers/PointLightHelper.js` | ソース | メイン実装 |
| PointLight.js | `src/lights/PointLight.js` | ソース | 対象光源クラス |
| Mesh.js | `src/objects/Mesh.js` | ソース | 親クラス（メッシュオブジェクト） |
| Object3D.js | `src/core/Object3D.js` | ソース | 基底クラス（3Dオブジェクト） |
| SphereGeometry.js | `src/geometries/SphereGeometry.js` | ソース | 球体ジオメトリ |
| MeshBasicMaterial.js | `src/materials/MeshBasicMaterial.js` | ソース | ワイヤーフレーム用マテリアル |
