# 機能設計書 171-ImageUtils

## 概要

本ドキュメントは、Three.jsの画像処理ユーティリティクラス「ImageUtils」の機能設計について記述する。ImageUtilsは画像データの変換・処理を行うための静的メソッドを提供するユーティリティクラスである。

### 本機能の処理概要

ImageUtilsクラスは、3Dグラフィックス開発において必要となる画像処理機能を集約したユーティリティクラスである。主に画像からのData URI生成と、sRGB色空間からリニア色空間への変換機能を提供する。

**業務上の目的・背景**：3Dグラフィックスにおいて、画像データは様々な形式で扱われる。テクスチャとして使用する画像をシリアライズしたり、色空間を適切に変換したりする必要がある。ImageUtilsはこれらの一般的な画像処理タスクを統一されたインターフェースで提供し、開発者が個別に実装する手間を省く。特に、物理ベースレンダリング（PBR）では色空間の正確な管理が重要であり、sRGBからリニア色空間への変換は頻繁に必要となる。

**機能の利用シーン**：
- シーンデータのエクスポート時に画像をData URIとして埋め込む場合
- 外部ツールとの連携のために画像データを文字列として保存する場合
- PBRマテリアルで使用する画像の色空間を補正する場合
- カスタムシェーダーで正確な色計算を行うために入力画像をリニア化する場合

**主要な処理内容**：
1. getDataURL: HTMLImageElement、HTMLCanvasElement、またはImageDataからData URI文字列を生成
2. sRGBToLinear: sRGB色空間の画像データをリニア色空間に変換

**関連システム・外部連携**：
- ColorManagementモジュールのSRGBToLinear関数を使用して色空間変換を実行
- Web標準のCanvas APIを使用して画像データの読み取りと書き込みを行う

**権限による制御**：特になし（クライアントサイドで実行される汎用ユーティリティ）

## 関連画面

画面機能マッピングに直接的な関連画面の登録なし。本機能は内部ユーティリティとして他の機能から呼び出される。

## 機能種別

データ変換処理 / ユーティリティ

## 入力仕様

### 入力パラメータ

#### getDataURL メソッド

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| image | HTMLImageElement \| HTMLCanvasElement \| ImageData | Yes | 変換対象の画像オブジェクト | - |
| type | string | No | 出力する画像フォーマット（デフォルト: 'image/png'） | MIMEタイプ形式 |

#### sRGBToLinear メソッド

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| image | HTMLImageElement \| HTMLCanvasElement \| ImageBitmap \| Object | Yes | 変換対象の画像オブジェクト | dataプロパティを持つオブジェクトも可 |

### 入力データソース

- HTMLImageElement: img要素から取得した画像
- HTMLCanvasElement: Canvas要素に描画された画像
- ImageData: Canvas 2Dコンテキストから取得したピクセルデータ
- ImageBitmap: createImageBitmap()で生成されたビットマップ
- カスタムオブジェクト: data、width、heightプロパティを持つオブジェクト

## 出力仕様

### 出力データ

#### getDataURL メソッド

| 項目名 | 型 | 説明 |
|--------|-----|------|
| dataURI | string | Base64エンコードされたData URI文字列 |

#### sRGBToLinear メソッド

| 項目名 | 型 | 説明 |
|--------|-----|------|
| result | HTMLCanvasElement \| Object | リニア色空間に変換された画像データ |

### 出力先

- メモリ上のオブジェクト（戻り値として返却）

## 処理フロー

### 処理シーケンス

#### getDataURL

```
1. 入力画像のsrcがData URIか確認
   └─ Data URIの場合はそのまま返却
2. HTMLCanvasElementが利用可能か確認
   └─ 利用不可の場合はimage.srcを返却
3. 入力がHTMLCanvasElementか判定
   └─ HTMLCanvasElementの場合はそのまま使用
4. キャンバスにコンテンツを転写
   └─ ImageDataの場合はputImageData()を使用
   └─ その他の場合はdrawImage()を使用
5. canvas.toDataURL()でData URI文字列を生成して返却
```

#### sRGBToLinear

```
1. 入力画像の型を判定
   └─ HTMLImageElement/HTMLCanvasElement/ImageBitmapの場合
      1-1. 新規Canvasを作成
      1-2. 画像をCanvasに描画
      1-3. getImageData()でピクセルデータを取得
      1-4. 各ピクセル値にSRGBToLinear変換を適用
      1-5. putImageData()で変換後データを書き込み
      1-6. Canvasを返却
   └─ dataプロパティを持つオブジェクトの場合
      2-1. dataをコピー
      2-2. 各値にSRGBToLinear変換を適用
      2-3. 変換後のdata、width、heightを持つオブジェクトを返却
   └─ サポートされない型の場合
      3-1. 警告を出力
      3-2. 入力をそのまま返却
```

### フローチャート

```mermaid
flowchart TD
    A[開始: getDataURL] --> B{image.srcがData URI?}
    B -->|Yes| C[image.srcを返却]
    B -->|No| D{HTMLCanvasElement利用可能?}
    D -->|No| E[image.srcを返却]
    D -->|Yes| F{imageはHTMLCanvasElement?}
    F -->|Yes| G[imageをそのまま使用]
    F -->|No| H[内部キャンバスを準備]
    H --> I{imageはImageData?}
    I -->|Yes| J[putImageData実行]
    I -->|No| K[drawImage実行]
    J --> L[toDataURLで変換]
    K --> L
    G --> L
    L --> M[Data URI文字列を返却]
    M --> N[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-171-01 | Data URI透過 | 入力画像が既にData URIの場合は変換処理をスキップ | getDataURL実行時 |
| BR-171-02 | 環境依存処理 | HTMLCanvasElementが利用不可能な環境ではsrcをそのまま返却 | Node.js等のサーバー環境 |
| BR-171-03 | 色空間変換精度 | Uint8Array/Uint8ClampedArrayの場合は整数値に丸める | sRGBToLinear実行時 |
| BR-171-04 | Float配列処理 | Float配列の場合は0-1の範囲で処理 | sRGBToLinear実行時 |

### 計算ロジック

**sRGBToLinear変換式**：
ColorManagement.jsで定義されたSRGBToLinear関数を使用。標準的なsRGB→リニア変換式を適用：
- 入力値 <= 0.04045 の場合: 出力 = 入力 / 12.92
- 入力値 > 0.04045 の場合: 出力 = ((入力 + 0.055) / 1.055) ^ 2.4

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 警告 | sRGBToLinearでサポートされない画像タイプが渡された場合 | コンソールに警告を出力し、入力をそのまま返却 |

### リトライ仕様

リトライ処理は実装されていない（同期処理のため）

## トランザクション仕様

トランザクション処理は適用されない（メモリ上の処理のみ）

## パフォーマンス要件

- getDataURL: 画像サイズに比例した処理時間（Canvas APIの性能に依存）
- sRGBToLinear: 全ピクセルをループ処理するため、大きな画像では処理時間が増加

## セキュリティ考慮事項

- クロスオリジン画像に対してgetDataURLを実行する場合、CORSポリシーにより制限される可能性がある
- Data URIはBase64エンコードのため、元のバイナリサイズより約33%大きくなる

## 備考

- 内部キャンバス（_canvas）は再利用されるため、連続した処理でもメモリ効率が良い
- sRGBToLinear変換後の画像はPBRシェーダーでの使用に適している

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

本機能は静的メソッドのみで構成されており、特別なデータ構造は定義されていない。入力・出力の型を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ImageUtils.js | `src/extras/ImageUtils.js` | 6-11行目でクラス定義とJSDocコメントを確認 |

**読解のコツ**: JavaScriptの静的クラスメソッド（static）の構文を理解していると読みやすい。各メソッドはインスタンス化せずに直接呼び出せる。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ImageUtils.js | `src/extras/ImageUtils.js` | getDataURLメソッド（20-65行目）とsRGBToLinearメソッド（73-133行目）がエントリーポイント |

**主要処理フロー**:
1. **20-26行目**: getDataURLでData URIチェック - 正規表現で既存Data URIを検出
2. **28-32行目**: 環境チェック - HTMLCanvasElementの存在確認
3. **36-61行目**: Canvas描画処理 - ImageDataとそれ以外で処理分岐
4. **73-77行目**: sRGBToLinearの型判定 - instanceof演算子で画像タイプを識別
5. **90-94行目**: ピクセル単位の色空間変換ループ

#### Step 3: 依存関係を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | utils.js | `src/utils.js` | createElementNS関数 - 名前空間を考慮したDOM要素作成 |
| 3-2 | ColorManagement.js | `src/math/ColorManagement.js` | SRGBToLinear関数 - 色空間変換の数学的実装 |

**主要処理フロー**:
- **ColorManagement.js**: SRGBToLinear関数は標準的なガンマ補正の逆変換を実装

### プログラム呼び出し階層図

```
ImageUtils.getDataURL(image, type)
    │
    └─ createElementNS('canvas')  [utils.js]
           └─ document.createElementNS()

ImageUtils.sRGBToLinear(image)
    │
    ├─ createElementNS('canvas')  [utils.js]
    │
    ├─ SRGBToLinear()  [ColorManagement.js]
    │
    └─ warn()  [utils.js]
```

### データフロー図

```
[入力]                    [処理]                         [出力]

HTMLImageElement ──┐
HTMLCanvasElement ─┼──▶ getDataURL() ──────────────▶ Data URI文字列
ImageData ─────────┘         │
                             └─ Canvas 2D API
                                (drawImage/putImageData/toDataURL)


HTMLImageElement ──┐
HTMLCanvasElement ─┤
ImageBitmap ───────┼──▶ sRGBToLinear() ────────────▶ HTMLCanvasElement
Object{data} ──────┘         │                         または
                             ├─ SRGBToLinear()       Object{data,width,height}
                             └─ Canvas 2D API
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ImageUtils.js | `src/extras/ImageUtils.js` | ソース | 画像処理ユーティリティの本体 |
| ColorManagement.js | `src/math/ColorManagement.js` | ソース | 色空間変換関数の提供 |
| utils.js | `src/utils.js` | ソース | 汎用ユーティリティ関数（createElementNS, warn） |
