# 機能設計書 174-TSL

## 概要

本ドキュメントは、Three.jsのノードベースシェーダー言語「TSL（Three.js Shading Language）」の機能設計について記述する。TSLはJavaScriptベースのDSL（Domain Specific Language）として、シェーダープログラムを直感的に記述するためのノードシステムを提供する。

### 本機能の処理概要

TSLは、Three.jsのノードベースマテリアルシステムの中核を成すシェーダー言語である。従来のGLSL/WGSLを直接記述する代わりに、JavaScriptの関数呼び出しチェーンでシェーダーロジックを構築できる。これにより、WebGLとWebGPUの両方のバックエンドに対応した抽象化されたシェーダー記述が可能となる。

**業務上の目的・背景**：従来のシェーダー開発では、GLSLやWGSLの専門知識が必要であり、WebGLとWebGPUで異なるコードを書く必要があった。TSLはJavaScript開発者にとって親しみやすい構文で、両バックエンドに自動的に対応するシェーダーを記述できるようにする。また、ノードベースの構造により、シェーダーの部品を再利用しやすくなっている。

**機能の利用シーン**：
- カスタムマテリアルのシェーダーを定義する場合
- ポストプロセッシングエフェクトを実装する場合
- GPU計算（GPGPU）を実装する場合
- 物理ベースレンダリングの拡張を行う場合
- 手続き型テクスチャやプロシージャルジオメトリを生成する場合

**主要な処理内容**：
1. ノード定義エクスポート: 数学、ユーティリティ、アクセサなど各種ノードタイプを提供
2. TSLベース機能: 変数宣言、条件分岐、ループなどの制御構造
3. BSDF関数: 物理ベースのシェーディング関数（GGX、Lambert、Schlick等）
4. ライティング/シャドウ: ライトとシャドウのノード処理
5. GPGPU: コンピュートシェーダーと並列処理

**関連システム・外部連携**：
- NodeMaterial: TSLで定義されたノードを使用するマテリアル
- WebGLRenderer/WebGPURenderer: TSLノードをGLSL/WGSLにコンパイル
- NodeBuilder: ノードグラフからシェーダーコードを生成

**権限による制御**：特になし（クライアントサイドで実行されるシェーダー言語）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 24 | WebGPUサンプル | 補助機能 | ノードベースシェーダーの記述 |

## 機能種別

シェーダー言語 / DSL / ノードシステム

## 入力仕様

### 入力パラメータ

TSLはモジュールエクスポートの集合体であり、直接的な入力パラメータはない。各ノード関数は独自のパラメータを受け取る。

#### 主要なノード関数の例

| 関数名 | 型 | 説明 |
|--------|-----|------|
| float(value) | FloatNode | 浮動小数点数ノードを作成 |
| vec2(x, y) | Vec2Node | 2Dベクトルノードを作成 |
| vec3(x, y, z) | Vec3Node | 3Dベクトルノードを作成 |
| vec4(x, y, z, w) | Vec4Node | 4Dベクトルノードを作成 |
| texture(tex, uv) | TextureNode | テクスチャサンプリングノードを作成 |
| uniform(type, value) | UniformNode | ユニフォーム変数ノードを作成 |

### 入力データソース

- JavaScriptコード: TSL関数を使用したシェーダー定義
- Textureオブジェクト: テクスチャノードの入力
- 数値/ベクトル: 定数値やユニフォーム値

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| Node | Node | シェーダーグラフのノードオブジェクト |
| shaderCode | string | コンパイル後のGLSL/WGSLコード |

### 出力先

- NodeMaterial: マテリアルのシェーダーとして使用
- WebGLRenderer/WebGPURenderer: コンパイルされたシェーダープログラム

## 処理フロー

### 処理シーケンス

#### TSLシェーダー定義から実行まで

```
1. TSL関数を使用してノードグラフを構築
   └─ ノード関数の連鎖呼び出しでシェーダーロジックを定義
2. NodeMaterialにノードを設定
   └─ fragmentNode, vertexNode等のプロパティに代入
3. レンダラーがマテリアルを処理
   └─ NodeBuilderがノードグラフをトラバース
4. ターゲットバックエンドに応じたシェーダーコードを生成
   └─ WebGL: GLSLを生成
   └─ WebGPU: WGSLを生成
5. シェーダープログラムをコンパイルして実行
```

### フローチャート

```mermaid
flowchart TD
    A[TSL関数呼び出し] --> B[ノードグラフ構築]
    B --> C[NodeMaterialに設定]
    C --> D{バックエンド判定}
    D -->|WebGL| E[GLSLコード生成]
    D -->|WebGPU| F[WGSLコード生成]
    E --> G[シェーダーコンパイル]
    F --> G
    G --> H[GPU実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-174-01 | バックエンド抽象化 | 同一のTSLコードがWebGLとWebGPUの両方で動作 | シェーダーコンパイル時 |
| BR-174-02 | 型推論 | ノード間の接続時に自動的に型変換 | ノードグラフ構築時 |
| BR-174-03 | 遅延評価 | ノードの評価は実際のレンダリング時に行われる | レンダーパイプライン実行時 |
| BR-174-04 | モジュール再エクスポート | TSL.jsは各種ノードモジュールを集約してエクスポート | インポート時 |

### 計算ロジック

TSLは各種数学関数や物理ベースシェーディング関数を提供：

**BRDF関数（例: GGX分布）**：
```
D_GGX = alpha^2 / (PI * (NdotH^2 * (alpha^2 - 1) + 1)^2)
```

**フレネル関数（Schlick近似）**：
```
F = F0 + (1 - F0) * (1 - VdotH)^5
```

## データベース操作仕様

本機能はデータベース操作を行わない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| TSLエラー | コンパイルエラー | ノードグラフに不正な接続がある場合 | NodeBuilderがフォールバックマテリアルを使用 |
| 型エラー | 実行時エラー | 互換性のない型のノードを接続した場合 | コンソールにエラーメッセージを出力 |

### リトライ仕様

シェーダーコンパイルエラー時はフォールバックマテリアル（NodeMaterial）が使用される

## トランザクション仕様

トランザクション処理は適用されない（GPUシェーダー処理）

## パフォーマンス要件

- ノードグラフ構築: JavaScript実行時間に依存
- シェーダーコンパイル: 初回レンダリング時に発生（キャッシュされる）
- シェーダー実行: GPU処理能力に依存

## セキュリティ考慮事項

- シェーダーコードはクライアント側で生成・実行
- 悪意のあるコードがGPU上で実行される可能性（ブラウザのサンドボックスで保護）

## 備考

- TSL.jsは主にre-exportファイルとして機能し、実際のノード実装は各サブモジュールに分散
- ノードシステムはr166以降のThree.jsで大幅に改善されている
- WebGPUバックエンドではWGSLに直接コンパイルされ、より効率的なシェーダーが生成される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Node.js | `src/nodes/core/Node.js` | ノードの基底クラス、ノードグラフの基本構造 |
| 1-2 | constants.js | `src/nodes/core/constants.js` | ノードタイプの定数定義 |

**読解のコツ**: TSLはノードグラフシステムであり、各ノードはgetCode()メソッドでシェーダーコードを生成する。ノード間の依存関係がグラフを形成する。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | TSL.js | `src/nodes/TSL.js` | エクスポートの集約ファイル、利用可能なすべてのノードを確認 |

**主要処理フロー**:
1. **1-2行目**: 定数のエクスポート
2. **4-18行目**: コアノード（AssignNode, AttributeNode等）のエクスポート
3. **21-28行目**: 数学ノード（BitcastNode, Hash等）のエクスポート
4. **30-50行目**: ユーティリティノードのエクスポート
5. **52-53行目**: TSLBase（基本関数）のエクスポート
6. **55-93行目**: アクセサノード（テクスチャ、カメラ、位置等）のエクスポート
7. **95-115行目**: ディスプレイノード（色調整、トーンマッピング等）のエクスポート
8. **139-148行目**: ライティングノードのエクスポート
9. **163-177行目**: BSDF関数のエクスポート

#### Step 3: TSLベース機能を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | TSLBase.js | `src/nodes/tsl/TSLBase.js` | float(), vec2(), vec3(), vec4()等の基本関数 |

**主要処理フロー**:
- TSLベース関数は型付きノードを生成するファクトリ関数
- チェイン可能なAPIでノードグラフを構築

#### Step 4: 特定のノード実装を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | TextureNode.js | `src/nodes/accessors/TextureNode.js` | テクスチャサンプリングの実装 |
| 4-2 | UniformNode.js | `src/nodes/core/UniformNode.js` | ユニフォーム変数の実装 |
| 4-3 | BRDF_GGX.js | `src/nodes/functions/BSDF/BRDF_GGX.js` | PBRシェーディングの実装 |

### プログラム呼び出し階層図

```
TSL.js (エクスポート集約)
    │
    ├─ core/
    │      ├─ constants.js (ノードタイプ定数)
    │      ├─ AssignNode.js
    │      ├─ AttributeNode.js
    │      ├─ UniformNode.js
    │      └─ ...
    │
    ├─ tsl/
    │      └─ TSLBase.js (float, vec2, vec3, vec4等)
    │
    ├─ math/
    │      ├─ MathUtils.js
    │      ├─ Hash.js
    │      └─ ...
    │
    ├─ accessors/
    │      ├─ TextureNode.js
    │      ├─ Camera.js
    │      ├─ Position.js
    │      └─ ...
    │
    ├─ display/
    │      ├─ ToneMappingNode.js
    │      ├─ ColorSpaceNode.js
    │      └─ ...
    │
    ├─ lighting/
    │      ├─ LightsNode.js
    │      ├─ ShadowNode.js
    │      └─ ...
    │
    └─ functions/BSDF/
           ├─ BRDF_GGX.js
           ├─ BRDF_Lambert.js
           ├─ F_Schlick.js
           └─ ...
```

### データフロー図

```
[入力]                         [処理]                           [出力]

JavaScript TSLコード ─────▶ TSL関数呼び出し ─────────────────▶ Nodeオブジェクト
  │                              │
  │                              ├─ float(), vec3()等
  │                              ├─ texture(), uniform()等
  │                              └─ add(), mul()等演算
  │
  └──────────────────────▶ NodeMaterial設定 ───────────────▶ マテリアル
                                 │
                                 └─ fragmentNode
                                 └─ vertexNode
                                 │
                                 ▼
                          NodeBuilder処理 ─────────────────▶ シェーダーコード
                                 │
                                 ├─ WebGL → GLSL
                                 └─ WebGPU → WGSL
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| TSL.js | `src/nodes/TSL.js` | ソース | エクスポート集約（メインエントリ） |
| TSLBase.js | `src/nodes/tsl/TSLBase.js` | ソース | 基本関数（float, vec2, vec3等） |
| Node.js | `src/nodes/core/Node.js` | ソース | ノード基底クラス |
| constants.js | `src/nodes/core/constants.js` | ソース | ノードタイプ定数 |
| NodeBuilder.js | `src/nodes/core/NodeBuilder.js` | ソース | シェーダーコード生成 |
| NodeMaterial.js | `src/materials/nodes/NodeMaterial.js` | ソース | ノードベースマテリアル |
| TextureNode.js | `src/nodes/accessors/TextureNode.js` | ソース | テクスチャアクセス |
| UniformNode.js | `src/nodes/core/UniformNode.js` | ソース | ユニフォーム変数 |
| BRDF_GGX.js | `src/nodes/functions/BSDF/BRDF_GGX.js` | ソース | PBR GGX分布関数 |
| LightsNode.js | `src/nodes/lighting/LightsNode.js` | ソース | ライティング処理 |
