# 機能設計書 25-Fog

## 概要

本ドキュメントは、Three.jsにおけるFog（フォグ）機能の詳細設計を記述する。Fogは距離ベースのリニア（線形）フォグ効果を提供するクラスである。

### 本機能の処理概要

Fogは、カメラからの距離に基づいて3Dシーン内のオブジェクトを徐々にフォグ色にブレンドするリニアフォグ効果を提供するクラスである。near（開始距離）とfar（終了距離）の間で線形に補間され、奥のオブジェクトほどフォグ色に近づく。

**業務上の目的・背景**：フォグ効果は3Dシーンにおいて大気遠近法を表現し、シーンの奥行き感を強調するために使用される。また、遠景のレンダリング負荷を軽減する目的でも活用される。リニアフォグは最もシンプルで直感的なフォグ効果であり、開始距離と終了距離を明示的に指定できるため、制御が容易である。

**機能の利用シーン**：
- 屋外シーンの大気遠近法表現
- 霧や靄のビジュアル効果
- 遠景の詳細を隠すことによるレンダリング最適化
- シーンの雰囲気演出（ホラー、幻想的な表現等）
- 視界距離の制限

**主要な処理内容**：
1. フォグ色（color）の管理
2. フォグ開始距離（near）の管理
3. フォグ終了距離（far）の管理
4. シリアライズ/デシリアライズ（JSON変換）
5. クローン機能

**関連システム・外部連携**：Scene、WebGLRenderer、シェーダープログラムと連携して動作する。

**権限による制御**：特になし（ライブラリ機能として権限制御は行わない）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | Sidebar - Scene | 補助機能 | リニアフォグ効果の設定 |

## 機能種別

設定管理 / ビジュアル効果

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number\|Color | Yes | フォグの色 | - |
| near | number | No | フォグ開始距離 | デフォルト: 1 |
| far | number | No | フォグ終了距離 | デフォルト: 1000 |

### インスタンスプロパティ

| プロパティ名 | 型 | デフォルト | 説明 |
|-------------|-----|----------|------|
| isFog | boolean | true | Fogインスタンスを識別するフラグ |
| name | string | '' | フォグの名前 |
| color | Color | - | フォグの色 |
| near | number | 1 | フォグ開始距離 |
| far | number | 1000 | フォグ終了距離 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isFog | boolean | Fogであることを示すフラグ（true） |
| color | Color | フォグの色 |
| near | number | フォグ開始距離 |
| far | number | フォグ終了距離 |

### 出力先

- Scene.fogプロパティに設定
- JSON形式でのシリアライズ出力

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ実行
   └─ isFog = true を設定
   └─ name = '' を設定
   └─ color = new Color(color) を設定
   └─ near = near を設定
   └─ far = far を設定

2. シーンへの適用
   └─ scene.fog = fogInstance
   └─ レンダラーがfogを検出
   └─ シェーダーにfogパラメータを渡す

3. レンダリング時の処理（シェーダー側）
   └─ カメラからの距離を計算
   └─ near〜farの範囲で線形補間
   └─ オブジェクト色とfog色をブレンド
```

### フローチャート

```mermaid
flowchart TD
    A[Fog生成] --> B[color設定]
    B --> C[near/far設定]
    C --> D[scene.fogに設定]
    D --> E[レンダリング実行]
    E --> F{距離 < near?}
    F -->|Yes| G[フォグなし]
    F -->|No| H{距離 > far?}
    H -->|Yes| I[完全にフォグ色]
    H -->|No| J[線形補間でブレンド]
    G --> K[最終色出力]
    I --> K
    J --> K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 距離範囲 | near未満の距離ではフォグは適用されない | 常時 |
| BR-02 | 完全フォグ | far以上の距離ではオブジェクトは完全にフォグ色になる | 常時 |
| BR-03 | 線形補間 | near〜farの間で距離に比例してフォグが濃くなる | 常時 |

### 計算ロジック

リニアフォグの計算式:

```
fogFactor = (far - distance) / (far - near)
fogFactor = clamp(fogFactor, 0.0, 1.0)
finalColor = mix(fogColor, objectColor, fogFactor)
```

- `distance`: カメラからオブジェクトまでの距離
- `fogFactor`: フォグの適用度（0.0 = 完全フォグ、1.0 = フォグなし）
- `finalColor`: 最終的な色

## データベース操作仕様

該当なし（クライアントサイドのみの処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 明示的なエラー処理なし | 入力検証は呼び出し側の責任 |

### リトライ仕様

リトライ処理は不要

## トランザクション仕様

該当なし（クライアントサイドのみの処理）

## パフォーマンス要件

- フォグの計算はシェーダー側で行われるため、CPU負荷は最小限
- far距離を適切に設定することで、遠景のレンダリングを省略可能

## セキュリティ考慮事項

特になし（ブラウザのセキュリティモデルに依存）

## 備考

- Colorクラスは数値（0xRRGGBB形式）またはColorインスタンスを受け取る
- FogExp2（指数フォグ）と比較して、明確な境界設定が可能
- マテリアルによってはfogプロパティでフォグの適用/非適用を切り替え可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FogはシンプルなクラスでObject3Dを継承していない。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Fog.js | `src/scenes/Fog.js` | クラスの全体構造（99行） |
| 1-2 | Color.js | `src/math/Color.js` | 色の管理クラス |

**読解のコツ**: Fogは純粋なデータオブジェクトで、レンダリングロジックはシェーダー側で実装される。

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Fog.js | `src/scenes/Fog.js` | 21-64行目 |

**主要処理フロー**:
1. **30行目**: isFog = true を設定
2. **37行目**: name = '' を設定
3. **44行目**: color = new Color(color) を設定
4. **53行目**: near = near を設定（デフォルト1）
5. **63行目**: far = far を設定（デフォルト1000）

#### Step 3: clone()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Fog.js | `src/scenes/Fog.js` | 72-76行目 |

**主要処理フロー**:
- **74行目**: 新しいFogインスタンスを生成して返す

#### Step 4: toJSON()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Fog.js | `src/scenes/Fog.js` | 84-93行目 |

**主要処理フロー**:
- **86-92行目**: type, name, color, near, farをJSONオブジェクトとして返す
- **89行目**: color.getHex()で色を16進数に変換

### プログラム呼び出し階層図

```
Fog
    |
    +-- constructor(color, near = 1, far = 1000)
    |       |
    |       +-- new Color(color)
    |
    +-- clone()
    |       |
    |       +-- new Fog(this.color, this.near, this.far)
    |
    +-- toJSON()
            |
            +-- color.getHex()
```

### データフロー図

```
[入力]                      [処理]                         [出力]

color (number/Color) ───> Fog構築 ─────────────> Fogインスタンス
near (number)         |                              |
far (number)          |                              v
                      |                    scene.fog = fog
                      |                              |
                      |                              v
                      |                    レンダラーで使用
                      |                              |
                      v                              v
               Color変換             シェーダーにパラメータ渡す
                      |                              |
                      v                              v
              color.getHex()            フラグメントシェーダーで
                                        線形補間処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Fog.js | `src/scenes/Fog.js` | ソース | Fogクラスの実装 |
| FogExp2.js | `src/scenes/FogExp2.js` | ソース | 指数フォグ（比較参照） |
| Scene.js | `src/scenes/Scene.js` | ソース | fogプロパティを持つ |
| Color.js | `src/math/Color.js` | ソース | 色の管理 |
