# 機能設計書 26-FogExp2

## 概要

本ドキュメントは、Three.jsにおけるFogExp2（指数フォグ）機能の詳細設計を記述する。FogExp2は距離に応じて指数関数的に濃くなるフォグ効果を提供するクラスである。

### 本機能の処理概要

FogExp2は、カメラからの距離に基づいて指数関数的にフォグが濃くなる効果を提供するクラスである。Fog（リニアフォグ）とは異なり、密度（density）パラメータのみで制御され、カメラ近くでは視界がクリアで、遠くなるにつれて急速にフォグが濃くなる特性を持つ。

**業務上の目的・背景**：指数フォグは、よりリアルな大気効果を表現するのに適している。実際の霧や霞は指数関数的な減衰特性を持つため、FogExp2を使用することでより自然なビジュアル効果を実現できる。また、密度パラメータのみで制御できるため、シンプルな設定で効果を調整できる。

**機能の利用シーン**：
- 自然な霧や霞の表現
- 汚染された大気の表現
- 水中シーンの視界制限
- ホラーゲームでの不気味な雰囲気演出
- 大規模な屋外シーンの遠景処理

**主要な処理内容**：
1. フォグ色（color）の管理
2. フォグ密度（density）の管理
3. シリアライズ/デシリアライズ（JSON変換）
4. クローン機能

**関連システム・外部連携**：Scene、WebGLRenderer、シェーダープログラムと連携して動作する。

**権限による制御**：特になし（ライブラリ機能として権限制御は行わない）。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 3 | Sidebar - Scene | 補助機能 | 指数フォグ効果の設定 |

## 機能種別

設定管理 / ビジュアル効果

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number\|Color | Yes | フォグの色 | - |
| density | number | No | フォグの密度 | デフォルト: 0.00025 |

### インスタンスプロパティ

| プロパティ名 | 型 | デフォルト | 説明 |
|-------------|-----|----------|------|
| isFogExp2 | boolean | true | FogExp2インスタンスを識別するフラグ |
| name | string | '' | フォグの名前 |
| color | Color | - | フォグの色 |
| density | number | 0.00025 | フォグの密度 |

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isFogExp2 | boolean | FogExp2であることを示すフラグ（true） |
| color | Color | フォグの色 |
| density | number | フォグの密度 |

### 出力先

- Scene.fogプロパティに設定
- JSON形式でのシリアライズ出力

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ実行
   └─ isFogExp2 = true を設定
   └─ name = '' を設定
   └─ color = new Color(color) を設定
   └─ density = density を設定

2. シーンへの適用
   └─ scene.fog = fogExp2Instance
   └─ レンダラーがfogを検出
   └─ シェーダーにfogパラメータを渡す

3. レンダリング時の処理（シェーダー側）
   └─ カメラからの距離を計算
   └─ 指数関数で減衰率を計算
   └─ オブジェクト色とfog色をブレンド
```

### フローチャート

```mermaid
flowchart TD
    A[FogExp2生成] --> B[color設定]
    B --> C[density設定]
    C --> D[scene.fogに設定]
    D --> E[レンダリング実行]
    E --> F[距離計算]
    F --> G[指数減衰計算]
    G --> H[fogFactor = exp-density*density*distance*distance]
    H --> I[最終色 = mix fogColor objectColor fogFactor]
    I --> J[最終色出力]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | 指数減衰 | 距離の2乗に比例してフォグが濃くなる | 常時 |
| BR-02 | 密度範囲 | densityは正の小数値、一般的に0.0001〜0.1程度 | 常時 |
| BR-03 | 境界なし | Fogと異なり、明確な開始/終了距離がない | 常時 |

### 計算ロジック

指数フォグの計算式:

```
fogFactor = exp(-density * density * distance * distance)
fogFactor = clamp(fogFactor, 0.0, 1.0)
finalColor = mix(fogColor, objectColor, fogFactor)
```

- `distance`: カメラからオブジェクトまでの距離
- `density`: フォグの密度
- `fogFactor`: フォグの適用度（0.0 = 完全フォグ、1.0 = フォグなし）
- `finalColor`: 最終的な色

## データベース操作仕様

該当なし（クライアントサイドのみの処理）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 明示的なエラー処理なし | 入力検証は呼び出し側の責任 |

### リトライ仕様

リトライ処理は不要

## トランザクション仕様

該当なし（クライアントサイドのみの処理）

## パフォーマンス要件

- フォグの計算はシェーダー側で行われるため、CPU負荷は最小限
- Fogと同等のパフォーマンス特性

## セキュリティ考慮事項

特になし（ブラウザのセキュリティモデルに依存）

## 備考

- デフォルトのdensity（0.00025）は非常に薄いフォグ
- 一般的な使用では0.001〜0.01程度の値が多い
- FogExp2は"Exponential Squared Fog"（指数二乗フォグ）の略

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

FogExp2はFogと同様にシンプルなデータクラス。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | FogExp2.js | `src/scenes/FogExp2.js` | クラスの全体構造（87行） |
| 1-2 | Fog.js | `src/scenes/Fog.js` | 比較参照 |
| 1-3 | Color.js | `src/math/Color.js` | 色の管理クラス |

**読解のコツ**: FogExp2はFogと非常によく似た構造。near/farの代わりにdensityのみを使用する点が異なる。

#### Step 2: コンストラクタを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FogExp2.js | `src/scenes/FogExp2.js` | 21-54行目 |

**主要処理フロー**:
1. **30行目**: isFogExp2 = true を設定
2. **37行目**: name = '' を設定
3. **44行目**: color = new Color(color) を設定
4. **52行目**: density = density を設定（デフォルト0.00025）

#### Step 3: clone()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FogExp2.js | `src/scenes/FogExp2.js` | 61-65行目 |

**主要処理フロー**:
- **63行目**: 新しいFogExp2インスタンスを生成して返す

#### Step 4: toJSON()メソッドを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | FogExp2.js | `src/scenes/FogExp2.js` | 73-81行目 |

**主要処理フロー**:
- **75-80行目**: type, name, color, densityをJSONオブジェクトとして返す
- **78行目**: color.getHex()で色を16進数に変換

### プログラム呼び出し階層図

```
FogExp2
    |
    +-- constructor(color, density = 0.00025)
    |       |
    |       +-- new Color(color)
    |
    +-- clone()
    |       |
    |       +-- new FogExp2(this.color, this.density)
    |
    +-- toJSON()
            |
            +-- color.getHex()
```

### データフロー図

```
[入力]                      [処理]                         [出力]

color (number/Color) ───> FogExp2構築 ──────────> FogExp2インスタンス
density (number)      |                              |
                      |                              v
                      |                    scene.fog = fogExp2
                      |                              |
                      |                              v
                      |                    レンダラーで使用
                      |                              |
                      v                              v
               Color変換             シェーダーにパラメータ渡す
                      |                              |
                      v                              v
              color.getHex()            フラグメントシェーダーで
                                        指数減衰処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FogExp2.js | `src/scenes/FogExp2.js` | ソース | FogExp2クラスの実装 |
| Fog.js | `src/scenes/Fog.js` | ソース | リニアフォグ（比較参照） |
| Scene.js | `src/scenes/Scene.js` | ソース | fogプロパティを持つ |
| Color.js | `src/math/Color.js` | ソース | 色の管理 |
