# 機能設計書 32-TorusGeometry

## 概要

本ドキュメントはThree.jsにおけるトーラス（ドーナツ型）ジオメトリ（TorusGeometry）クラスの機能設計を記述する。TorusGeometryはBufferGeometryを継承し、パラメトリックなトーラス形状を生成するジオメトリクラスである。

### 本機能の処理概要

TorusGeometryは3Dグラフィックスにおいてトーラス（ドーナツ型）形状のメッシュを生成するためのジオメトリクラスである。中心からチューブ中心までの距離、チューブの半径、周方向・チューブ方向の分割数を指定することで、様々なサイズと詳細度のトーラス形状を作成できる。

**業務上の目的・背景**：3Dグラフィックスアプリケーションでは、リング状やドーナツ状のオブジェクトが頻繁に必要となる。タイヤ、指輪、パイプの接合部、装飾的なデザイン要素など、多くのオブジェクトがトーラス形状を基本としている。TorusGeometryはこれらの形状を数学的に正確に生成し、3Dシーンの構築を効率化する。

**機能の利用シーン**：3Dエディタでのオブジェクト追加（Geometry Browserでのプレビュー）、ゲーム開発におけるリング状オブジェクトの作成、機械設計の可視化における部品表示、教育用3Dコンテンツにおける幾何学形状の描画などで利用される。

**主要な処理内容**：
1. トーラスの半径とチューブ半径からパラメトリックに頂点座標を計算
2. 周方向（tubularSegments）とチューブ断面方向（radialSegments）で分割
3. 各頂点の法線ベクトルをチューブ中心から外向きに計算
4. UV座標をセグメント分割に基づいて生成
5. 三角形面のインデックスを生成

**関連システム・外部連携**：レンダラー（WebGLRenderer、WebGPURenderer）と連携してメッシュを描画する。マテリアルと組み合わせてMeshオブジェクトを構成する。

**権限による制御**：特になし。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 5 | Sidebar - Geometry | 補助機能 | トーラスパラメータの編集 |
| 12 | Menubar - Add | 補助機能 | Torusメッシュの追加 |
| 35 | Geometry Browser | 主機能 | トーラスジオメトリのプレビュー |

## 機能種別

計算処理 / ジオメトリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| radius | number | No | トーラス中心からチューブ中心までの半径（デフォルト: 1） | なし |
| tube | number | No | チューブの半径（デフォルト: 0.4） | radiusより小さい必要あり |
| radialSegments | number | No | チューブ断面方向の分割数（デフォルト: 12） | Math.floorで整数化 |
| tubularSegments | number | No | 周方向の分割数（デフォルト: 48） | Math.floorで整数化 |
| arc | number | No | 周方向の中心角（ラジアン、デフォルト: Math.PI * 2） | なし |
| thetaStart | number | No | チューブ断面の開始角度（ラジアン、デフォルト: 0） | なし |
| thetaLength | number | No | チューブ断面の中心角（ラジアン、デフォルト: Math.PI * 2） | なし |

### 入力データソース

コンストラクタ引数として直接指定される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| position | Float32BufferAttribute | 頂点の3D位置座標 |
| normal | Float32BufferAttribute | 頂点の法線ベクトル |
| uv | Float32BufferAttribute | テクスチャマッピング用UV座標 |
| index | Uint16/Uint32BufferAttribute | 三角形面のインデックス |

### 出力先

BufferGeometry属性として保持され、Meshオブジェクトを通じてGPUに送信される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 引数（radius, tube, radialSegments等）を受け取る
2. パラメータを整数化
   └─ radialSegments, tubularSegmentsをMath.floorで処理
3. 頂点・法線・UV生成ループ
   └─ 二重ループでradialSegments × tubularSegmentsの頂点を生成
4. インデックス生成ループ
   └─ 四角形を2つの三角形に分割してインデックスを生成
5. BufferGeometry属性を設定
   └─ setIndex, setAttributeで属性を登録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[コンストラクタ引数を受け取る]
    B --> C[segments値を整数化]
    C --> D[radialSegments × tubularSegmentsループ]
    D --> E[頂点座標計算: radius + tube*cos v × cos/sin u]
    E --> F[法線計算: 頂点 - チューブ中心]
    F --> G[UV座標生成]
    G --> H{全セグメント完了?}
    H -->|No| D
    H -->|Yes| I[インデックス生成]
    I --> J[BufferGeometry属性設定]
    J --> K[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-32-001 | トーラス頂点計算 | vertex.x = (radius + tube*cos(v))*cos(u), vertex.y = (radius + tube*cos(v))*sin(u), vertex.z = tube*sin(v) | 常時 |
| BR-32-002 | 法線計算 | 頂点位置からチューブ中心を引いて正規化 | 常時 |
| BR-32-003 | パラメータ保存 | 生成後のパラメータ変更は形状に反映されない | 常時 |

### 計算ロジック

トーラスのパラメトリック方程式:
- x = (R + r*cos(v)) * cos(u)
- y = (R + r*cos(v)) * sin(u)
- z = r * sin(v)

ここで R=radius（主半径）、r=tube（チューブ半径）、u=周方向角度、v=チューブ断面角度

## データベース操作仕様

該当なし（クライアントサイドのみで動作）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 特別なエラー処理なし | パラメータは暗黙的に処理される |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 頂点数は (radialSegments+1) × (tubularSegments+1)
- 三角形数は radialSegments × tubularSegments × 2
- デフォルト設定で約600頂点、約1,100三角形

## セキュリティ考慮事項

特になし（クライアントサイドのジオメトリ生成処理）

## 備考

- arcパラメータで部分的なトーラス（扇形）を生成可能
- thetaStart/thetaLengthでチューブの断面形状を制御可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

BufferGeometryの属性構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点属性の管理方法 |
| 1-2 | Vector3.js | `src/math/Vector3.js` | 3Dベクトル演算（subVectors, normalize） |

**読解のコツ**: トーラスの頂点計算には三角関数が多用される。center、vertex、normalの各Vector3オブジェクトの役割を理解する。

#### Step 2: エントリーポイントを理解する

TorusGeometryクラスのコンストラクタが処理の起点。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | TorusGeometry.js | `src/geometries/TorusGeometry.js` | コンストラクタ全体の処理フロー |

**主要処理フロー**:
1. **31行目**: コンストラクタでパラメータを受け取る
2. **54-55行目**: radialSegments, tubularSegmentsを整数化
3. **72-103行目**: 二重ループで頂点・法線・UV生成
4. **107-125行目**: インデックス生成ループ
5. **129-132行目**: BufferGeometry属性設定

#### Step 3: 頂点計算の詳細を理解する

頂点座標と法線の計算ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | TorusGeometry.js | `src/geometries/TorusGeometry.js` | 72-103行目の頂点生成ループ |

**主要処理フロー**:
- **74行目**: v = thetaStart + (j/radialSegments) * thetaLength（チューブ断面角度）
- **78行目**: u = i/tubularSegments * arc（周方向角度）
- **82-84行目**: 頂点座標のパラメトリック計算
- **90-92行目**: 法線 = 頂点 - チューブ中心

### プログラム呼び出し階層図

```
TorusGeometry.constructor()
    │
    ├─ BufferGeometry.constructor()
    │
    ├─ 頂点生成ループ（j: 0→radialSegments）
    │      └─ 内側ループ（i: 0→tubularSegments）
    │             ├─ vertex計算
    │             ├─ normal計算（subVectors, normalize）
    │             └─ uv計算
    │
    ├─ インデックス生成ループ
    │      └─ 四角形→2三角形変換
    │
    └─ setAttribute()
           ├─ position
           ├─ normal
           └─ uv
```

### データフロー図

```
[入力]               [処理]                    [出力]

パラメータ ───▶ TorusGeometry ───▶ BufferGeometry属性
(radius,           │                 (position,
 tube,             │                  normal,
 segments,         ▼                  uv,
 arc等)        頂点計算              index)
               (パラメトリック方程式)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| TorusGeometry.js | `src/geometries/TorusGeometry.js` | ソース | トーラスジオメトリクラス定義 |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | 基底クラス、属性管理 |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | Float32BufferAttribute定義 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル計算（subVectors, normalize） |
