# 機能設計書 34-CircleGeometry

## 概要

本ドキュメントはThree.jsにおける円形ジオメトリ（CircleGeometry）クラスの機能設計を記述する。CircleGeometryはBufferGeometryを継承し、2D円形（ディスク）形状を生成するジオメトリクラスである。

### 本機能の処理概要

CircleGeometryは3Dグラフィックスにおいて円形（ディスク）のメッシュを生成するためのジオメトリクラスである。半径と分割数を指定することで、様々なサイズと詳細度の円形平面を作成できる。正多角形（三角形、四角形、五角形等）もセグメント数を調整することで生成可能。

**業務上の目的・背景**：3Dグラフィックスでは円形の平面オブジェクトが頻繁に必要となる。地面のマーカー、ボタン、コイン、スポットライトのエリア表示、UI要素など多くのオブジェクトが円形を基本としている。CircleGeometryはこれらの形状を効率的に生成する。

**機能の利用シーン**：3Dエディタでの平面オブジェクト作成、ゲーム開発におけるUI要素やマーカーの作成、データ可視化における円グラフの基盤、デカールや影の投影領域の表示。

**主要な処理内容**：
1. 中心点から放射状に頂点を配置
2. 開始角度と中心角を指定して部分的な扇形も生成可能
3. すべての法線はZ軸正方向（0, 0, 1）
4. UV座標は中心を(0.5, 0.5)として放射状にマッピング
5. 三角形ファンとしてインデックスを生成

**関連システム・外部連携**：レンダラー（WebGLRenderer、WebGPURenderer）と連携してメッシュを描画する。マテリアルと組み合わせてMeshオブジェクトを構成する。

**権限による制御**：特になし。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連なし |

## 機能種別

計算処理 / ジオメトリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| radius | number | No | 円の半径（デフォルト: 1） | なし |
| segments | number | No | 分割数（三角形の数、デフォルト: 32） | Math.max(3, segments) |
| thetaStart | number | No | 開始角度（ラジアン、デフォルト: 0） | なし |
| thetaLength | number | No | 中心角（ラジアン、デフォルト: Math.PI * 2） | なし |

### 入力データソース

コンストラクタ引数として直接指定される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| position | Float32BufferAttribute | 頂点の3D位置座標（z=0） |
| normal | Float32BufferAttribute | 頂点の法線ベクトル（すべて(0,0,1)） |
| uv | Float32BufferAttribute | テクスチャマッピング用UV座標 |
| index | Uint16/Uint32BufferAttribute | 三角形面のインデックス |

### 出力先

BufferGeometry属性として保持され、Meshオブジェクトを通じてGPUに送信される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 引数（radius, segments, thetaStart, thetaLength）を受け取る
2. segments値を検証
   └─ Math.max(3, segments)で最小3を保証
3. 中心点を追加
   └─ 位置(0,0,0)、法線(0,0,1)、UV(0.5,0.5)
4. 外周頂点を生成
   └─ segmentsループで放射状に頂点を配置
5. インデックス生成
   └─ 三角形ファンとして中心点を共有
6. BufferGeometry属性を設定
   └─ setIndex, setAttributeで属性を登録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[コンストラクタ引数を受け取る]
    B --> C[segments = max 3, segments]
    C --> D[中心点を追加: 0,0,0]
    D --> E[segmentsループ]
    E --> F[頂点座標: radius*cos/sin]
    F --> G[法線: 0,0,1]
    G --> H[UV座標計算]
    H --> I{全セグメント完了?}
    I -->|No| E
    I -->|Yes| J[インデックス生成: 三角形ファン]
    J --> K[BufferGeometry属性設定]
    K --> L[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-34-001 | 最小セグメント数 | segments は最小3（三角形）を保証 | 常時 |
| BR-34-002 | 頂点座標計算 | x = radius*cos(theta), y = radius*sin(theta), z = 0 | 常時 |
| BR-34-003 | 法線方向 | すべての頂点の法線は(0, 0, 1) | 常時 |
| BR-34-004 | UV座標 | 中心(0.5, 0.5)から放射状に配置 | 常時 |

### 計算ロジック

円形頂点のパラメトリック方程式:
- x = radius * cos(thetaStart + s/segments * thetaLength)
- y = radius * sin(thetaStart + s/segments * thetaLength)
- z = 0

UV座標:
- u = (x / radius + 1) / 2
- v = (y / radius + 1) / 2

## データベース操作仕様

該当なし（クライアントサイドのみで動作）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | segments < 3 | 自動的に3に補正 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 頂点数は segments + 1（中心点 + 外周頂点）
- 三角形数は segments
- デフォルト設定（segments=32）で33頂点、32三角形

## セキュリティ考慮事項

特になし（クライアントサイドのジオメトリ生成処理）

## 備考

- segments=3で正三角形、segments=4で正方形、segments=6で正六角形を生成可能
- thetaStartとthetaLengthで扇形（パイチャートの一部）を生成可能
- XY平面上に生成されるため、回転操作が必要な場合はMesh側で行う

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

BufferGeometryの属性構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点属性の管理方法 |
| 1-2 | Vector3.js | `src/math/Vector3.js` | 3Dベクトル操作 |
| 1-3 | Vector2.js | `src/math/Vector2.js` | 2Dベクトル（UV座標） |

**読解のコツ**: CircleGeometryはすべての頂点がz=0平面上にあり、法線もすべて同じ方向を向くシンプルな構造。

#### Step 2: エントリーポイントを理解する

CircleGeometryクラスのコンストラクタが処理の起点。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CircleGeometry.js | `src/geometries/CircleGeometry.js` | コンストラクタ全体の処理フロー |

**主要処理フロー**:
1. **35行目**: コンストラクタでパラメータを受け取る
2. **55行目**: segments = Math.max(3, segments)で最小値を保証
3. **71-73行目**: 中心点を追加（位置(0,0,0)、法線(0,0,1)、UV(0.5,0.5)）
4. **75-97行目**: segmentsループで外周頂点を生成
5. **101-105行目**: インデックス生成（三角形ファン）
6. **109-112行目**: BufferGeometry属性設定

#### Step 3: 頂点計算の詳細を理解する

頂点座標とUV座標の計算ロジックを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CircleGeometry.js | `src/geometries/CircleGeometry.js` | 75-97行目の頂点生成ループ |

**主要処理フロー**:
- **77行目**: segment = thetaStart + s/segments * thetaLength
- **81-82行目**: vertex.x = radius * cos(segment), vertex.y = radius * sin(segment)
- **92-93行目**: UV座標計算: (vertices[i]/radius + 1) / 2

### プログラム呼び出し階層図

```
CircleGeometry.constructor()
    │
    ├─ BufferGeometry.constructor()
    │
    ├─ 中心点追加
    │      ├─ vertices.push(0, 0, 0)
    │      ├─ normals.push(0, 0, 1)
    │      └─ uvs.push(0.5, 0.5)
    │
    ├─ segmentsループ（s: 0→segments）
    │      ├─ vertex.x = radius * cos(segment)
    │      ├─ vertex.y = radius * sin(segment)
    │      ├─ normals.push(0, 0, 1)
    │      └─ uvs計算
    │
    ├─ インデックス生成（三角形ファン）
    │      └─ indices.push(i, i+1, 0)
    │
    └─ setAttribute()
           ├─ position
           ├─ normal
           └─ uv
```

### データフロー図

```
[入力]               [処理]                    [出力]

パラメータ ───▶ CircleGeometry ───▶ BufferGeometry属性
(radius,           │                 (position,
 segments,         │                  normal,
 thetaStart,       ▼                  uv,
 thetaLength)   頂点計算              index)
               (cos/sin計算)
                   │
                   ▼
              三角形ファン生成
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CircleGeometry.js | `src/geometries/CircleGeometry.js` | ソース | 円形ジオメトリクラス定義 |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | 基底クラス、属性管理 |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | Float32BufferAttribute定義 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル計算 |
| Vector2.js | `src/math/Vector2.js` | ソース | 2Dベクトル（UV座標） |
