# 機能設計書 36-CapsuleGeometry

## 概要

本ドキュメントはThree.jsにおけるカプセル形状ジオメトリ（CapsuleGeometry）クラスの機能設計を記述する。CapsuleGeometryはBufferGeometryを継承し、円柱と両端の半球を組み合わせたカプセル形状を生成するジオメトリクラスである。

### 本機能の処理概要

CapsuleGeometryは3Dグラフィックスにおいてカプセル形状（円柱の両端に半球がついた形状）のメッシュを生成するためのジオメトリクラスである。半径、高さ（中央部分）、キャップの分割数を指定することで、様々なサイズと詳細度のカプセル形状を作成できる。

**業務上の目的・背景**：3Dグラフィックスにおいてカプセル形状は非常に重要な基本形状である。物理シミュレーションにおけるコリジョン形状、薬のカプセル、弾丸、キャラクターの衝突判定用ボリュームなど、多くの用途で使用される。特にカプセルは球と円柱の利点を組み合わせた形状で、衝突判定が高速に行える特性を持つ。

**機能の利用シーン**：3Dエディタでのオブジェクト作成、ゲーム開発におけるキャラクターのコライダー形状、物理シミュレーションのプリミティブ形状、医療・製薬分野の3D可視化、UI要素としてのピル型ボタン。

**主要な処理内容**：
1. 下部キャップ（半球）の頂点を生成
2. 中央の円柱部分の頂点を生成
3. 上部キャップ（半球）の頂点を生成
4. 各頂点の法線を正しく計算（キャップ部分は球面法線、円柱部分は水平法線）
5. UV座標を連続的に配置

**関連システム・外部連携**：レンダラー（WebGLRenderer、WebGPURenderer）と連携してメッシュを描画する。マテリアルと組み合わせてMeshオブジェクトを構成する。

**権限による制御**：特になし。全てのユーザーが利用可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連なし |

## 機能種別

計算処理 / ジオメトリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| radius | number | No | カプセルの半径（デフォルト: 1） | なし |
| height | number | No | 中央部分（円柱部分）の高さ（デフォルト: 1） | Math.max(0, height) |
| capSegments | number | No | キャップ（半球）の分割数（デフォルト: 4） | Math.max(1, Math.floor(capSegments)) |
| radialSegments | number | No | 周方向の分割数（デフォルト: 8） | Math.max(3, Math.floor(radialSegments)) |
| heightSegments | number | No | 中央部分の高さ方向分割数（デフォルト: 1） | Math.max(1, Math.floor(heightSegments)) |

### 入力データソース

コンストラクタ引数として直接指定される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| position | Float32BufferAttribute | 頂点の3D位置座標 |
| normal | Float32BufferAttribute | 頂点の法線ベクトル |
| uv | Float32BufferAttribute | テクスチャマッピング用UV座標 |
| index | Uint16/Uint32BufferAttribute | 三角形面のインデックス |

### 出力先

BufferGeometry属性として保持され、Meshオブジェクトを通じてGPUに送信される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 引数（radius, height, capSegments等）を受け取る
2. パラメータを検証・整数化
   └─ height >= 0, capSegments >= 1, radialSegments >= 3, heightSegments >= 1
3. 総弧長を計算
   └─ 上下キャップ + 中央部分の全長
4. 頂点生成ループ（iy: 0 → numVerticalSegments）
   └─ キャップと円柱部分で計算式を切り替え
5. 各ループでインデックスを生成
6. BufferGeometry属性を設定
   └─ setIndex, setAttributeで属性を登録
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[コンストラクタ引数を受け取る]
    B --> C[パラメータ検証・整数化]
    C --> D[総弧長計算]
    D --> E[numVerticalSegmentsループ]
    E --> F{iy <= capSegments?}
    F -->|Yes| G[下部キャップ頂点計算]
    F -->|No| H{iy <= capSegments + heightSegments?}
    H -->|Yes| I[中央円柱部分頂点計算]
    H -->|No| J[上部キャップ頂点計算]
    G --> K[法線・UV計算]
    I --> K
    J --> K
    K --> L[radialSegmentsループ]
    L --> M[頂点・法線・UV追加]
    M --> N[インデックス生成]
    N --> O{全セグメント完了?}
    O -->|No| E
    O -->|Yes| P[BufferGeometry属性設定]
    P --> Q[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-36-001 | 高さ検証 | heightは0以上を保証 | 常時 |
| BR-36-002 | セグメント検証 | capSegments >= 1, radialSegments >= 3, heightSegments >= 1 | 常時 |
| BR-36-003 | キャップ弧長 | capArcLength = (PI/2) * radius | 常時 |
| BR-36-004 | UV連続性 | v座標は全体の弧長に対する比率 | 常時 |

### 計算ロジック

カプセルは3つの部分で構成:
1. **下部キャップ**: 半球（角度 -PI/2 から 0）
2. **中央部分**: 円柱（高さ height）
3. **上部キャップ**: 半球（角度 0 から PI/2）

頂点座標計算:
- 下部キャップ: y = -halfHeight - radius*cos(angle), r = radius*sin(angle)
- 中央部分: y = -halfHeight + progress*height, r = radius
- 上部キャップ: y = halfHeight + radius*sin(angle), r = radius*cos(angle)

## データベース操作仕様

該当なし（クライアントサイドのみで動作）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | height < 0 | 自動的に0に補正 |
| - | - | capSegments < 1 | 自動的に1に補正 |
| - | - | radialSegments < 3 | 自動的に3に補正 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 頂点数は (capSegments*2 + heightSegments + 1) × (radialSegments + 1)
- デフォルト設定で約90頂点
- 滑らかなカプセルにはcapSegments=8以上を推奨

## セキュリティ考慮事項

特になし（クライアントサイドのジオメトリ生成処理）

## 備考

- height=0の場合は球体と同等の形状になる
- 物理エンジンとの連携時はこの形状がそのままコリジョン形状として使用可能
- Y軸方向に配置される（長軸がY軸）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

BufferGeometryの属性構造を理解することが重要。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点属性の管理方法 |
| 1-2 | Vector3.js | `src/math/Vector3.js` | 3Dベクトル操作、normalize |

**読解のコツ**: カプセルは3つの領域（下部キャップ、中央円柱、上部キャップ）で構成される。各領域で頂点計算ロジックが異なる。

#### Step 2: エントリーポイントを理解する

CapsuleGeometryクラスのコンストラクタが処理の起点。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CapsuleGeometry.js | `src/geometries/CapsuleGeometry.js` | コンストラクタ全体の処理フロー |

**主要処理フロー**:
1. **29行目**: コンストラクタでパラメータを受け取る
2. **50-53行目**: パラメータの検証と整数化
3. **64-67行目**: 弧長と分割数の計算
4. **77-182行目**: 頂点生成とインデックス生成のループ
5. **186-189行目**: BufferGeometry属性設定

#### Step 3: 3領域の頂点計算を理解する

下部キャップ、中央部分、上部キャップの計算ロジック。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | CapsuleGeometry.js | `src/geometries/CapsuleGeometry.js` | 84-115行目の3領域判定と計算 |

**主要処理フロー**:
- **84-93行目**: 下部キャップ（iy <= capSegments）
- **94-101行目**: 中央円柱部分（iy <= capSegments + heightSegments）
- **103-114行目**: 上部キャップ（残り）

### プログラム呼び出し階層図

```
CapsuleGeometry.constructor()
    │
    ├─ BufferGeometry.constructor()
    │
    ├─ パラメータ検証
    │      ├─ height = Math.max(0, height)
    │      ├─ capSegments = Math.max(1, Math.floor(capSegments))
    │      ├─ radialSegments = Math.max(3, Math.floor(radialSegments))
    │      └─ heightSegments = Math.max(1, Math.floor(heightSegments))
    │
    ├─ numVerticalSegmentsループ（iy: 0→numVerticalSegments）
    │      │
    │      ├─ 領域判定と頂点計算
    │      │      ├─ 下部キャップ（半球）
    │      │      ├─ 中央円柱
    │      │      └─ 上部キャップ（半球）
    │      │
    │      └─ radialSegmentsループ
    │             ├─ vertex計算
    │             ├─ normal計算・正規化
    │             ├─ uv計算
    │             └─ indices生成
    │
    └─ setAttribute()
           ├─ position
           ├─ normal
           └─ uv
```

### データフロー図

```
[入力]               [処理]                    [出力]

パラメータ ───▶ CapsuleGeometry ───▶ BufferGeometry属性
(radius,           │                 (position,
 height,           │                  normal,
 segments)         ▼                  uv,
               3領域計算              index)
                   │
         ┌────────┼────────┐
         ▼        ▼        ▼
     下部キャップ  中央円柱  上部キャップ
      (半球)              (半球)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CapsuleGeometry.js | `src/geometries/CapsuleGeometry.js` | ソース | カプセルジオメトリクラス定義 |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | 基底クラス、属性管理 |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | Float32BufferAttribute定義 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル計算 |
