# 機能設計書 41-PolyhedronGeometry

## 概要

本ドキュメントは、Three.jsライブラリにおける多面体ジオメトリの基底クラスであるPolyhedronGeometryの機能設計を記述する。

### 本機能の処理概要

PolyhedronGeometryは、多面体（ポリへドロン）のジオメトリを生成するための基底クラスである。頂点配列とインデックス配列を入力として受け取り、それらを球面上に投影して指定された詳細度で細分化することで、滑らかな多面体形状を生成する。

**業務上の目的・背景**：3Dグラフィックスにおいて、正多面体（正4面体、正8面体、正12面体、正20面体など）は基本的な形状として頻繁に使用される。これらの形状を効率的に生成し、詳細度を調整可能にすることで、様々な用途（科学的可視化、ゲーム、アート作品など）に対応できる柔軟なジオメトリ生成機能を提供する必要がある。PolyhedronGeometryは、これらの正多面体ジオメトリクラス（IcosahedronGeometry、OctahedronGeometry等）の共通基盤として機能する。

**機能の利用シーン**：
- 正多面体をベースとした3Dオブジェクトの作成
- 球体に近似した滑らかな形状の生成（詳細度を上げることで実現）
- 科学的データの可視化（分子構造など）
- ゲームやインタラクティブコンテンツでの基本形状の配置

**主要な処理内容**：
1. 入力された頂点・インデックス配列から基本形状を構築
2. 各三角形面を指定された詳細度で細分化（subdivide処理）
3. 全頂点を正規化して球面上に投影し、指定された半径を適用
4. UV座標を自動生成（球面マッピング）
5. シーム（継ぎ目）補正によるUV座標の調整
6. 法線ベクトルの計算（詳細度に応じてフラット/スムーズ）

**関連システム・外部連携**：BufferGeometry、Float32BufferAttribute、Vector3、Vector2などのThree.jsコアクラスと連携する。

**権限による制御**：なし（クライアントサイドのJavaScriptライブラリのため）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに関連画面の定義なし |

## 機能種別

計算処理 / ジオメトリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| vertices | Array<number> | No | 基本形状を定義する頂点座標のフラット配列（x,y,z,x,y,z,...） | デフォルト: [] |
| indices | Array<number> | No | 三角形面を定義するインデックスのフラット配列 | デフォルト: [] |
| radius | number | No | 生成される形状の半径 | デフォルト: 1 |
| detail | number | No | 細分化の詳細度（0以上の整数） | デフォルト: 0 |

### 入力データソース

コンストラクタ引数として直接渡される。派生クラス（IcosahedronGeometry等）では、各正多面体に固有の頂点・インデックスデータがハードコードされている。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| position属性 | Float32BufferAttribute | 頂点位置データ（3成分/頂点） |
| normal属性 | Float32BufferAttribute | 法線ベクトルデータ（3成分/頂点） |
| uv属性 | Float32BufferAttribute | UV座標データ（2成分/頂点） |
| parameters | Object | コンストラクタパラメータの保存 |

### 出力先

BufferGeometryの属性として格納され、Meshオブジェクトなどで使用される。

## 処理フロー

### 処理シーケンス

```
1. BufferGeometryの初期化
   └─ super()を呼び出し、基底クラスを初期化
2. パラメータの保存
   └─ this.parametersにコンストラクタ引数を格納
3. 頂点バッファの細分化（subdivide）
   └─ 各三角形面を詳細度に応じて再帰的に分割
4. 半径の適用（applyRadius）
   └─ 全頂点を正規化して球面上に配置し、半径を乗算
5. UV座標の生成（generateUVs）
   └─ 球面座標からUV座標を計算
6. シーム補正
   └─ テクスチャの継ぎ目を修正
7. BufferAttribute の設定
   └─ position、normal、uv属性を設定
8. 法線の計算
   └─ detail=0ならフラット法線、それ以外はスムーズ法線
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[BufferGeometry初期化]
    B --> C[パラメータ保存]
    C --> D[subdivide: 三角形面の細分化]
    D --> E[applyRadius: 球面投影と半径適用]
    E --> F[generateUVs: UV座標生成]
    F --> G[correctSeam: シーム補正]
    G --> H[BufferAttribute設定]
    H --> I{detail === 0?}
    I -->|Yes| J[computeVertexNormals: フラット法線]
    I -->|No| K[normalizeNormals: スムーズ法線]
    J --> L[終了]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 細分化アルゴリズム | 各三角形面をdetail+1の列に分割し、線形補間で中間頂点を生成 | 常時 |
| BR-002 | 球面投影 | 全頂点を原点からの方向に正規化し、指定半径を乗算 | 常時 |
| BR-003 | UV計算 | azimuth（方位角）とinclination（仰角）から球面UV座標を算出 | 常時 |
| BR-004 | シーム補正 | UV座標が0.9を超える場合と0.1未満の場合で補正を実施 | UV値の不連続点 |
| BR-005 | 法線計算方式 | detail=0ではフラット法線、それ以外ではスムーズ法線を使用 | 法線計算時 |

### 計算ロジック

**UV座標計算**:
- azimuth（方位角）: `Math.atan2(z, -x)`
- inclination（仰角）: `Math.atan2(-y, Math.sqrt(x*x + z*z))`
- u = azimuth / (2 * PI) + 0.5
- v = inclination / PI + 0.5

**細分化アルゴリズム**:
- cols = detail + 1
- 各三角形の辺をcols等分して中間頂点を線形補間で生成
- 生成された頂点グリッドから新しい三角形を構築

## データベース操作仕様

該当なし（クライアントサイド処理のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 暗黙的エラー | 頂点配列が空の場合 | 空のジオメトリが生成される |
| - | 暗黙的エラー | インデックスが頂点数を超える場合 | undefined値が使用される |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（クライアントサイド処理のみ）

## パフォーマンス要件

- 細分化処理は詳細度に応じて計算量が増加（O(4^detail)）
- 高い詳細度ではメモリ使用量も増加

## セキュリティ考慮事項

該当なし（クライアントサイドのジオメトリ生成処理）

## 備考

- このクラスは抽象基底クラスとして設計されており、通常はIcosahedronGeometry、OctahedronGeometry等の派生クラスを使用する
- JSON形式でのシリアライズ・デシリアライズに対応（fromJSONメソッド）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、PolyhedronGeometryが依存するコアクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | 基底クラスの属性管理方法を理解 |
| 1-2 | BufferAttribute.js | `src/core/BufferAttribute.js` | Float32BufferAttributeの構造を理解 |
| 1-3 | Vector3.js | `src/math/Vector3.js` | ベクトル演算（lerp、normalize等）を理解 |

**読解のコツ**: Three.jsのジオメトリは非インデックス形式で頂点データを格納する点に注意。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | コンストラクタの全体フローを把握 |

**主要処理フロー**:
1. **23-41行目**: コンストラクタ初期化とパラメータ保存
2. **50行目**: subdivide関数呼び出し（細分化処理）
3. **54行目**: applyRadius関数呼び出し（球面投影）
4. **58行目**: generateUVs関数呼び出し（UV生成）
5. **62-64行目**: BufferAttribute設定
6. **66-74行目**: 法線計算（detail値に応じた分岐）

#### Step 3: 細分化アルゴリズムを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | subdivide関数とsubdivideFace関数 |

**主要処理フロー**:
- **78-100行目**: subdivide関数 - 全面に対する細分化の反復
- **102-163行目**: subdivideFace関数 - 単一三角形の細分化ロジック
- **112-135行目**: 頂点グリッドの生成（lerp補間）
- **139-161行目**: 新しい三角形の構築

#### Step 4: UV座標生成を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | generateUVs、correctUVs、correctSeam関数 |

**主要処理フロー**:
- **187-207行目**: generateUVs関数 - 球面座標からUV計算
- **209-236行目**: correctSeam関数 - テクスチャ継ぎ目の補正
- **254-286行目**: correctUVs関数 - 極付近のUV補正
- **306-308行目**: azimuth関数 - 方位角計算
- **315-318行目**: inclination関数 - 仰角計算

### プログラム呼び出し階層図

```
PolyhedronGeometry.constructor()
    │
    ├─ subdivide(detail)
    │      └─ subdivideFace(a, b, c, detail)
    │             ├─ Vector3.lerp()
    │             └─ pushVertex()
    │
    ├─ applyRadius(radius)
    │      └─ Vector3.normalize().multiplyScalar()
    │
    └─ generateUVs()
           ├─ azimuth()
           ├─ inclination()
           ├─ correctUVs()
           │      └─ correctUV()
           └─ correctSeam()
```

### データフロー図

```
[入力]                      [処理]                           [出力]

vertices[]  ─────────────▶ subdivide()  ─────────────────▶ vertexBuffer[]
indices[]                      │
                               ▼
                         applyRadius()
                               │
                               ▼
                         generateUVs() ───────────────────▶ uvBuffer[]
                               │
                               ▼
                    setAttribute('position')
                    setAttribute('normal')  ──────────────▶ BufferGeometry
                    setAttribute('uv')
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | ソース | 多面体ジオメトリ基底クラス |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | ジオメトリ基底クラス |
| BufferAttribute.js | `src/core/BufferAttribute.js` | ソース | 頂点属性データ管理 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3次元ベクトル演算 |
| Vector2.js | `src/math/Vector2.js` | ソース | 2次元ベクトル演算（UV用） |
| IcosahedronGeometry.js | `src/geometries/IcosahedronGeometry.js` | ソース | 派生クラス（正20面体） |
| OctahedronGeometry.js | `src/geometries/OctahedronGeometry.js` | ソース | 派生クラス（正8面体） |
| TetrahedronGeometry.js | `src/geometries/TetrahedronGeometry.js` | ソース | 派生クラス（正4面体） |
| DodecahedronGeometry.js | `src/geometries/DodecahedronGeometry.js` | ソース | 派生クラス（正12面体） |
