# 機能設計書 42-IcosahedronGeometry

## 概要

本ドキュメントは、Three.jsライブラリにおける正20面体ジオメトリを生成するIcosahedronGeometryクラスの機能設計を記述する。

### 本機能の処理概要

IcosahedronGeometryは、正20面体（Icosahedron）のジオメトリを生成するクラスである。正20面体は12個の頂点と20個の正三角形の面から構成される正多面体であり、黄金比に基づく座標で定義される。詳細度（detail）パラメータを調整することで、より滑らかな球体に近い形状を生成することも可能である。

**業務上の目的・背景**：正20面体は、3Dグラフィックスにおいて球体を近似するための基本形状として広く使用される。各面が正三角形であるため、テッセレーション（細分化）によって均一に滑らかな球面を生成できる。ゲーム開発、科学的可視化（分子モデル、惑星など）、アート作品などで頻繁に活用される。また、サッカーボールの基本形状としても知られている。

**機能の利用シーン**：
- 球体に近似した低ポリゴンモデルの作成
- 惑星や天体のベース形状として使用
- 分子構造の可視化
- ゲームにおける基本形状やパーティクル
- ジオデシックドームなどの建築的形状の基盤

**主要な処理内容**：
1. 黄金比を用いた12個の頂点座標を定義
2. 20個の三角形面を定義するインデックス配列を構築
3. 親クラスPolyhedronGeometryのコンストラクタを呼び出し
4. 指定された半径と詳細度でジオメトリを生成
5. parametersプロパティにコンストラクタ引数を保存

**関連システム・外部連携**：PolyhedronGeometry（親クラス）、BufferGeometry、Meshなど

**権限による制御**：なし（クライアントサイドのJavaScriptライブラリのため）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに関連画面の定義なし |

## 機能種別

計算処理 / ジオメトリ生成

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| radius | number | No | 正20面体の半径 | デフォルト: 1 |
| detail | number | No | 細分化の詳細度（0以上の整数） | デフォルト: 0 |

### 入力データソース

コンストラクタ引数として直接渡される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| position属性 | Float32BufferAttribute | 頂点位置データ（3成分/頂点） |
| normal属性 | Float32BufferAttribute | 法線ベクトルデータ（3成分/頂点） |
| uv属性 | Float32BufferAttribute | UV座標データ（2成分/頂点） |
| parameters | Object | コンストラクタパラメータの保存（radius, detail） |

### 出力先

BufferGeometryの属性として格納され、Meshオブジェクトなどで使用される。

## 処理フロー

### 処理シーケンス

```
1. 黄金比tの計算
   └─ t = (1 + sqrt(5)) / 2 ≈ 1.618
2. 12頂点の座標定義
   └─ 黄金比を使用した座標配列を構築
3. 20面のインデックス定義
   └─ 各三角形面の頂点インデックスを指定
4. 親クラスコンストラクタ呼び出し
   └─ PolyhedronGeometry(vertices, indices, radius, detail)
5. typeプロパティの設定
   └─ 'IcosahedronGeometry'
6. parametersの保存
   └─ radius, detailを格納
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[黄金比tを計算]
    B --> C[12頂点の座標配列を定義]
    C --> D[20面のインデックス配列を定義]
    D --> E[super: PolyhedronGeometry呼び出し]
    E --> F[type = 'IcosahedronGeometry']
    F --> G[parameters保存]
    G --> H[終了]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-001 | 黄金比使用 | t = (1 + sqrt(5)) / 2 を頂点座標の計算に使用 | 常時 |
| BR-002 | 頂点数固定 | 基本形状は常に12頂点 | detail=0時 |
| BR-003 | 面数固定 | 基本形状は常に20面 | detail=0時 |
| BR-004 | 細分化継承 | detail>0の場合、親クラスの細分化アルゴリズムを適用 | detail>0時 |

### 計算ロジック

**黄金比（Golden Ratio）**:
- t = (1 + sqrt(5)) / 2 ≈ 1.6180339887

**頂点座標**:
正20面体の12頂点は以下の3つの直交する長方形の頂点として定義される：
- (-1, t, 0), (1, t, 0), (-1, -t, 0), (1, -t, 0)
- (0, -1, t), (0, 1, t), (0, -1, -t), (0, 1, -t)
- (t, 0, -1), (t, 0, 1), (-t, 0, -1), (-t, 0, 1)

## データベース操作仕様

該当なし（クライアントサイド処理のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 暗黙的エラー | 負のradiusが指定された場合 | 反転したジオメトリが生成される |
| - | 暗黙的エラー | 負のdetailが指定された場合 | 予期しない結果となる可能性 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし（クライアントサイド処理のみ）

## パフォーマンス要件

- detail=0: 12頂点、20面（60頂点：非インデックス形式）
- detail=1: 42頂点、80面
- detail=2: 162頂点、320面
- 高いdetail値ではメモリと計算時間が指数的に増加

## セキュリティ考慮事項

該当なし（クライアントサイドのジオメトリ生成処理）

## 備考

- 正20面体はプラトン立体の一つ
- detail値を上げると球体に近づく（ジオデシック球）
- JSON形式でのシリアライズ・デシリアライズに対応（fromJSONメソッド）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、正20面体の数学的構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | IcosahedronGeometry.js | `src/geometries/IcosahedronGeometry.js` | 頂点・インデックス定義を確認 |
| 1-2 | PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | 親クラスの処理フローを理解 |

**読解のコツ**: 頂点配列は黄金比を使用した数学的に美しい配置。インデックス配列は5つの頂点を共有する各頂点の周りに5つの三角形が形成される。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | IcosahedronGeometry.js | `src/geometries/IcosahedronGeometry.js` | コンストラクタの全体フロー |

**主要処理フロー**:
1. **26行目**: 黄金比tの計算 `const t = ( 1 + Math.sqrt( 5 ) ) / 2`
2. **28-32行目**: 12頂点の座標配列定義
3. **34-39行目**: 20面のインデックス配列定義
4. **41行目**: 親クラスコンストラクタ呼び出し
5. **43行目**: typeプロパティ設定
6. **52-55行目**: parameters保存

#### Step 3: 頂点構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | IcosahedronGeometry.js | `src/geometries/IcosahedronGeometry.js` | vertices配列の構造 |

**頂点配列の構造**:
```javascript
const vertices = [
    -1, t, 0,     // 頂点0: (-1, t, 0)
     1, t, 0,     // 頂点1: (1, t, 0)
    -1, -t, 0,    // 頂点2: (-1, -t, 0)
     1, -t, 0,    // 頂点3: (1, -t, 0)
     0, -1, t,    // 頂点4: (0, -1, t)
     0, 1, t,     // 頂点5: (0, 1, t)
     0, -1, -t,   // 頂点6: (0, -1, -t)
     0, 1, -t,    // 頂点7: (0, 1, -t)
     t, 0, -1,    // 頂点8: (t, 0, -1)
     t, 0, 1,     // 頂点9: (t, 0, 1)
    -t, 0, -1,    // 頂点10: (-t, 0, -1)
    -t, 0, 1      // 頂点11: (-t, 0, 1)
];
```

### プログラム呼び出し階層図

```
IcosahedronGeometry.constructor(radius, detail)
    │
    └─ PolyhedronGeometry.constructor(vertices, indices, radius, detail)
           │
           ├─ subdivide(detail)
           │      └─ subdivideFace()
           │
           ├─ applyRadius(radius)
           │
           └─ generateUVs()
```

### データフロー図

```
[入力]                      [処理]                           [出力]

radius ──────────────────▶ IcosahedronGeometry
detail                        │
                              │ 黄金比計算
                              │ 頂点/インデックス定義
                              ▼
                    PolyhedronGeometry処理
                              │
                              ▼
                    ┌─────────────────────┐
                    │ BufferGeometry      │
                    │  - position属性     │
                    │  - normal属性       │ ────▶ Meshで使用
                    │  - uv属性           │
                    │  - parameters       │
                    └─────────────────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| IcosahedronGeometry.js | `src/geometries/IcosahedronGeometry.js` | ソース | 正20面体ジオメトリクラス |
| PolyhedronGeometry.js | `src/geometries/PolyhedronGeometry.js` | ソース | 多面体ジオメトリ基底クラス |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | ジオメトリ基底クラス |
| Geometries.js | `src/geometries/Geometries.js` | ソース | ジオメトリクラスのエクスポート |
