# 機能設計書 53-MeshPhysicalMaterial

## 概要

本ドキュメントは、Three.jsにおけるMeshPhysicalMaterialクラスの機能設計を定義する。MeshPhysicalMaterialは、MeshStandardMaterialを拡張し、クリアコート、異方性、虹色効果、透過など高度な物理ベースレンダリング機能を提供するマテリアルクラスである。

### 本機能の処理概要

MeshPhysicalMaterialは、MeshStandardMaterialの拡張として、より高度な物理ベースレンダリング（PBR）プロパティを提供する。車の塗装、カーボンファイバー、濡れた表面のようなクリアコート層、ブラッシュドメタルのような異方性反射、シャボン玉や油膜のような虹色効果、ガラスのような物理的に正確な透過表現、布地のようなシーン効果など、複雑なマテリアル表現を実現する。

**業務上の目的・背景**：プロダクトビジュアライゼーション、建築レンダリング、自動車デザインなど、高品質なレンダリングが求められる分野では、標準的なPBRマテリアルでは表現できない複雑な表面特性が必要となる。MeshPhysicalMaterialは、これらの高度なニーズに対応するために設計されている。

**機能の利用シーン**：車の塗装（クリアコート）、ガラスや宝石（透過・分散）、カーボンファイバー（異方性）、布地（シーン）、シャボン玉や油膜（虹色効果）などのリアルな表現が必要な場面で使用される。

**主要な処理内容**：
1. MeshStandardMaterialの全機能を継承
2. クリアコート層の設定と制御
3. 異方性（Anisotropy）反射の設定
4. 虹色効果（Iridescence）の設定
5. 物理ベース透過（Transmission）の設定
6. シーン（Sheen）効果の設定
7. 分散（Dispersion）効果の設定
8. 高度なスペキュラ制御

**関連システム・外部連携**：WebGLRenderer、WebGPURendererと連携。透過マテリアルはレンダリングパイプラインで特別な処理が必要（オフスクリーンレンダリング等）。

**権限による制御**：特になし。マテリアルの設定はすべてのユーザーが実行可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 6 | Sidebar - Material | 補助機能 | 高度なPBRマテリアルの設定 |
| 36 | Material Browser | 主機能 | 物理マテリアルのプレビュー |

## 機能種別

マテリアル設定 / 高度PBRシェーディング処理

## 入力仕様

### 入力パラメータ

MeshStandardMaterialの全パラメータに加え、以下の追加パラメータを持つ：

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| anisotropy | number | No | 異方性強度（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| anisotropyRotation | number | No | 異方性回転角（ラジアン、デフォルト: 0） | 数値 |
| anisotropyMap | Texture | No | 異方性マップ | Texture型またはnull |
| clearcoat | number | No | クリアコート強度（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| clearcoatMap | Texture | No | クリアコートマップ | Texture型またはnull |
| clearcoatRoughness | number | No | クリアコート粗さ（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| clearcoatRoughnessMap | Texture | No | クリアコート粗さマップ | Texture型またはnull |
| clearcoatNormalMap | Texture | No | クリアコート法線マップ | Texture型またはnull |
| clearcoatNormalScale | Vector2 | No | クリアコート法線スケール（デフォルト: (1,1)） | Vector2型 |
| ior | number | No | 屈折率（デフォルト: 1.5、範囲: 1.0-2.333） | 1.0-2.333の数値 |
| reflectivity | number | No | 反射率（デフォルト: 0.5、IORから計算） | 0-1の数値 |
| iridescence | number | No | 虹色効果強度（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| iridescenceMap | Texture | No | 虹色効果マップ | Texture型またはnull |
| iridescenceIOR | number | No | 虹色効果IOR（デフォルト: 1.3、範囲: 1.0-2.333） | 1.0-2.333の数値 |
| iridescenceThicknessRange | Array | No | 虹色層厚み範囲（デフォルト: [100, 400]） | [min, max]の配列 |
| iridescenceThicknessMap | Texture | No | 虹色層厚みマップ（緑チャンネル） | Texture型またはnull |
| sheen | number | No | シーン強度（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| sheenColor | Color | No | シーン色（デフォルト: 0x000000） | Color型 |
| sheenColorMap | Texture | No | シーン色マップ | Texture型またはnull |
| sheenRoughness | number | No | シーン粗さ（デフォルト: 1.0、範囲: 0-1） | 0-1の数値 |
| sheenRoughnessMap | Texture | No | シーン粗さマップ（アルファチャンネル） | Texture型またはnull |
| transmission | number | No | 透過度（デフォルト: 0、範囲: 0-1） | 0-1の数値 |
| transmissionMap | Texture | No | 透過マップ（赤チャンネル） | Texture型またはnull |
| thickness | number | No | 体積厚み（デフォルト: 0） | 0以上の数値 |
| thicknessMap | Texture | No | 厚みマップ（緑チャンネル） | Texture型またはnull |
| attenuationDistance | number | No | 減衰距離（デフォルト: Infinity） | 0より大きい数値 |
| attenuationColor | Color | No | 減衰色（デフォルト: 0xffffff） | Color型 |
| specularIntensity | number | No | スペキュラ強度（デフォルト: 1、範囲: 0-1） | 0-1の数値 |
| specularIntensityMap | Texture | No | スペキュラ強度マップ（アルファチャンネル） | Texture型またはnull |
| specularColor | Color | No | スペキュラ色（デフォルト: 0xffffff） | Color型 |
| specularColorMap | Texture | No | スペキュラ色マップ | Texture型またはnull |
| dispersion | number | No | 分散強度（デフォルト: 0） | 0以上の数値 |

### 入力データソース

コンストラクタへのパラメータオブジェクト、またはプロパティへの直接代入

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isMeshPhysicalMaterial | boolean | 型判定フラグ（常にtrue） |
| type | string | マテリアルタイプ（'MeshPhysicalMaterial'） |
| defines | Object | シェーダー定義（{ 'STANDARD': '', 'PHYSICAL': '' }） |
| マテリアルプロパティ | 各種 | 設定されたマテリアル属性 |

### 出力先

WebGLRenderer/WebGPURendererによるシェーダープログラムへの入力

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 親クラスMeshStandardMaterialのコンストラクタを呼び出し
2. 型フラグの設定
   └─ isMeshPhysicalMaterial = true
3. シェーダー定義の設定
   └─ defines = { 'STANDARD': '', 'PHYSICAL': '' }
4. マテリアルタイプの設定
   └─ type = 'MeshPhysicalMaterial'
5. 高度なPBRプロパティの初期化
   └─ anisotropy, clearcoat, iridescence, sheen, transmission等
6. IORとreflectivityの連動設定
   └─ Object.definePropertyでgetterとsetterを定義
7. パラメータの適用
   └─ setValues(parameters)でパラメータを適用
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new MeshPhysicalMaterial] --> B[super() 呼び出し]
    B --> C[isMeshPhysicalMaterial = true]
    C --> D[defines = STANDARD + PHYSICAL]
    D --> E[type = 'MeshPhysicalMaterial']
    E --> F[異方性プロパティ初期化]
    F --> G[クリアコートプロパティ初期化]
    G --> H[IOR/reflectivity連動設定]
    H --> I[虹色効果プロパティ初期化]
    I --> J[シーンプロパティ初期化]
    J --> K[透過プロパティ初期化]
    K --> L[スペキュラプロパティ初期化]
    L --> M[setValues parameters]
    M --> N[終了: マテリアルインスタンス]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | IOR-reflectivity連動 | reflectivityはIORから計算され、設定するとIORが逆算される | reflectivityまたはIORを設定時 |
| BR-02 | 透過とopacity | transmissionが非ゼロの場合、opacityは1に設定すべき | transmission > 0の場合 |
| BR-03 | 機能有効化 | anisotropy, clearcoat, iridescence, sheen, transmissionが0より大きくなるとバージョンがインクリメント | 各プロパティが0から非0に変化時 |
| BR-04 | dispersion制限 | dispersionは透過オブジェクトでのみ使用可能 | transmission > 0の場合のみ有効 |
| BR-05 | 非金属のみスペキュラ | specularIntensityとspecularColorはmetalness < 1の場合のみ効果がある | metalness < 1の場合 |

### 計算ロジック

IORとreflectivityの関係式：
```
reflectivity = clamp(2.5 * (ior - 1) / (ior + 1), 0, 1)
ior = (1 + 0.4 * reflectivity) / (1 - 0.4 * reflectivity)
```

## データベース操作仕様

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| N/A | Warning | 未定義のプロパティを設定しようとした場合 | コンソールに警告を出力 |
| N/A | Warning | undefinedの値を設定しようとした場合 | コンソールに警告を出力し、設定をスキップ |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- MeshStandardMaterialより計算コストが高い
- 有効化された機能ごとに追加のシェーダー処理が発生
- ほとんどの効果はデフォルトで無効化されており、必要に応じて有効化
- 最良の結果を得るには環境マップを常に指定することが推奨

## セキュリティ考慮事項

特になし

## 備考

- 複雑なシェーディング機能のため、他のマテリアルよりピクセルあたりの計算コストが高い
- 透過（transmission）使用時はオフスクリーンレンダリングが必要になる場合がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、継承関係と高度なPBRプロパティを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Material.js | `src/materials/Material.js` | マテリアルの基底クラス |
| 1-2 | MeshStandardMaterial.js | `src/materials/MeshStandardMaterial.js` | 標準PBRプロパティ |
| 1-3 | MeshPhysicalMaterial.js | `src/materials/MeshPhysicalMaterial.js` | 高度なPBRプロパティ |

**読解のコツ**: MeshPhysicalMaterialはMeshStandardMaterialを継承し、definesに'PHYSICAL'を追加することでシェーダーに高度なPBRモードを指示している。プライベートプロパティ（_anisotropy等）とgetterを使用して、値変更時にバージョンをインクリメントする仕組みに注目。

#### Step 2: エントリーポイントを理解する

マテリアルのインスタンス化と初期化プロセスを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MeshPhysicalMaterial.js | `src/materials/MeshPhysicalMaterial.js` | constructorメソッドがエントリーポイント |

**主要処理フロー**:
1. **43-46行目**: コンストラクタでsuper()を呼び出し
2. **54行目**: isMeshPhysicalMaterialフラグをtrueに設定
3. **56-61行目**: definesに{ 'STANDARD': '', 'PHYSICAL': '' }を設定
4. **63行目**: typeを'MeshPhysicalMaterial'に設定
5. **73行目**: anisotropyRotationを0で初期化
6. **92行目**: clearcoatMapをnullで初期化
7. **100行目**: clearcoatRoughnessを0.0で初期化
8. **134行目**: iorを1.5で初期化
9. **147-158行目**: reflectivityのgetter/setterをObject.definePropertyで定義
10. **315-321行目**: プライベートプロパティ（_anisotropy, _clearcoat等）の初期化
11. **322行目**: setValues(parameters)でパラメータを適用

#### Step 3: プロパティのgetter/setterを理解する

値変更時のバージョン管理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | MeshPhysicalMaterial.js | `src/materials/MeshPhysicalMaterial.js` | anisotropy, clearcoat等のgetter/setter（326-478行目） |

**主要処理フロー**:
- **332-348行目**: anisotropyのgetter/setter。値が0から非0に変化するとversion++
- **358-374行目**: clearcoatのgetter/setter。同様の仕組み
- **382-398行目**: iridescenceのgetter/setter

### プログラム呼び出し階層図

```
MeshPhysicalMaterial (constructor)
    │
    ├─ MeshStandardMaterial (super())
    │      └─ Material (super())
    │             └─ EventDispatcher (super())
    │
    ├─ Vector2 (new Vector2())
    │      └─ clearcoatNormalScale の初期化
    │
    ├─ Color (new Color())
    │      └─ sheenColor, attenuationColor, specularColor の初期化
    │
    ├─ Object.defineProperty
    │      └─ reflectivity の getter/setter 定義
    │
    └─ setValues (parameters)
           └─ Material.setValues (inherited)
```

### データフロー図

```
[入力]                         [処理]                           [出力]

parameters ─────────────▶ constructor() ─────────▶ MeshPhysicalMaterial
  (MeshStandardMaterial      - super()呼び出し        インスタンス
   の全パラメータ +          - defines設定            (高度なPBRシェーダー
   clearcoat, iridescence,   - 高度PBRプロパティ        で使用される
   transmission, sheen,        初期化                   マテリアル設定)
   anisotropy, dispersion,   - setValues()適用
   specular関連, etc.)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Material.js | `src/materials/Material.js` | ソース | マテリアル基底クラス |
| MeshStandardMaterial.js | `src/materials/MeshStandardMaterial.js` | ソース | PBR標準マテリアル |
| MeshPhysicalMaterial.js | `src/materials/MeshPhysicalMaterial.js` | ソース | 高度なPBRマテリアル実装 |
| Color.js | `src/math/Color.js` | ソース | 色データ管理 |
| Vector2.js | `src/math/Vector2.js` | ソース | 2次元ベクトル |
| MathUtils.js | `src/math/MathUtils.js` | ソース | clamp関数（IOR計算用） |
| constants.js | `src/constants.js` | ソース | 定数定義 |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベント管理 |
