# 機能設計書 56-MeshMatcapMaterial

## 概要

本ドキュメントは、Three.jsにおけるMeshMatcapMaterialクラスの機能設計を定義する。MeshMatcapMaterialは、MatCap（Material Capture、Lit Sphere）テクスチャを使用してマテリアルの色とシェーディングをエンコードするマテリアルクラスである。

### 本機能の処理概要

MeshMatcapMaterialは、MatCapテクスチャを使用してマテリアルのライティングと色を事前にベイクした見た目を実現するマテリアルである。MatCapテクスチャには、球体に適用された場合のマテリアルの見た目が画像としてエンコードされており、実行時のライティング計算なしに複雑なシェーディング効果を表現できる。

**業務上の目的・背景**：リアルタイムのライティング計算を行わずに、スタイライズドな見た目や複雑なシェーディング効果を実現したい場面がある。MatCapテクスチャを使用することで、事前にベイクされたライティング効果を高速に表示でき、アーティスティックな表現やパフォーマンス最適化に有効である。

**機能の利用シーン**：スカルプチャ表示（ZBrush風のプレビュー）、アーティスティックなレンダリング、パフォーマンス重視のシーン、事前計算されたライティングを使用したい場面で使用される。

**主要な処理内容**：
1. マテリアルの初期化とパラメータ設定
2. MatCapテクスチャによるシェーディング
3. 拡散反射色（diffuse color）による変調
4. 各種テクスチャマップ（カラーマップ、法線マップ、バンプマップ等）の適用
5. アルファマップによる透明度制御
6. ワイヤーフレームモードとフラットシェーディングのサポート

**関連システム・外部連携**：WebGLRenderer、WebGPURenderer、SVGRendererと連携。ライトへの応答なし（matcap画像がライティングをベイク）。シャドウのキャストは可能だが、セルフシャドウや受影は不可。

**権限による制御**：特になし。マテリアルの設定はすべてのユーザーが実行可能。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | （画面機能マッピングに登録なし） |

## 機能種別

マテリアル設定 / Matcapシェーディング処理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | Color | No | 拡散反射色（デフォルト: 0xffffff） | Color型として有効な値 |
| matcap | Texture | No | MatCapテクスチャ | Texture型またはnull |
| map | Texture | No | カラーマップ | Texture型またはnull |
| bumpMap | Texture | No | バンプマップ | Texture型またはnull |
| bumpScale | number | No | バンプスケール（デフォルト: 1） | 数値 |
| normalMap | Texture | No | 法線マップ | Texture型またはnull |
| normalMapType | constant | No | 法線マップタイプ（デフォルト: TangentSpaceNormalMap） | TangentSpaceNormalMapまたはObjectSpaceNormalMap |
| normalScale | Vector2 | No | 法線マップスケール（デフォルト: (1,1)） | Vector2型 |
| displacementMap | Texture | No | ディスプレイスメントマップ | Texture型またはnull |
| displacementScale | number | No | ディスプレイスメントスケール（デフォルト: 1） | 数値 |
| displacementBias | number | No | ディスプレイスメントバイアス（デフォルト: 0） | 数値 |
| alphaMap | Texture | No | アルファマップ | Texture型またはnull |
| wireframe | boolean | No | ワイヤーフレーム描画（デフォルト: false） | true/false |
| wireframeLinewidth | number | No | ワイヤーフレーム線幅（デフォルト: 1） | 1以上の数値 |
| flatShading | boolean | No | フラットシェーディング（デフォルト: false） | true/false |
| fog | boolean | No | フォグ影響（デフォルト: true） | true/false |

### 入力データソース

コンストラクタへのパラメータオブジェクト、またはプロパティへの直接代入

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isMeshMatcapMaterial | boolean | 型判定フラグ（常にtrue） |
| type | string | マテリアルタイプ（'MeshMatcapMaterial'） |
| defines | Object | シェーダー定義（{ 'MATCAP': '' }） |
| マテリアルプロパティ | 各種 | 設定されたマテリアル属性 |

### 出力先

WebGLRenderer/WebGPURendererによるシェーダープログラムへの入力

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 親クラスMaterialのコンストラクタを呼び出し
2. 型フラグの設定
   └─ isMeshMatcapMaterial = true
3. シェーダー定義の設定
   └─ defines = { 'MATCAP': '' }
4. マテリアルタイプの設定
   └─ type = 'MeshMatcapMaterial'
5. デフォルトプロパティの初期化
   └─ color, matcap等の初期値設定
6. テクスチャマップの初期化
   └─ map, normalMap, bumpMap等をnullで初期化
7. パラメータの適用
   └─ setValues(parameters)でパラメータを適用
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new MeshMatcapMaterial] --> B[super() 呼び出し]
    B --> C[isMeshMatcapMaterial = true]
    C --> D[defines = MATCAP]
    D --> E[type = 'MeshMatcapMaterial']
    E --> F[color初期化 0xffffff]
    F --> G[matcap初期化 null]
    G --> H[テクスチャマップ初期化]
    H --> I[wireframe/flatShading初期化]
    I --> J[setValues parameters]
    J --> K[終了: マテリアルインスタンス]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ライト非対応 | MatCapマテリアルはシーンのライトに反応しない | 常時 |
| BR-02 | シャドウキャスト可 | 影を他のオブジェクトにキャストできる（shadow clipping有効） | castShadow=trueの場合 |
| BR-03 | セルフシャドウ不可 | 自己影や受影はできない | 常時 |
| BR-04 | 法線マップ優先 | normalMapが設定されている場合、bumpMapは無視される | normalMapがnull以外の場合 |
| BR-05 | matcap画像形式 | matcapテクスチャは球体のシェーディングを含む正方形画像であるべき | matcapを使用する場合 |

### 計算ロジック

MatCapルックアップ:
- ビュー空間法線のxy成分を使用してmatcapテクスチャをサンプリング
- UV = (viewNormal.xy + 1.0) / 2.0
- 色 = matcapTexture.sample(UV) * color

## データベース操作仕様

該当なし（本機能はデータベース操作を行わない）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| N/A | Warning | 未定義のプロパティを設定しようとした場合 | コンソールに警告を出力 |
| N/A | Warning | undefinedの値を設定しようとした場合 | コンソールに警告を出力し、設定をスキップ |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ライティング計算不要のため高速
- matcapテクスチャのサンプリングのみ

## セキュリティ考慮事項

特になし

## 備考

- SVGRendererを使用する場合のみwireframeLinewidthが有効
- matcapテクスチャは通常512x512〜1024x1024ピクセルの解像度を使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、マテリアルの基本構造とMatcap固有のプロパティを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Material.js | `src/materials/Material.js` | マテリアルの基底クラス |
| 1-2 | MeshMatcapMaterial.js | `src/materials/MeshMatcapMaterial.js` | Matcap固有のプロパティ（matcap等） |

**読解のコツ**: MeshMatcapMaterialはMaterialを直接継承し、definesに'MATCAP'を設定することでシェーダーにMatcapモードを指示している。matcapプロパティがこのマテリアルの核心である。

#### Step 2: エントリーポイントを理解する

マテリアルのインスタンス化と初期化プロセスを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | MeshMatcapMaterial.js | `src/materials/MeshMatcapMaterial.js` | constructorメソッドがエントリーポイント |

**主要処理フロー**:
1. **29-31行目**: コンストラクタでsuper()を呼び出し
2. **40行目**: isMeshMatcapMaterialフラグをtrueに設定
3. **42行目**: definesに{ 'MATCAP': '' }を設定
4. **44行目**: typeを'MeshMatcapMaterial'に設定
5. **52行目**: colorを白(0xffffff)で初期化
6. **60行目**: matcapをnullで初期化
7. **70行目**: mapをnullで初期化
8. **81行目**: bumpMapをnullで初期化
9. **89行目**: bumpScaleを1で初期化
10. **102行目**: normalMapをnullで初期化
11. **110行目**: normalMapTypeをTangentSpaceNormalMapで初期化
12. **118行目**: normalScaleを(1,1)で初期化
13. **131行目**: displacementMapをnullで初期化
14. **141行目**: displacementScaleを1で初期化
15. **151行目**: displacementBiasを0で初期化
16. **166行目**: alphaMapをnullで初期化
17. **174行目**: wireframeをfalseで初期化
18. **184行目**: wireframeLinewidthを1で初期化
19. **192行目**: flatShadingをfalseで初期化
20. **200行目**: fogをtrueで初期化
21. **202行目**: setValues(parameters)でパラメータを適用

#### Step 3: copyメソッドを理解する

マテリアルのクローン作成時の動作を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | MeshMatcapMaterial.js | `src/materials/MeshMatcapMaterial.js` | copyメソッド（207-241行目） |

**主要処理フロー**:
- **209行目**: 親クラスのcopyを呼び出し
- **211行目**: definesを再設定
- **213行目**: colorをコピー
- **215行目**: matcapをコピー
- **217-237行目**: その他のプロパティをコピー

### プログラム呼び出し階層図

```
MeshMatcapMaterial (constructor)
    │
    ├─ Material (super())
    │      └─ EventDispatcher (super())
    │
    ├─ Color (new Color())
    │      └─ color の初期化
    │
    ├─ Vector2 (new Vector2())
    │      └─ normalScale の初期化
    │
    └─ setValues (parameters)
           └─ Material.setValues (inherited)
```

### データフロー図

```
[入力]                      [処理]                         [出力]

parameters ────────▶ constructor() ────────▶ MeshMatcapMaterial
  (color, matcap,       - super()呼び出し       インスタンス
   map, normalMap,      - defines設定           (Matcapシェーダー
   etc.)                - プロパティ初期化        で使用される
                        - setValues()適用         マテリアル設定)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Material.js | `src/materials/Material.js` | ソース | マテリアル基底クラス |
| MeshMatcapMaterial.js | `src/materials/MeshMatcapMaterial.js` | ソース | Matcapマテリアル実装 |
| Color.js | `src/math/Color.js` | ソース | 色データ管理 |
| Vector2.js | `src/math/Vector2.js` | ソース | 2次元ベクトル（normalScale用） |
| constants.js | `src/constants.js` | ソース | 定数定義（TangentSpaceNormalMap等） |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベント管理 |
