# 機能設計書 61-PointsMaterial

## 概要

本ドキュメントは、Three.jsライブラリにおける点群描画用マテリアル「PointsMaterial」の機能設計について記述する。PointsMaterialは、Pointsオブジェクトと組み合わせて使用され、3D空間上に大量の点（パーティクル）を効率的に描画するための外観定義を提供する。

### 本機能の処理概要

PointsMaterialは、3Dシーン内で点群（パーティクルシステム）を描画する際の見た目を定義するマテリアルクラスである。粒子効果、星空、データ可視化、点群データの表示など、多様な用途に対応する。

**業務上の目的・背景**：3Dグラフィックスにおいて、大量の点を効率的に描画するニーズは非常に多い。パーティクルエフェクト（火花、雪、煙など）、科学的データの3D可視化、LiDARなどから取得した点群データの表示、星空や宇宙空間の表現など、点群描画は3Dアプリケーションの基本的な表現技術である。PointsMaterialは、これらの点群描画に必要な視覚的プロパティ（色、サイズ、テクスチャ、透明度など）を一元的に管理し、WebGLを通じて高速にレンダリングするための基盤を提供する。

**機能の利用シーン**：
- パーティクルエフェクト（火花、雪、雨、煙、爆発など）の作成
- 科学的データセットの3D可視化（散布図、分子構造など）
- 点群データ（LiDAR、3Dスキャンデータ）の表示
- 星空、銀河、宇宙空間の表現
- データポイントのインタラクティブな表示

**主要な処理内容**：
1. マテリアルプロパティの初期化（色、サイズ、テクスチャマップなど）
2. パラメータオブジェクトからのプロパティ設定（setValues）
3. マテリアルの複製（copy/clone）
4. レンダラーへの描画パラメータ提供

**関連システム・外部連携**：WebGLRenderer/WebGPURendererと連携してGPUへの描画命令を発行する。Pointsオブジェクトと組み合わせて使用される。

**権限による制御**：特になし（クライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに直接の関連画面なし |

## 機能種別

データ定義 / 外観設定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number/Color/string | No | 点の色（デフォルト: 0xffffff） | Color.setで受け付ける形式 |
| map | Texture | No | カラーマップテクスチャ | Textureインスタンス |
| alphaMap | Texture | No | アルファマップ（透明度制御用グレースケールテクスチャ） | Textureインスタンス |
| size | number | No | 点のサイズ（ピクセル単位、デフォルト: 1） | 正の数値 |
| sizeAttenuation | boolean | No | カメラ距離による点サイズの減衰（デフォルト: true） | boolean |
| fog | boolean | No | フォグの影響を受けるか（デフォルト: true） | boolean |

### 入力データソース

コンストラクタのparametersオブジェクトから入力。Material基底クラスのプロパティ（opacity、transparent、blending等）も継承して使用可能。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isPointsMaterial | boolean | 型判定用フラグ（常にtrue） |
| type | string | マテリアルタイプ識別子（'PointsMaterial'） |
| color | Color | マテリアルの色 |
| map | Texture/null | カラーマップテクスチャ |
| alphaMap | Texture/null | アルファマップテクスチャ |
| size | number | 点のサイズ |
| sizeAttenuation | boolean | サイズ減衰フラグ |
| fog | boolean | フォグ影響フラグ |

### 出力先

レンダラー（WebGLRenderer/WebGPURenderer）に描画パラメータとして提供される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ Material基底クラスのコンストラクタ実行
2. デフォルトプロパティの初期化
   └─ isPointsMaterial, type, color, map, alphaMap, size, sizeAttenuation, fog
3. setValues(parameters)でパラメータ適用
   └─ Material基底クラスのsetValuesメソッドで各プロパティを設定
4. レンダリング時にレンダラーがプロパティを参照
   └─ シェーダーユニフォームとして各プロパティが渡される
```

### フローチャート

```mermaid
flowchart TD
    A[new PointsMaterial] --> B[super - Material初期化]
    B --> C[デフォルト値設定]
    C --> D{parameters存在?}
    D -->|Yes| E[setValues実行]
    D -->|No| F[初期化完了]
    E --> F
    F --> G[レンダラーに提供]
    G --> H[GPUシェーダーでレンダリング]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-61-1 | サイズ制限 | sizeプロパティはハードウェア制限（gl.ALIASED_POINT_SIZE_RANGE）を超える場合、クランプされる | 常時 |
| BR-61-2 | サイズ減衰 | sizeAttenuationがtrueの場合、透視投影カメラでのみ距離による減衰が適用される | PerspectiveCamera使用時 |
| BR-61-3 | アルファマップ | alphaMapはグレースケール画像として扱われ、緑チャンネルが使用される（DXT圧縮互換性のため） | alphaMap指定時 |

### 計算ロジック

サイズ減衰の計算（シェーダー内）:
```
実際のサイズ = size * (1.0 / distance)
```
※distanceはカメラからの距離

## データベース操作仕様

該当なし（クライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 警告 | 未定義のパラメータを渡した場合 | コンソールに警告を出力 |
| - | 警告 | undefinedの値を持つパラメータを渡した場合 | コンソールに警告を出力、スキップ |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 点群の数に応じたGPUメモリ消費
- sizeAttenuationを無効にすることでシェーダー計算を軽減可能
- テクスチャマップ使用時はVRAM消費増加

## セキュリティ考慮事項

特になし（クライアントサイドライブラリ）

## 備考

- Three.js r169以降で動作確認
- WebGL2環境を推奨（gl.POINTS対応）

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、PointsMaterialが継承するMaterialクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Material.js | `src/materials/Material.js` | 基底クラスのプロパティ（opacity、transparent、blending等）とsetValuesメソッドの実装 |
| 1-2 | Color.js | `src/math/Color.js` | color プロパティで使用されるColorクラスの構造 |

**読解のコツ**: Material.jsの17-477行目でマテリアルの共通プロパティが定義されている。特にsetValues（555-594行目）メソッドがパラメータ適用の核心部分。

#### Step 2: エントリーポイントを理解する

PointsMaterialクラス自体の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | PointsMaterial.js | `src/materials/PointsMaterial.js` | クラス全体（140行程度の簡潔な実装） |

**主要処理フロー**:
1. **29-39行目**: コンストラクタ開始、isPointsMaterialフラグ設定
2. **51-61行目**: colorプロパティ初期化（デフォルト白色）
3. **63-86行目**: map、alphaMapプロパティ（テクスチャ）
4. **88-104行目**: size、sizeAttenuationプロパティ
5. **106-112行目**: fogプロパティ
6. **114行目**: setValuesでパラメータ適用
7. **118-135行目**: copyメソッドの実装

#### Step 3: 使用されるオブジェクトを理解する

PointsMaterialが適用されるPointsオブジェクトを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Points.js | `src/objects/Points.js` | PointsMaterialを使用するオブジェクトの実装 |

### プログラム呼び出し階層図

```
PointsMaterial (継承: Material)
    │
    ├─ constructor(parameters)
    │      │
    │      ├─ super() → Material.constructor()
    │      │                 └─ EventDispatcher()
    │      │
    │      └─ setValues(parameters)
    │               └─ Material.setValues()
    │
    ├─ copy(source)
    │      └─ super.copy(source) → Material.copy()
    │
    └─ [レンダリング時]
           └─ WebGLRenderer.renderPoints()
                   └─ シェーダーへユニフォーム送信
```

### データフロー図

```
[入力]                    [処理]                         [出力]

parameters ───────▶ PointsMaterial.constructor()
{                           │
  color: 0xff0000,          ├─ プロパティ初期化
  size: 2,                  │
  map: texture              └─ setValues() ───────▶ マテリアルインスタンス
}                                                        │
                                                         │
BufferGeometry ─────────────────────────────────────────┼──▶ Points
{vertices}                                               │    オブジェクト
                                                         │
                                                         ▼
                                              WebGLRenderer.render()
                                                         │
                                                         ▼
                                              GPU シェーダー描画
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| PointsMaterial.js | `src/materials/PointsMaterial.js` | ソース | PointsMaterialクラス定義 |
| Material.js | `src/materials/Material.js` | ソース | 基底マテリアルクラス |
| Color.js | `src/math/Color.js` | ソース | 色管理クラス |
| Points.js | `src/objects/Points.js` | ソース | 点群オブジェクトクラス |
| WebGLRenderer.js | `src/renderers/WebGLRenderer.js` | ソース | WebGLレンダラー |
| points.glsl.js | `src/renderers/shaders/ShaderLib/points.glsl.js` | シェーダー | 点群描画用シェーダー |
