# 機能設計書 64-RawShaderMaterial

## 概要

本ドキュメントは、Three.jsライブラリにおけるRawシェーダー用マテリアル「RawShaderMaterial」の機能設計について記述する。RawShaderMaterialは、ShaderMaterialを継承し、Three.jsの組み込みシェーダーコード（属性、ユニフォーム定義）を自動追加しない純粋なカスタムシェーダー実装を可能にする。

### 本機能の処理概要

RawShaderMaterialは、ShaderMaterialの機能を継承しつつ、Three.jsが通常自動挿入する組み込み属性・ユニフォーム定義をシェーダーコードに追加しないマテリアルクラスである。開発者は完全に自前のGLSLコードを記述する必要があるが、Three.jsの内部実装に依存しない完全な制御が可能になる。

**業務上の目的・背景**：ShaderMaterialでは、Three.jsが利便性のために組み込み属性（position、normal、uv等）やユニフォーム（modelMatrix、viewMatrix等）の定義を自動的にシェーダーコードに追加する。しかし、GLSL 3.0（WebGL2）の新機能を使用する場合や、シェーダーコードを完全に制御したい場合、この自動追加が問題になることがある。RawShaderMaterialは、シェーダーコードへの自動追加を行わないことで、開発者に完全なシェーダー制御を提供する。

**機能の利用シーン**：
- GLSL ES 3.0（WebGL2）の新機能を使用するシェーダー
- 外部ツールで生成されたシェーダーコードの直接使用
- シェーダーコードの最適化（不要な定義の削除）
- 既存のGLSLライブラリとの統合
- 教育目的でのGLSL学習

**主要な処理内容**：
1. ShaderMaterialの全機能を継承
2. Three.jsの組み込み定義を自動追加しない設定
3. 完全なカスタムシェーダーコードの実行

**関連システム・外部連携**：WebGLRendererと連携。WebGPURendererでは使用不可。

**権限による制御**：特になし（クライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 17 | Script Editor | 主機能 | Rawシェーダーの編集 |

## 機能種別

データ定義 / カスタムシェーダー管理

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| uniforms | Object | No | シェーダーユニフォーム変数 | オブジェクト |
| uniformsGroups | Array | No | UBO用ユニフォームグループ配列 | UniformsGroup配列 |
| defines | Object | No | #defineディレクティブ用キー/値ペア | オブジェクト |
| vertexShader | string | Yes* | 頂点シェーダーGLSLコード（全定義を含む） | 文字列 |
| fragmentShader | string | Yes* | フラグメントシェーダーGLSLコード（全定義を含む） | 文字列 |
| linewidth | number | No | 線の太さ（デフォルト: 1） | 正の数値 |
| wireframe | boolean | No | ワイヤーフレーム表示 | boolean |
| fog | boolean | No | フォグユニフォーム有効化 | boolean |
| lights | boolean | No | ライティングユニフォーム有効化 | boolean |
| clipping | boolean | No | クリッピングプレーンサポート | boolean |
| glslVersion | string | No | GLSLバージョン（GLSL1/GLSL3） | GLSL1/GLSL3/null |

*注: vertexShader/fragmentShaderはデフォルト値があるが、RawShaderMaterialでは実用上必須

### 入力データソース

コンストラクタのparametersオブジェクトから入力。シェーダーコードには全ての属性・ユニフォーム定義を含める必要がある。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isRawShaderMaterial | boolean | 型判定用フラグ（常にtrue） |
| type | string | マテリアルタイプ識別子（'RawShaderMaterial'） |
| （ShaderMaterialの全プロパティ） | - | ShaderMaterialから継承 |

### 出力先

WebGLRendererに描画パラメータとシェーダーコードとして提供される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ super(parameters) - ShaderMaterial.constructor実行
2. isRawShaderMaterialフラグ設定
3. type = 'RawShaderMaterial' 設定
4. レンダリング時の処理:
   a. WebGLRendererがシェーダーをコンパイル
      - 組み込み定義の自動追加なし
   b. uniformsの値がGPUに転送
   c. メッシュがカスタムシェーダーで描画
```

### フローチャート

```mermaid
flowchart TD
    A[new RawShaderMaterial] --> B[super - ShaderMaterial初期化]
    B --> C[isRawShaderMaterial = true]
    C --> D[type = 'RawShaderMaterial']
    D --> E[初期化完了]
    E --> F[レンダラーに提供]
    F --> G{シェーダーコンパイル}
    G -->|組み込み定義追加なし| H[ユニフォーム転送]
    H --> I[GPUで描画実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-64-1 | 組み込み定義なし | Three.jsの組み込み属性・ユニフォーム定義が自動追加されない | 常時 |
| BR-64-2 | 完全なコード記述 | position、normal等の属性やmodelMatrix等のユニフォームを開発者が明示的に定義する必要がある | 常時 |
| BR-64-3 | WebGL専用 | RawShaderMaterialはWebGLRendererでのみ使用可能 | 常時 |
| BR-64-4 | GLSL3対応 | glslVersion=GLSL3でWebGL2のGLSL ES 3.0機能が使用可能 | WebGL2環境 |

### 計算ロジック

RawShaderMaterialは機能的にはShaderMaterialと同じだが、以下のコードが自動挿入されない:
```glsl
// ShaderMaterialで自動追加されるが、RawShaderMaterialでは追加されない
uniform mat4 modelMatrix;
uniform mat4 modelViewMatrix;
uniform mat4 projectionMatrix;
uniform mat4 viewMatrix;
uniform mat3 normalMatrix;
uniform vec3 cameraPosition;

attribute vec3 position;
attribute vec3 normal;
attribute vec2 uv;
```

## データベース操作仕様

該当なし（クライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| GL_COMPILE_ERROR | シェーダーコンパイルエラー | 属性/ユニフォーム定義不足、GLSLシンタックスエラー | 必要な定義を追加 |
| GL_LINK_ERROR | シェーダーリンクエラー | vertex/fragmentシェーダー間の不整合 | varying/in-out変数を確認 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- ShaderMaterialと同等
- 不要な定義を省略できるため、若干のシェーダーサイズ削減が可能

## セキュリティ考慮事項

- ShaderMaterialと同様

## 備考

- ShaderMaterialとの主な違いは組み込み定義の自動追加の有無のみ
- 43行という非常に簡潔な実装
- WebGL2/GLSL 3.0を使用する場合に特に有用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 親クラスを理解する

RawShaderMaterialはShaderMaterialを継承しているため、まず親クラスを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ShaderMaterial.js | `src/materials/ShaderMaterial.js` | 親クラスの全機能 |
| 1-2 | Material.js | `src/materials/Material.js` | 基底クラスの構造 |

**読解のコツ**: RawShaderMaterialは非常に短いコードなので、ShaderMaterialの理解が本質的に重要。

#### Step 2: エントリーポイントを理解する

RawShaderMaterialクラス自体の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | RawShaderMaterial.js | `src/materials/RawShaderMaterial.js` | クラス全体（43行の非常に簡潔な実装） |

**主要処理フロー**:
1. **1行目**: ShaderMaterialをインポート
2. **12行目**: `class RawShaderMaterial extends ShaderMaterial` - 継承宣言
3. **23行目**: `super( parameters )` - 親クラスコンストラクタ呼び出し（全パラメータを委譲）
4. **34行目**: `this.isRawShaderMaterial = true` - 型判定フラグ
5. **36行目**: `this.type = 'RawShaderMaterial'` - タイプ識別子

#### Step 3: WebGLRendererでの違いを理解する

レンダラー側でのShaderMaterialとRawShaderMaterialの扱いの違いを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | WebGLProgram.js | `src/renderers/webgl/WebGLProgram.js` | isRawShaderMaterial判定によるコード追加制御 |

### プログラム呼び出し階層図

```
RawShaderMaterial (継承: ShaderMaterial → Material)
    │
    └─ constructor(parameters)
           │
           └─ super(parameters)
                   │
                   └─ ShaderMaterial.constructor(parameters)
                            │
                            ├─ super() → Material.constructor()
                            │
                            ├─ デフォルトプロパティ設定
                            │
                            └─ setValues(parameters)
           │
           ├─ this.isRawShaderMaterial = true
           │
           └─ this.type = 'RawShaderMaterial'

[レンダリング時]
    │
    └─ WebGLProgram
            │
            └─ if (!material.isRawShaderMaterial) {
                    // 組み込み定義を追加
               }
               // RawShaderMaterialの場合は追加しない
```

### データフロー図

```
[入力]                              [処理]                         [出力]

完全なシェーダーコード ──────▶ RawShaderMaterial.constructor()
{                                       │
  vertexShader: `                       │
    #version 300 es                     │
    in vec3 position;                   ├─ super(parameters)
    uniform mat4 mvp;                   │     └─ ShaderMaterial初期化
    void main() {                       │
      gl_Position = mvp * vec4(...);    ├─ isRawShaderMaterial = true
    }                                   │
  `,                                    │
  fragmentShader: `...`                 └─ type設定
}                                                │
                                                 ▼
                                    マテリアルインスタンス
                                                 │
                                                 ▼
                                        WebGLProgram
                                    (組み込み定義追加なし)
                                                 │
                                                 ▼
                                    シェーダーコンパイル
                                    (ユーザーコードのみ)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| RawShaderMaterial.js | `src/materials/RawShaderMaterial.js` | ソース | RawShaderMaterialクラス定義（43行） |
| ShaderMaterial.js | `src/materials/ShaderMaterial.js` | ソース | 親クラス（ShaderMaterial） |
| Material.js | `src/materials/Material.js` | ソース | 基底マテリアルクラス |
| WebGLProgram.js | `src/renderers/webgl/WebGLProgram.js` | ソース | シェーダーコンパイル処理 |
