# 機能設計書 66-Light

## 概要

本ドキュメントは、Three.jsライブラリにおける光源の基底クラス「Light」の機能設計について記述する。Lightクラスは、すべての光源タイプ（AmbientLight、DirectionalLight、PointLight、SpotLight等）の共通機能を提供する抽象基底クラスである。

### 本機能の処理概要

Lightは、3Dシーンを照らす光源の基本的な属性（色、強度）と動作を定義する抽象基底クラスである。Object3Dを継承しているため、3D空間内での位置決めや親子関係の構築が可能である。

**業務上の目的・背景**：3Dグラフィックスにおいて、リアルな照明表現は視覚的品質の要である。太陽光、室内照明、街灯、スポットライトなど、様々な光源タイプが存在するが、それらには「色」と「強度」という共通の属性がある。Lightクラスは、これらの共通属性とObject3Dから継承される位置・回転・スケールの管理機能を統合し、すべての光源タイプの基盤を提供する。これにより、一貫したAPIでの光源操作と、コードの再利用性が実現される。

**機能の利用シーン**：
- 全光源タイプの基盤として使用
- 光源の色と強度の統一的な管理
- シーングラフ内での光源の位置管理
- 光源のイベント管理（dispose等）

**主要な処理内容**：
1. 光源の色（Color）の管理
2. 光源の強度（intensity）の管理
3. Object3Dの機能継承（位置、回転、スケール、親子関係）
4. リソース解放（dispose）イベントの発行
5. JSON形式へのシリアライズ

**関連システム・外部連携**：WebGLRenderer/WebGPURendererと連携してシェーダーにライティングデータを提供。

**権限による制御**：特になし（クライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 基底クラスのため直接の関連画面なし（派生クラスで使用） |

## 機能種別

データ定義 / 基底クラス

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number/Color/string | No | 光の色（デフォルト: 0xffffff） | Color.setで受け付ける形式 |
| intensity | number | No | 光の強度（デフォルト: 1） | 数値 |

### 入力データソース

コンストラクタの引数から直接入力。派生クラスから呼び出される。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isLight | boolean | 型判定用フラグ（常にtrue） |
| type | string | オブジェクトタイプ識別子（'Light'） |
| color | Color | 光の色 |
| intensity | number | 光の強度 |
| （Object3Dの全プロパティ） | - | position, rotation, scale, parent, children等 |

### 出力先

派生クラスを通じてレンダラーに光源情報として提供される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ super() - Object3Dのコンストラクタ実行
2. isLightフラグ設定
3. type設定
4. colorプロパティ初期化
5. intensityプロパティ初期化
6. 派生クラスで追加プロパティ設定
7. レンダリング時にシェーダーへライティングデータ提供
```

### フローチャート

```mermaid
flowchart TD
    A[new Light] --> B[super - Object3D初期化]
    B --> C[isLight = true]
    C --> D[type = 'Light']
    D --> E[color初期化]
    E --> F[intensity初期化]
    F --> G[初期化完了]
    G --> H{派生クラス?}
    H -->|Yes| I[追加プロパティ設定]
    H -->|No| J[基底クラスとして使用]
    I --> K[シーンに追加]
    J --> K
    K --> L[レンダリング時にライティング計算]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-66-1 | 抽象基底クラス | Lightは直接使用せず、派生クラス（AmbientLight等）を使用する | 常時 |
| BR-66-2 | Object3D継承 | Object3Dの全機能（位置、回転、スケール、親子関係）が利用可能 | 常時 |
| BR-66-3 | disposeイベント | dispose()呼び出し時に'dispose'イベントを発行する | dispose時 |
| BR-66-4 | 強度単位 | intensityの単位は派生クラスによって異なる（candela、lux等） | 派生クラス依存 |

### 計算ロジック

ライティング計算はシェーダー内で行われるが、基本的な計算式は:
```
最終色 = 物体の色 * 光の色 * 光の強度
```

## データベース操作仕様

該当なし（クライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 特になし | - |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 光源数が増えるとシェーダー計算負荷が増加
- ライト数の制限はレンダラー設定に依存

## セキュリティ考慮事項

特になし（クライアントサイドライブラリ）

## 備考

- 86行という簡潔な実装
- 派生クラス: AmbientLight, DirectionalLight, PointLight, SpotLight, HemisphereLight, RectAreaLight, LightProbe

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 親クラスを理解する

Lightが継承するObject3Dクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Object3D.js | `src/core/Object3D.js` | 親クラスの機能（位置、回転、スケール、親子関係） |
| 1-2 | Color.js | `src/math/Color.js` | colorプロパティで使用されるColorクラス |

**読解のコツ**: Object3D.jsは1670行の大きなファイルだが、position、rotation、quaternion、scaleプロパティ（140-206行目）とadd/remove/traverseメソッドを中心に理解する。

#### Step 2: エントリーポイントを理解する

Lightクラス自体の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Light.js | `src/lights/Light.js` | クラス全体（86行の簡潔な実装） |

**主要処理フロー**:
1. **1-2行目**: インポート（Object3D、Color）
2. **11行目**: `class Light extends Object3D` - Object3D継承
3. **18-19行目**: コンストラクタ引数（color, intensity=1）
4. **21行目**: `super()` - Object3Dコンストラクタ呼び出し
5. **30行目**: `this.isLight = true` - 型判定フラグ
6. **32行目**: `this.type = 'Light'` - タイプ識別子
7. **39行目**: `this.color = new Color( color )` - 色初期化
8. **47行目**: `this.intensity = intensity` - 強度初期化
9. **55-58行目**: dispose()メソッド - イベント発行
10. **61-69行目**: copy()メソッド - 色と強度のコピー
11. **72-81行目**: toJSON()メソッド - シリアライズ

#### Step 3: 派生クラスを確認する

Lightを継承する主要な派生クラスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AmbientLight.js | `src/lights/AmbientLight.js` | 最もシンプルな派生クラス |
| 3-2 | DirectionalLight.js | `src/lights/DirectionalLight.js` | target、shadowを追加する派生クラス |
| 3-3 | PointLight.js | `src/lights/PointLight.js` | distance、decay、powerを追加する派生クラス |

### プログラム呼び出し階層図

```
Light (継承: Object3D → EventDispatcher)
    │
    ├─ constructor(color, intensity)
    │      │
    │      ├─ super() → Object3D.constructor()
    │      │                 └─ EventDispatcher()
    │      │
    │      ├─ this.isLight = true
    │      ├─ this.type = 'Light'
    │      ├─ this.color = new Color(color)
    │      └─ this.intensity = intensity
    │
    ├─ dispose()
    │      └─ this.dispatchEvent({ type: 'dispose' })
    │
    ├─ copy(source, recursive)
    │      │
    │      ├─ super.copy(source, recursive) → Object3D.copy()
    │      ├─ this.color.copy(source.color)
    │      └─ this.intensity = source.intensity
    │
    └─ toJSON(meta)
           │
           ├─ super.toJSON(meta) → Object3D.toJSON()
           ├─ data.object.color = this.color.getHex()
           └─ data.object.intensity = this.intensity
```

### データフロー図

```
[入力]                    [処理]                         [出力]

color ─────────────▶ Light.constructor()
intensity                  │
                           ├─ Object3D初期化
                           ├─ isLight設定
                           ├─ type設定
                           ├─ color初期化
                           └─ intensity設定
                                         │
                                         ▼
                              Lightインスタンス
                                         │
                 ┌───────────────────────┼───────────────────────┐
                 ▼                       ▼                       ▼
          AmbientLight            DirectionalLight          PointLight
          (環境光)                 (平行光源)               (点光源)
                 │                       │                       │
                 └───────────────────────┼───────────────────────┘
                                         ▼
                                  Scene.add(light)
                                         │
                                         ▼
                               WebGLRenderer.render()
                                         │
                                         ▼
                              シェーダーライティング計算
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Light.js | `src/lights/Light.js` | ソース | Light基底クラス定義 |
| Object3D.js | `src/core/Object3D.js` | ソース | 親クラス（3Dオブジェクト基底） |
| Color.js | `src/math/Color.js` | ソース | 色管理クラス |
| AmbientLight.js | `src/lights/AmbientLight.js` | ソース | 環境光（派生クラス） |
| DirectionalLight.js | `src/lights/DirectionalLight.js` | ソース | 平行光源（派生クラス） |
| PointLight.js | `src/lights/PointLight.js` | ソース | 点光源（派生クラス） |
| SpotLight.js | `src/lights/SpotLight.js` | ソース | スポットライト（派生クラス） |
| HemisphereLight.js | `src/lights/HemisphereLight.js` | ソース | 半球光（派生クラス） |
| RectAreaLight.js | `src/lights/RectAreaLight.js` | ソース | 矩形エリアライト（派生クラス） |
