# 機能設計書 67-AmbientLight

## 概要

本ドキュメントは、Three.jsライブラリにおける環境光「AmbientLight」の機能設計について記述する。AmbientLightは、シーン全体を均一に照らす光源であり、方向を持たないため影を生成しない。

### 本機能の処理概要

AmbientLightは、3Dシーン内のすべてのオブジェクトを均一に照らす環境光を提供する光源クラスである。光線の方向がないため、オブジェクトのどの面も同じように照らされ、影を落とすことはできない。

**業務上の目的・背景**：現実世界では、直接光が当たらない部分も完全な暗闘にはならない。これは、光が壁や床などの物体に反射して間接光として届くためである。AmbientLightは、この間接光（環境光）を簡易的にシミュレートするために使用される。リアルタイム3Dグラフィックスにおいて、グローバルイルミネーション（GI）の計算は非常に高コストであるため、AmbientLightは低コストで影のない部分を適度に明るくする実用的な解決策を提供する。

**機能の利用シーン**：
- シーン全体の基本的な明るさの設定
- 影のない部分（アンビエントオクルージョン）の底上げ
- 屋外シーンでの空の散乱光のシミュレーション
- 室内シーンでの間接光の簡易表現
- デバッグ目的でのオブジェクト確認

**主要な処理内容**：
1. 環境光の色と強度の設定
2. シーン内の全オブジェクトへの均一な照明
3. 方向を持たない光としてのライティング計算

**関連システム・外部連携**：WebGLRenderer/WebGPURendererと連携してシェーダーにライティングデータを提供。

**権限による制御**：特になし（クライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | Menubar - Add | 補助機能 | 環境光の追加 |

## 機能種別

データ定義 / 光源設定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number/Color/string | No | 光の色（デフォルト: 0xffffff） | Color.setで受け付ける形式 |
| intensity | number | No | 光の強度（デフォルト: 1） | 数値 |

### 入力データソース

コンストラクタの引数から直接入力。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isAmbientLight | boolean | 型判定用フラグ（常にtrue） |
| type | string | オブジェクトタイプ識別子（'AmbientLight'） |
| color | Color | 光の色（Lightから継承） |
| intensity | number | 光の強度（Lightから継承） |

### 出力先

レンダラーに環境光情報として提供される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ super(color, intensity) - Lightのコンストラクタ実行
2. isAmbientLightフラグ設定
3. type設定
4. シーンに追加
5. レンダリング時:
   a. 全オブジェクトのシェーダーに環境光を加算
   b. 物体色 * 環境光色 * 強度 を計算
```

### フローチャート

```mermaid
flowchart TD
    A[new AmbientLight] --> B[super - Light初期化]
    B --> C[isAmbientLight = true]
    C --> D[type = 'AmbientLight']
    D --> E[初期化完了]
    E --> F[scene.add]
    F --> G[レンダリング時]
    G --> H[全オブジェクトに均一な光]
    H --> I{影の計算?}
    I -->|No| J[影を生成しない]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-67-1 | 影なし | AmbientLightは方向を持たないため、影を生成できない | 常時 |
| BR-67-2 | 均一照明 | シーン内の全オブジェクトを同じ強度で照らす | 常時 |
| BR-67-3 | 位置無関係 | ライトの位置（position）は照明計算に影響しない | 常時 |

### 計算ロジック

環境光のライティング計算:
```glsl
vec3 ambientLightColor = ambientLightColor * ambientLightIntensity;
vec3 ambient = diffuseColor * ambientLightColor;
```

## データベース操作仕様

該当なし（クライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | - | 特になし | - |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 非常に軽量な光源
- 影計算がないため、他の光源より低コスト

## セキュリティ考慮事項

特になし（クライアントサイドライブラリ）

## 備考

- 43行という非常に簡潔な実装
- 最もシンプルな光源タイプ
- 他の光源と組み合わせて使用することが多い

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 親クラスを理解する

AmbientLightが継承するLightクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Light.js | `src/lights/Light.js` | 親クラスの機能（color、intensity） |
| 1-2 | Object3D.js | `src/core/Object3D.js` | 大親クラス（位置管理等、ただしAmbientLightでは使用しない） |

**読解のコツ**: AmbientLightは位置（position）を持つが、照明計算では使用されない点に注意。

#### Step 2: エントリーポイントを理解する

AmbientLightクラス自体の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AmbientLight.js | `src/lights/AmbientLight.js` | クラス全体（43行の非常に簡潔な実装） |

**主要処理フロー**:
1. **1行目**: `import { Light } from './Light.js'` - 親クラスインポート
2. **15行目**: `class AmbientLight extends Light` - Light継承
3. **22-23行目**: コンストラクタ引数（color, intensity）
4. **25行目**: `super( color, intensity )` - Lightコンストラクタ呼び出し
5. **34行目**: `this.isAmbientLight = true` - 型判定フラグ
6. **36行目**: `this.type = 'AmbientLight'` - タイプ識別子

#### Step 3: シェーダーでの使用を確認する

AmbientLightがシェーダーでどのように使用されるか確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | lights_pars_begin.glsl.js | `src/renderers/shaders/ShaderChunk/lights_pars_begin.glsl.js` | 環境光ユニフォーム定義 |

### プログラム呼び出し階層図

```
AmbientLight (継承: Light → Object3D → EventDispatcher)
    │
    └─ constructor(color, intensity)
           │
           └─ super(color, intensity)
                   │
                   └─ Light.constructor(color, intensity)
                            │
                            ├─ super() → Object3D.constructor()
                            │
                            ├─ this.isLight = true
                            ├─ this.color = new Color(color)
                            └─ this.intensity = intensity
           │
           ├─ this.isAmbientLight = true
           │
           └─ this.type = 'AmbientLight'

[レンダリング時]
    │
    └─ WebGLLights
            │
            └─ ambientLightColor += light.color * light.intensity
```

### データフロー図

```
[入力]                    [処理]                         [出力]

color ─────────────▶ AmbientLight.constructor()
intensity                  │
                           ├─ Light初期化
                           ├─ isAmbientLight設定
                           └─ type設定
                                         │
                                         ▼
                              AmbientLightインスタンス
                                         │
                                         ▼
                                  scene.add(light)
                                         │
                                         ▼
                              WebGLLights.setup()
                                         │
                                         ▼
                              シェーダーユニフォーム
                              (ambientLightColor)
                                         │
                                         ▼
                              全オブジェクトに均一な光を加算
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AmbientLight.js | `src/lights/AmbientLight.js` | ソース | AmbientLightクラス定義（43行） |
| Light.js | `src/lights/Light.js` | ソース | 親クラス（Light） |
| Object3D.js | `src/core/Object3D.js` | ソース | 大親クラス |
| WebGLLights.js | `src/renderers/webgl/WebGLLights.js` | ソース | ライト管理 |
| lights_pars_begin.glsl.js | `src/renderers/shaders/ShaderChunk/lights_pars_begin.glsl.js` | シェーダー | 環境光ユニフォーム |
