# 機能設計書 68-DirectionalLight

## 概要

本ドキュメントは、Three.jsライブラリにおける平行光源「DirectionalLight」の機能設計について記述する。DirectionalLightは、特定の方向から無限遠に位置する光源（太陽光のような平行光線）をシミュレートし、影を生成できる光源である。

### 本機能の処理概要

DirectionalLightは、無限遠から特定の方向に向かって平行に照射される光をシミュレートする光源クラスである。太陽光のような自然光の表現に最適で、シャドウマップによる影の生成が可能である。

**業務上の目的・背景**：現実世界の太陽光は、地球との距離が非常に遠いため、地表に届く光線はほぼ平行と見なせる。この特性を3Dグラフィックスで再現するためにDirectionalLightが使用される。屋外シーンの照明、建築ビジュアライゼーション、ゲームの昼間シーンなど、自然光を必要とする多くの場面で使用される。平行光源は位置ではなく方向が重要であり、シーン全体を均一な方向から照らす。また、DirectionalLightShadowを使用することで、リアルな影を生成できる。

**機能の利用シーン**：
- 屋外シーンの太陽光シミュレーション
- 建築ビジュアライゼーションでの自然光表現
- ゲームの昼間シーン照明
- スタジオ照明のキーライトとして
- 影付きの製品レンダリング

**主要な処理内容**：
1. 光の方向（position → target）の設定
2. シャドウマップによる影の生成
3. 正射影カメラによる影計算
4. ターゲットオブジェクトの追従

**関連システム・外部連携**：WebGLRenderer/WebGPURendererと連携。DirectionalLightShadowによるシャドウマップ生成。

**権限による制御**：特になし（クライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | Menubar - Add | 補助機能 | 平行光源の追加 |
| 14 | Menubar - View | 補助機能 | ライトヘルパー表示切替 |

## 機能種別

データ定義 / 光源設定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| color | number/Color/string | No | 光の色（デフォルト: 0xffffff） | Color.setで受け付ける形式 |
| intensity | number | No | 光の強度（デフォルト: 1） | 数値 |

### 入力データソース

コンストラクタの引数から直接入力。position、targetで光の方向を制御。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isDirectionalLight | boolean | 型判定用フラグ（常にtrue） |
| type | string | オブジェクトタイプ識別子（'DirectionalLight'） |
| color | Color | 光の色（Lightから継承） |
| intensity | number | 光の強度（Lightから継承） |
| position | Vector3 | ライトの位置（方向計算に使用） |
| target | Object3D | ライトが向くターゲット |
| shadow | DirectionalLightShadow | シャドウ設定 |

### 出力先

レンダラーに平行光源情報として提供される。シャドウマップが生成される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ super(color, intensity) - Lightのコンストラクタ実行
2. isDirectionalLightフラグ設定
3. type設定
4. positionをDEFAULT_UP (0,1,0) にコピー
5. updateMatrix()で行列更新
6. target = new Object3D() でターゲット作成
7. shadow = new DirectionalLightShadow() でシャドウ設定初期化
8. シーンに追加（targetもシーンに追加が必要な場合あり）
9. レンダリング時:
   a. position → targetの方向を計算
   b. 平行光線としてライティング
   c. castShadow=trueの場合、シャドウマップを生成
```

### フローチャート

```mermaid
flowchart TD
    A[new DirectionalLight] --> B[super - Light初期化]
    B --> C[isDirectionalLight = true]
    C --> D[type = 'DirectionalLight']
    D --> E[position = DEFAULT_UP]
    E --> F[updateMatrix]
    F --> G[target = new Object3D]
    G --> H[shadow = new DirectionalLightShadow]
    H --> I[初期化完了]
    I --> J[scene.add]
    J --> K{castShadow?}
    K -->|Yes| L[シャドウマップ生成]
    K -->|No| M[影なし描画]
    L --> N[平行光線ライティング]
    M --> N
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-68-1 | 方向制御 | 光の方向はposition→targetのベクトルで決定される | 常時 |
| BR-68-2 | rotation無効 | rotationプロパティは光の方向に影響しない | 常時 |
| BR-68-3 | target追加 | targetの位置を変更するにはtargetをシーンに追加する必要がある | target位置変更時 |
| BR-68-4 | 正射影シャドウ | シャドウマップはOrthographicCameraで生成される | 影生成時 |
| BR-68-5 | デフォルト位置 | positionのデフォルトは(0,1,0)（上方向） | 初期化時 |

### 計算ロジック

光の方向計算:
```javascript
lightDirection = normalize(light.position - light.target.position)
```

シャドウマップカメラ（正射影）:
```javascript
// デフォルト設定
left: -5, right: 5, top: 5, bottom: -5, near: 0.5, far: 500
```

## データベース操作仕様

該当なし（クライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 視覚的問題 | targetがシーンに追加されていない場合、位置変更が反映されない | target をシーンに追加 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- シャドウマップ有効時はレンダリングパスが追加される
- シャドウマップサイズが大きいほど品質向上・負荷増加
- 複数のDirectionalLightで影を有効にすると負荷が増加

## セキュリティ考慮事項

特になし（クライアントサイドライブラリ）

## 備考

- 114行の実装
- シーン内で最も一般的に使用される光源の一つ
- targetオブジェクトを他の3Dオブジェクトに設定して追従させることも可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: 親クラスを理解する

DirectionalLightが継承するLightクラスの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Light.js | `src/lights/Light.js` | 親クラスの機能（color、intensity） |
| 1-2 | Object3D.js | `src/core/Object3D.js` | position、DEFAULT_UPの定義 |

**読解のコツ**: Object3D.DEFAULT_UP（1647行目）が(0,1,0)であることを確認。

#### Step 2: エントリーポイントを理解する

DirectionalLightクラス自体の実装を確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | DirectionalLight.js | `src/lights/DirectionalLight.js` | クラス全体（114行） |

**主要処理フロー**:
1. **1-3行目**: インポート（Light、DirectionalLightShadow、Object3D）
2. **32行目**: `class DirectionalLight extends Light` - Light継承
3. **39-40行目**: コンストラクタ引数（color, intensity）
4. **42行目**: `super( color, intensity )` - Lightコンストラクタ呼び出し
5. **51行目**: `this.isDirectionalLight = true` - 型判定フラグ
6. **53行目**: `this.type = 'DirectionalLight'` - タイプ識別子
7. **55行目**: `this.position.copy( Object3D.DEFAULT_UP )` - 初期位置設定
8. **56行目**: `this.updateMatrix()` - 行列更新
9. **70行目**: `this.target = new Object3D()` - ターゲット作成
10. **77行目**: `this.shadow = new DirectionalLightShadow()` - シャドウ設定
11. **81-86行目**: dispose()メソッド
12. **89-97行目**: copy()メソッド
13. **100-108行目**: toJSON()メソッド

#### Step 3: シャドウ設定を確認する

DirectionalLightShadowクラスを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | DirectionalLightShadow.js | `src/lights/DirectionalLightShadow.js` | 正射影カメラでのシャドウ設定 |
| 3-2 | LightShadow.js | `src/lights/LightShadow.js` | シャドウ基底クラス |

### プログラム呼び出し階層図

```
DirectionalLight (継承: Light → Object3D → EventDispatcher)
    │
    ├─ constructor(color, intensity)
    │      │
    │      ├─ super(color, intensity) → Light.constructor()
    │      │
    │      ├─ this.isDirectionalLight = true
    │      ├─ this.type = 'DirectionalLight'
    │      ├─ this.position.copy(Object3D.DEFAULT_UP)
    │      ├─ this.updateMatrix()
    │      ├─ this.target = new Object3D()
    │      └─ this.shadow = new DirectionalLightShadow()
    │                              │
    │                              └─ new OrthographicCamera(-5,5,5,-5,0.5,500)
    │
    ├─ dispose()
    │      │
    │      ├─ super.dispose() → Light.dispose()
    │      └─ this.shadow.dispose()
    │
    ├─ copy(source)
    │      │
    │      ├─ super.copy(source)
    │      ├─ this.target = source.target.clone()
    │      └─ this.shadow = source.shadow.clone()
    │
    └─ toJSON(meta)
           │
           ├─ super.toJSON(meta)
           ├─ data.object.shadow = this.shadow.toJSON()
           └─ data.object.target = this.target.uuid
```

### データフロー図

```
[入力]                    [処理]                         [出力]

color ─────────────▶ DirectionalLight.constructor()
intensity                  │
                           ├─ Light初期化
                           ├─ position = (0,1,0)
                           ├─ target = new Object3D()
                           └─ shadow = new DirectionalLightShadow()
                                         │
                                         ▼
                              DirectionalLightインスタンス
                                         │
                    ┌────────────────────┼────────────────────┐
                    ▼                    ▼                    ▼
              position              target               shadow
              (0,1,0)              (0,0,0)         OrthographicCamera
                    │                    │                    │
                    └────────────────────┼────────────────────┘
                                         │
                              lightDirection =
                              normalize(position - target.position)
                                         │
                                         ▼
                    ┌────────────────────┴────────────────────┐
                    ▼                                         ▼
           平行光線ライティング                        シャドウマップ生成
           (全オブジェクトに同方向)              (OrthographicCamera使用)
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| DirectionalLight.js | `src/lights/DirectionalLight.js` | ソース | DirectionalLightクラス定義（114行） |
| DirectionalLightShadow.js | `src/lights/DirectionalLightShadow.js` | ソース | シャドウ設定クラス |
| Light.js | `src/lights/Light.js` | ソース | 親クラス（Light） |
| LightShadow.js | `src/lights/LightShadow.js` | ソース | シャドウ基底クラス |
| Object3D.js | `src/core/Object3D.js` | ソース | 大親クラス、DEFAULT_UP定義 |
| OrthographicCamera.js | `src/cameras/OrthographicCamera.js` | ソース | シャドウマップカメラ |
| DirectionalLightHelper.js | `src/helpers/DirectionalLightHelper.js` | ソース | 可視化ヘルパー |
