# 機能設計書 78-Line

## 概要

本ドキュメントは、Three.jsライブラリにおけるLine（線分）機能の設計を記述する。Lineは連続した頂点を直線で結んで描画する3Dオブジェクトであり、ワイヤーフレーム、パス、グラフなどの線描画に使用される。

### 本機能の処理概要

Lineは、BufferGeometryの頂点を順番に直線で接続して描画するクラスである。Object3Dを継承し、3Dシーングラフに配置可能で、LineBasicMaterialまたはLineDashedMaterialと組み合わせて使用される。レイキャスティングによる線との交差判定も提供する。

**業務上の目的・背景**：3Dグラフィックスにおいて、線描画はワイヤーフレーム表示、パスの可視化、グラフ描画、デバッグ用の補助線など、多くの用途で必要とされる。Lineクラスは、頂点の配列から連続した線分を生成し、効率的にレンダリングする機能を提供する。

**機能の利用シーン**：ワイヤーフレーム表示、移動軌跡の可視化、グラフや図表の3D表現、デバッグ用の補助線、3Dパスの可視化、エッジ検出結果の表示。

**主要な処理内容**：
1. ジオメトリ（BufferGeometry）とマテリアル（LineBasicMaterial）の管理
2. 連続した頂点を直線で接続して描画
3. 破線描画用の距離計算（computeLineDistances）
4. レイキャスティングによる線との交差判定
5. モーフターゲットのサポート

**関連システム・外部連携**：WebGLRenderer/WebGPURenderer、Raycaster、BufferGeometry、LineBasicMaterial、LineDashedMaterial。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 画面機能マッピングに登録なし |

## 機能種別

3Dオブジェクト生成 / 線描画

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| geometry | BufferGeometry | No | 線のジオメトリ（デフォルト: 新規BufferGeometry） | BufferGeometry型 |
| material | Material/Array<Material> | No | 線のマテリアル（デフォルト: 新規LineBasicMaterial） | Material型または配列 |

### 入力データソース

JavaScriptコードからのコンストラクタ呼び出し。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isLine | boolean | 型識別フラグ（常にtrue） |
| type | string | オブジェクト種別（'Line'） |
| geometry | BufferGeometry | 線のジオメトリ |
| material | Material/Array<Material> | 線のマテリアル |
| morphTargetDictionary | Object | モーフターゲット辞書（オプション） |
| morphTargetInfluences | Array<number> | モーフターゲット影響度（オプション） |

### 出力先

3Dシーングラフ、WebGLレンダリングパイプライン

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ Object3Dの初期化
   └─ geometryとmaterialの設定
   └─ updateMorphTargets()の呼び出し
2. computeLineDistances() - 破線用距離計算
   └─ 各頂点間の累積距離を計算
   └─ lineDistance属性として保存
3. raycast() - レイキャスト
   └─ 境界球テスト
   └─ 線分ごとの交差判定
```

### フローチャート

```mermaid
flowchart TD
    A[開始] --> B[Object3Dコンストラクタ]
    B --> C[isLine = true]
    C --> D[geometry設定]
    D --> E[material設定]
    E --> F[updateMorphTargets]
    F --> G[終了]

    H[computeLineDistances] --> I{インデックス付き?}
    I -->|No| J[頂点間距離を計算]
    I -->|Yes| K[警告を出力]
    J --> L[lineDistance属性を設定]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-78-1 | 非インデックス推奨 | computeLineDistancesは非インデックスジオメトリのみ対応 | 破線描画時 |
| BR-78-2 | 閾値ベース交差 | レイキャストはthreshold（Line.threshold）を使用 | raycast時 |
| BR-78-3 | 連続描画 | 頂点は0→1→2→...の順で連続して接続される | 常時 |

### 計算ロジック

**線分との交差判定（レイキャスト）**：
```javascript
// 線分と点の最小距離を計算
distSq = ray.distanceSqToSegment(vStart, vEnd, intersectPointOnRay, intersectPointOnSegment);

// 閾値（localThreshold）以内なら交差とみなす
if (distSq <= localThresholdSq) {
    // 交差点を返却
}
```

**破線用距離計算**：
```javascript
lineDistances[i] = lineDistances[i-1] + vStart.distanceTo(vEnd);
```

## データベース操作仕様

データベース操作なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 警告 | インデックス付きジオメトリでcomputeLineDistances | 非インデックスジオメトリを使用 |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- 線の描画は低コストで実行される
- 頂点数に比例してレイキャスト処理時間が増加
- computeLineDistancesは一度実行すれば十分

## セキュリティ考慮事項

特になし。

## 備考

- LineはGL_LINE_STRIP相当の描画を行う
- 個別の線分を描画する場合はLineSegmentsを使用
- 閉じた線（ループ）を描画する場合はLineLoopを使用

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Lineが持つプロパティの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | BufferGeometry.js | `src/core/BufferGeometry.js` | ジオメトリの構造 |
| 1-2 | LineBasicMaterial.js | `src/materials/LineBasicMaterial.js` | デフォルトマテリアル |

**読解のコツ**: Lineのジオメトリはposition属性のみ必須で、オプションでlineDistance属性を持つ。

#### Step 2: エントリーポイントを理解する

Lineクラスの構造を把握する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Line.js | `src/objects/Line.js` | クラス全体の構造 |

**主要処理フロー**:
1. **49行目**: コンストラクタでgeometryとmaterialを受け取る
2. **60行目**: isLineフラグの設定
3. **62行目**: typeを'Line'に設定
4. **69行目**: geometryの設定
5. **77行目**: materialの設定
6. **99行目**: updateMorphTargets()の呼び出し

#### Step 3: 破線距離計算を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Line.js | `src/objects/Line.js` | computeLineDistances()（121-152行目） |

**主要処理フロー**:
- **127-128行目**: インデックスなしチェック
- **129-141行目**: 頂点間の累積距離計算
- **142行目**: lineDistance属性として設定

#### Step 4: レイキャスト処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Line.js | `src/objects/Line.js` | raycast()（160-257行目） |
| 4-2 | Line.js | `src/objects/Line.js` | checkIntersection()（296-327行目） |

**主要処理フロー**:
- **169行目**: 境界球の計算
- **171-175行目**: 境界球との交差判定
- **182-183行目**: 閾値の計算（スケール考慮）
- **185行目**: stepの設定（Lineは1、LineSegmentsは2）
- **191-256行目**: 線分ごとの交差判定

### プログラム呼び出し階層図

```
Line
    │
    ├─ extends Object3D
    │      └─ extends EventDispatcher
    │
    ├─ uses BufferGeometry
    │      ├─ attributes.position
    │      └─ attributes.lineDistance (optional)
    │
    └─ uses LineBasicMaterial (default)

Line.raycast()
    │
    ├─ geometry.computeBoundingSphere()
    ├─ _sphere.applyMatrix4()
    ├─ ray.intersectsSphere()
    └─ checkIntersection()
           └─ ray.distanceSqToSegment()
```

### データフロー図

```
[入力]                         [処理]                           [出力]

geometry (引数) ──────────────▶ 直接代入 ───────────────────────▶ this.geometry
material (引数) ──────────────▶ 直接代入 ───────────────────────▶ this.material

computeLineDistances()
positions ────────────────────▶ 距離計算 ──────────────────────▶ lineDistance属性

raycast()
Raycaster.ray ────────────────▶ 線分交差判定 ──────────────────▶ intersects[]
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Line.js | `src/objects/Line.js` | ソース | 線分クラス |
| Object3D.js | `src/core/Object3D.js` | ソース | 基底クラス |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | ジオメトリ |
| LineBasicMaterial.js | `src/materials/LineBasicMaterial.js` | ソース | 基本線マテリアル |
| LineDashedMaterial.js | `src/materials/LineDashedMaterial.js` | ソース | 破線マテリアル |
| Ray.js | `src/math/Ray.js` | ソース | レイ（光線） |
| Sphere.js | `src/math/Sphere.js` | ソース | 境界球 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル |
