# 機能設計書 81-Points

## 概要

本ドキュメントは、Three.jsにおける点群オブジェクト `Points` クラスの機能設計を定義する。Pointsは3D空間に点の集合を描画するためのオブジェクトクラスであり、パーティクルシステムや散布図、点群データの可視化に使用される。

### 本機能の処理概要

Pointsクラスは、3D空間に点（ポイント）の集合を表示するためのレンダリングオブジェクトである。BufferGeometryで定義された頂点位置データとPointsMaterialを組み合わせ、各頂点位置にポイントスプライトを描画する。

**業務上の目的・背景**：3Dアプリケーションにおいて、パーティクルエフェクト（雪、雨、煙など）、科学データの可視化（散布図、点群データ）、星空などの天体表現を実現するために必要な機能である。ポリゴンメッシュとは異なり、個々の点を独立して描画できるため、大量の小さなオブジェクトを効率的にレンダリングできる。

**機能の利用シーン**：
- パーティクルシステム（雪、雨、火花、爆発エフェクト）
- LiDAR等の点群データ（ポイントクラウド）の可視化
- 散布図やデータビジュアライゼーション
- 星空や銀河の表現
- モーフターゲットを使用したアニメーション付き点群

**主要な処理内容**：
1. BufferGeometryとPointsMaterialの関連付けによるPoints構築
2. レイキャスティングによる点との交差判定
3. モーフターゲットによる点群アニメーション
4. 描画範囲（drawRange）に基づく部分描画

**関連システム・外部連携**：WebGLRenderer/WebGPURendererと連携し、GL_POINTSプリミティブとして描画される。Raycasterと連携してマウスピッキングを実現する。

**権限による制御**：特になし（Three.jsはクライアントサイドライブラリ）

## 関連画面

画面機能マッピング.csvに本機能（No.81）の関連画面情報は記載されていない。

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | - |

## 機能種別

オブジェクト生成・レンダリング / レイキャスト判定

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| geometry | BufferGeometry | No | 点群のジオメトリ | 省略時はデフォルトの空BufferGeometry |
| material | Material\|Array\<Material\> | No | 点群のマテリアル | 省略時はデフォルトのPointsMaterial |

### 入力データソース

- BufferGeometry: 頂点位置データ（position属性）
- PointsMaterial: 点のサイズ、色、テクスチャ等の外観定義

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isPoints | boolean | 型判定フラグ（常にtrue） |
| geometry | BufferGeometry | 関連付けられたジオメトリ |
| material | Material\|Array\<Material\> | 関連付けられたマテリアル |
| morphTargetDictionary | Object\<string,number\>\|undefined | モーフターゲット辞書 |
| morphTargetInfluences | Array\<number\>\|undefined | モーフターゲット影響度配列 |

### 出力先

レンダリング結果はWebGLまたはWebGPUフレームバッファに出力される。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ Object3Dの初期化、geometry/materialの設定
2. updateMorphTargets()
   └─ ジオメトリのモーフ属性を検出し、辞書と影響度配列を初期化
3. レンダリング時
   └─ レンダラーがgeometryとmaterialを使用して点を描画
4. raycast()呼び出し時
   └─ 境界球チェック → 各点との距離計算 → 交差結果の収集
```

### フローチャート

```mermaid
flowchart TD
    A[Points生成] --> B[geometry/material設定]
    B --> C[updateMorphTargets呼び出し]
    C --> D{モーフ属性あり?}
    D -->|Yes| E[morphTargetDictionary/Influences初期化]
    D -->|No| F[スキップ]
    E --> G[レンダリング待機]
    F --> G
    G --> H{raycast要求?}
    H -->|Yes| I[境界球チェック]
    I --> J{交差あり?}
    J -->|Yes| K[各点との距離計算]
    J -->|No| L[処理終了]
    K --> M[交差結果返却]
    H -->|No| N[レンダリング実行]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-81-01 | 点判定閾値 | レイキャスト時、raycaster.params.Points.thresholdで判定距離を制御 | raycast実行時 |
| BR-81-02 | スケール補正 | 閾値はオブジェクトのスケールで補正される | raycast実行時 |
| BR-81-03 | 描画範囲制限 | geometry.drawRangeで描画・判定対象を制限可能 | レンダリング・raycast時 |

### 計算ロジック

レイキャスト判定時のローカル閾値計算:
```javascript
localThreshold = threshold / ((scale.x + scale.y + scale.z) / 3)
localThresholdSq = localThreshold * localThreshold
```

## データベース操作仕様

該当なし（Three.jsはクライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 境界球未計算 | boundingSphere===null | 自動でcomputeBoundingSphere()を呼び出し |

### リトライ仕様

特になし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 大量の点（数十万点以上）を効率的に描画可能
- インデックス付きジオメトリと非インデックスジオメトリの両方をサポート
- モーフターゲットによるGPUベースのアニメーション

## セキュリティ考慮事項

特になし（クライアントサイドレンダリング）

## 備考

- Spriteと異なり、Pointsは個々の点を回転させることはできない
- WebGLのgl.ALIASED_POINT_SIZE_RANGEによりポイントサイズに制限がある場合がある

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Pointsオブジェクトの構造と、使用されるマテリアルの定義を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | PointsMaterial.js | `src/materials/PointsMaterial.js` | 点の外観（色、サイズ、テクスチャ）を定義するマテリアル |
| 1-2 | BufferGeometry.js | `src/core/BufferGeometry.js` | 頂点データを格納するジオメトリの基底クラス |

**読解のコツ**: PointsMaterialのプロパティ（size, sizeAttenuation, color, map等）がポイントの見た目をどう制御するか把握する。

#### Step 2: エントリーポイントを理解する

Pointsクラス本体の構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Points.js | `src/objects/Points.js` | 点群オブジェクトのメインクラス |

**主要処理フロー**:
1. **27-79行目**: コンストラクタでgeometry/materialを設定、updateMorphTargets()を呼び出し
2. **98-157行目**: raycast()でレイキャスト処理を実行
3. **163-192行目**: updateMorphTargets()でモーフターゲットを初期化
4. **196-226行目**: testPoint()ヘルパー関数で個別点との交差判定

#### Step 3: 継承関係を理解する

Object3Dからの継承による基本機能を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Object3D.js | `src/core/Object3D.js` | 位置・回転・スケール管理、親子関係の基底クラス |

**主要処理フロー**:
- **140-143行目**: position, rotation, quaternion, scaleのプロパティ定義
- **744-785行目**: add()による子オブジェクト追加
- **1163-1201行目**: updateMatrixWorld()によるワールド行列更新

### プログラム呼び出し階層図

```
Points
    │
    ├─ Object3D (継承)
    │      ├─ EventDispatcher (継承)
    │      └─ Layers (レイヤー管理)
    │
    ├─ BufferGeometry (geometry)
    │      └─ BufferAttribute (頂点データ)
    │
    ├─ PointsMaterial (material)
    │      └─ Material (継承)
    │
    └─ raycast()
           ├─ Sphere (境界球判定)
           ├─ Ray (レイ計算)
           ├─ Matrix4 (座標変換)
           └─ testPoint() (点判定ヘルパー)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

BufferGeometry ────▶ Points Constructor ────▶ Points Object
(position属性)              │                      │
                            │                      ├─ isPoints: true
PointsMaterial ────────────┘                      ├─ geometry
(size, color等)                                    ├─ material
                                                   └─ morphTargetInfluences

Raycaster.ray ────▶ raycast() ────▶ intersects[]
                       │              (distance, point, index, object)
                       │
                       ├─ 境界球チェック
                       ├─ ローカル空間変換
                       └─ testPoint()で各点判定
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Points.js | `src/objects/Points.js` | ソース | 点群オブジェクトのメインクラス |
| Object3D.js | `src/core/Object3D.js` | ソース | 3Dオブジェクトの基底クラス |
| BufferGeometry.js | `src/core/BufferGeometry.js` | ソース | 頂点データ管理 |
| PointsMaterial.js | `src/materials/PointsMaterial.js` | ソース | 点群マテリアル |
| Material.js | `src/materials/Material.js` | ソース | マテリアル基底クラス |
| Sphere.js | `src/math/Sphere.js` | ソース | 境界球計算 |
| Ray.js | `src/math/Ray.js` | ソース | レイ計算 |
| Matrix4.js | `src/math/Matrix4.js` | ソース | 行列変換 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル |
