# 機能設計書 83-Group

## 概要

本ドキュメントは、Three.jsにおけるオブジェクトグループ化機能 `Group` クラスの機能設計を定義する。GroupはObject3Dとほぼ同一の機能を持ち、複数の3Dオブジェクトを論理的にグループ化して一括操作を可能にするためのコンテナクラスである。

### 本機能の処理概要

Groupクラスは、複数のObject3Dインスタンスを子要素として持ち、グループ全体に対して変換（位置、回転、スケール）を適用できるコンテナである。Object3Dを直接使用することも可能だが、Groupを使うことでコードの可読性と意図の明確化が図れる。

**業務上の目的・背景**：3Dアプリケーションにおいて、関連する複数のオブジェクトを論理的にまとめ、一括で移動・回転・スケーリングする必要がある。例えば、車の部品（車体、タイヤ、ハンドル）をグループ化して車全体を操作したり、建物の複数フロアをグループ化して管理したりする場合に使用する。

**機能の利用シーン**：
- 複数パーツで構成されるモデルのグループ化（車、建物、キャラクター等）
- シーン内のオブジェクト階層構造の整理
- 複数オブジェクトの一括変換操作
- レンダリング順序の制御（renderOrderの一括適用）
- 可視性の一括制御（visible切り替え）

**主要な処理内容**：
1. 子オブジェクトの追加・削除（add/remove）
2. グループ全体への変換適用（position, rotation, scale）
3. 子オブジェクトへの変換の伝播
4. シリアライズ・デシリアライズ

**関連システム・外部連携**：Scene、その他すべてのObject3D派生クラスと連携。レンダラーは子要素を再帰的に描画する。

**権限による制御**：特になし（Three.jsはクライアントサイドライブラリ）

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 12 | Menubar - Add | 補助機能 | グループオブジェクトの追加 |

## 機能種別

コンテナ / オブジェクト管理

## 入力仕様

### 入力パラメータ

コンストラクタはパラメータを取らない。

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| - | - | - | - | - |

### 入力データソース

特になし（空のコンテナとして生成）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isGroup | boolean | 型判定フラグ（常にtrue） |
| type | string | オブジェクトタイプ（'Group'） |
| children | Array\<Object3D\> | 子オブジェクト配列（Object3Dから継承） |

### 出力先

グループ自体は描画されず、子オブジェクトが個別にレンダリングされる。

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ Object3Dの初期化、isGroup/typeの設定
2. add()による子オブジェクト追加
   └─ children配列に追加、親子関係の設定
3. 変換操作
   └─ グループのmatrixWorld更新 → 子へ伝播
4. レンダリング時
   └─ レンダラーがchildrenを再帰的に処理
```

### フローチャート

```mermaid
flowchart TD
    A[Group生成] --> B[Object3D初期化]
    B --> C[isGroup=true設定]
    C --> D[type='Group'設定]
    D --> E[待機状態]
    E --> F{add呼び出し?}
    F -->|Yes| G[children配列に追加]
    G --> H[親参照設定]
    H --> E
    F -->|No| I{変換操作?}
    I -->|Yes| J[matrixWorld更新]
    J --> K[子へ変換伝播]
    K --> E
    I -->|No| L{レンダリング?}
    L -->|Yes| M[children再帰処理]
    M --> E
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-83-01 | 変換の継承 | 子オブジェクトはグループの変換を継承する | 常時 |
| BR-83-02 | 可視性の継承 | グループのvisible=falseで子も非表示 | 常時 |
| BR-83-03 | renderOrderの影響 | グループのrenderOrderは子に影響し、同一renderOrderでソート | 常時 |
| BR-83-04 | 親子の排他性 | オブジェクトは1つの親しか持てない | add時 |

### 計算ロジック

ワールド行列の伝播:
```javascript
// 子のワールド行列 = 親のワールド行列 * 子のローカル行列
child.matrixWorld.multiplyMatrices(parent.matrixWorld, child.matrix)
```

## データベース操作仕様

該当なし（Three.jsはクライアントサイドライブラリ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | 自己参照エラー | object === thisで自分自身を追加 | エラーメッセージ出力、追加拒否 |
| - | 型エラー | Object3D以外を追加 | エラーメッセージ出力、追加拒否 |

### リトライ仕様

特になし

## トランザクション仕様

該当なし

## パフォーマンス要件

- Groupは軽量なコンテナであり、描画負荷は発生しない
- 深い階層構造は行列計算のオーバーヘッドとなるため注意

## セキュリティ考慮事項

特になし（クライアントサイドレンダリング）

## 備考

- GroupとObject3Dの機能的な違いはisGroupフラグとtypeのみ
- セマンティクス（意味的な明確さ）のために使用が推奨される
- Sceneも実質的にはGroupの一種

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

Groupクラスの非常にシンプルな構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Group.js | `src/objects/Group.js` | Groupクラス全体（非常に短い） |

**読解のコツ**: Groupは最小限の実装であり、ほとんどの機能はObject3Dから継承している点を理解する。

#### Step 2: 継承元を理解する

Object3Dの子オブジェクト管理機能を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Object3D.js | `src/core/Object3D.js` | add/remove/children管理 |

**主要処理フロー**:
1. **744-785行目**: add()で子オブジェクトを追加
2. **796-827行目**: remove()で子オブジェクトを削除
3. **1163-1201行目**: updateMatrixWorld()で変換を子に伝播

#### Step 3: Groupの実装を確認

Groupクラスの全コードを確認する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Group.js | `src/objects/Group.js` | 20-39行目の完全なクラス定義 |

**主要処理フロー**:
1. **22行目**: コンストラクタでsuper()呼び出し
2. **33行目**: `this.isGroup = true`
3. **35行目**: `this.type = 'Group'`

### プログラム呼び出し階層図

```
Group
    │
    └─ Object3D (継承)
           │
           ├─ EventDispatcher (継承)
           │      └─ イベント発行・購読
           │
           ├─ children[] (子オブジェクト配列)
           │      └─ Object3D[] (任意の3Dオブジェクト)
           │
           ├─ add() / remove() (子管理)
           │
           ├─ position / rotation / scale (変換)
           │
           └─ updateMatrixWorld() (行列更新・伝播)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

new Group() ─────────▶ コンストラクタ ─────────▶ Group Object
                            │                        │
                            │                        ├─ isGroup: true
                            │                        ├─ type: 'Group'
                            └────────────────────▶   └─ children: []

group.add(mesh) ─────▶ add() ─────▶ children更新
                          │           親子関係設定
                          │
                          └─ addedイベント発行

group.position.set() ─▶ updateMatrixWorld() ─▶ 子オブジェクトの
                              │                  matrixWorld更新
                              │
                              └─ children再帰処理
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Group.js | `src/objects/Group.js` | ソース | グループコンテナのメインクラス |
| Object3D.js | `src/core/Object3D.js` | ソース | 3Dオブジェクトの基底クラス |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベントシステム |
| Matrix4.js | `src/math/Matrix4.js` | ソース | 行列変換 |
| Vector3.js | `src/math/Vector3.js` | ソース | 3Dベクトル |
| Quaternion.js | `src/math/Quaternion.js` | ソース | 四元数（回転） |
| Euler.js | `src/math/Euler.js` | ソース | オイラー角 |
