# 機能設計書 97-FramebufferTexture

## 概要

本ドキュメントは、Three.jsライブラリにおけるフレームバッファからテクスチャを生成する機能（FramebufferTexture）の設計仕様を記述したものである。

### 本機能の処理概要

FramebufferTextureは、レンダラーのcopyFramebufferToTexture()メソッドと組み合わせて使用し、現在バインドされているフレームバッファの内容を抽出してテクスチャとして利用可能にする機能を提供する。画面の一部をキャプチャして他のオブジェクトのテクスチャとして使用するなどの用途に活用される。

**業務上の目的・背景**：レンダリング結果の一部を再利用したい場面で必要となる。ポータル効果、ミラー効果、スクリーンスペースエフェクト、ピクチャーインピクチャー表示など、レンダリング結果をリアルタイムで別のテクスチャとして使用したい場合に本機能は必要とされる。

**機能の利用シーン**：
- ポータル効果（別の視点からの描画結果を表示）
- リアルタイムミラー/反射
- セキュリティカメラ風の表示
- ミニマップの作成
- スクリーンショット機能
- ピクチャーインピクチャー効果

**主要な処理内容**：
1. 幅・高さパラメータを受け取りテクスチャを初期化
2. NearestFilterをデフォルトフィルターとして設定（フィルタリング無効化）
3. generateMipmaps = false に設定
4. needsUpdate = true を設定（即座に使用可能状態）
5. renderer.copyFramebufferToTexture()と連携してフレームバッファをコピー

**関連システム・外部連携**：WebGLRenderer.copyFramebufferToTexture()、WebGPURenderer.copyFramebufferToTexture()

**権限による制御**：特になし

## 関連画面

画面機能マッピングに該当する関連画面は存在しない（内部処理用テクスチャクラス）。

## 機能種別

データ変換 / テクスチャ生成 / フレームバッファキャプチャ

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| width | number | No | テクスチャの幅 | 正の整数 |
| height | number | No | テクスチャの高さ | 正の整数 |

### 入力データソース

renderer.copyFramebufferToTexture()を通じてコピーされるフレームバッファデータ

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| isFramebufferTexture | boolean | 型判定フラグ（常にtrue） |
| image | Object | {width, height}オブジェクト |
| magFilter | number | NearestFilter（デフォルト） |
| minFilter | number | NearestFilter（デフォルト） |
| generateMipmaps | boolean | ミップマップ生成フラグ（デフォルト: false） |
| needsUpdate | boolean | 更新フラグ（初期化時にtrue） |

### 出力先

GPUテクスチャメモリ

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ 親クラスTextureのコンストラクタを呼び出し（{width, height}）
2. プロパティ初期化
   └─ isFramebufferTexture = true
3. フィルター設定
   └─ magFilter = NearestFilter
   └─ minFilter = NearestFilter
4. ミップマップ設定
   └─ generateMipmaps = false
5. 更新フラグ設定
   └─ needsUpdate = true

使用時:
1. renderer.render()でシーンを描画
2. renderer.copyFramebufferToTexture(texture, position)を呼び出し
3. フレームバッファの内容がテクスチャにコピーされる
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new FramebufferTexture] --> B[親クラスTexture初期化]
    B --> C[isFramebufferTexture = true]
    C --> D[magFilter = NearestFilter]
    D --> E[minFilter = NearestFilter]
    E --> F[generateMipmaps = false]
    F --> G[needsUpdate = true]
    G --> H[終了: FramebufferTexture インスタンス]

    I[使用時] --> J[renderer.render]
    J --> K[copyFramebufferToTexture]
    K --> L[フレームバッファ → テクスチャ]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-97-01 | デフォルトフィルター | NearestFilterでフィルタリング無効化 | 常時 |
| BR-97-02 | ミップマップ無効 | generateMipmaps = false | 常時 |
| BR-97-03 | 即時使用可能 | needsUpdate = trueで初期化 | 常時 |
| BR-97-04 | copyと連携必須 | copyFramebufferToTexture()との組み合わせが必須 | 使用時 |

### 計算ロジック

コピー位置の計算:
```javascript
const vector = new Vector2();
vector.x = (window.innerWidth * pixelRatio / 2) - (textureSize / 2);
vector.y = (window.innerHeight * pixelRatio / 2) - (textureSize / 2);
```

## データベース操作仕様

該当なし（メモリ/GPU操作のみ）

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | サイズ不正 | width/heightが0以下 | 正のサイズを指定 |
| - | フレームバッファなし | 描画前にコピー | render()後にコピーを実行 |

### リトライ仕様

該当なし

## トランザクション仕様

該当なし

## パフォーマンス要件

- 毎フレームコピーする場合はGPU帯域幅に注意
- テクスチャサイズは必要最小限に設定

## セキュリティ考慮事項

- フレームバッファのコンテンツには機密情報が含まれる可能性がある

## 備考

- WebGLRendererとWebGPURendererの両方でcopyFramebufferToTextureがサポートされている
- devicePixelRatioを考慮したサイズ計算が必要

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Texture.js | `src/textures/Texture.js` | 基底クラスの構造を理解 |
| 1-2 | constants.js | `src/constants.js` | NearestFilter定数を確認 |

**読解のコツ**: FramebufferTextureは非常にシンプルな実装で、主要な処理はレンダラー側のcopyFramebufferToTextureにある。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | FramebufferTexture.js | `src/textures/FramebufferTexture.js` | シンプルなコンストラクタを確認 |

**主要処理フロー**:
1. **28行目**: クラス定義開始 `class FramebufferTexture extends Texture`
2. **36行目**: コンストラクタ定義 - width, heightのみを受け取る
3. **38行目**: 親クラスに{width, height}を渡して初期化
4. **47行目**: isFramebufferTexture = true
5. **57行目**: magFilter = NearestFilter
6. **67行目**: minFilter = NearestFilter
7. **77行目**: generateMipmaps = false
8. **79行目**: needsUpdate = true

#### Step 3: 使用方法を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FramebufferTexture.js | `src/textures/FramebufferTexture.js` | JSDocコメントの使用例を確認（9-24行目） |

**使用例（JSDocより）**:
```javascript
const pixelRatio = window.devicePixelRatio;
const textureSize = 128 * pixelRatio;
const frameTexture = new FramebufferTexture( textureSize, textureSize );

const vector = new Vector2();
vector.x = ( window.innerWidth * pixelRatio / 2 ) - ( textureSize / 2 );
vector.y = ( window.innerHeight * pixelRatio / 2 ) - ( textureSize / 2 );

renderer.render( scene, camera );
renderer.copyFramebufferToTexture( frameTexture, vector );
```

### プログラム呼び出し階層図

```
FramebufferTexture (constructor)
    │
    └─ Texture (super)
           │
           ├─ EventDispatcher (super)
           │
           └─ Source
                  └─ {width, height}

使用時:
WebGLRenderer.copyFramebufferToTexture
    │
    └─ FramebufferTexture
           └─ GPU Texture 更新
```

### データフロー図

```
[入力]                    [処理]                      [出力]

width, height       ───▶  FramebufferTexture         ───▶  テクスチャオブジェクト
                          (コンストラクタ)
                          │
                          └─ Texture継承初期化

[使用時]

フレームバッファ    ───▶  renderer.copyFramebuffer   ───▶  GPU Texture
(レンダリング結果)        ToTexture()
                          │
                          └─ 指定位置からコピー
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| FramebufferTexture.js | `src/textures/FramebufferTexture.js` | ソース | フレームバッファテクスチャクラス |
| Texture.js | `src/textures/Texture.js` | ソース | テクスチャ基底クラス |
| constants.js | `src/constants.js` | ソース | NearestFilter等の定数定義 |
| WebGLRenderer.js | `src/renderers/WebGLRenderer.js` | ソース | copyFramebufferToTextureの実装 |
