# 機能設計書 98-Loader

## 概要

本ドキュメントは、Three.jsライブラリにおけるローダーの基底クラス（Loader）の設計仕様を記述したものである。

### 本機能の処理概要

Loaderは、すべてのローダークラスの抽象基底クラスである。ファイル読み込みに必要な共通機能（パス管理、CORS設定、認証情報、リクエストヘッダー）を提供し、具象ローダー（TextureLoader、GLTFLoader等）が継承して使用する。

**業務上の目的・背景**：3Dアプリケーションでは様々な形式のリソース（テクスチャ、3Dモデル、アニメーション等）を読み込む必要がある。これらのローダーに共通する設定（CORS、パス、認証）を一元管理し、一貫したAPIを提供するために基底クラスが必要である。

**機能の利用シーン**：
- 新しいローダークラスを作成する際の基底クラスとして継承
- すべてのローダーで共通のパス設定
- CORS対応が必要なクロスオリジンリソースの読み込み
- 認証が必要なリソースへのアクセス
- カスタムリクエストヘッダーの設定

**主要な処理内容**：
1. LoadingManagerの管理（デフォルトまたはカスタム）
2. crossOrigin設定（CORS対応）
3. withCredentials設定（認証情報の送信）
4. path/resourcePath設定（ベースパス管理）
5. requestHeader設定（HTTPリクエストヘッダー）
6. load()、loadAsync()、parse()の抽象メソッド定義

**関連システム・外部連携**：LoadingManager、すべての具象ローダークラス

**権限による制御**：特になし

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 10 | Menubar - File | 補助機能 | ファイル読み込み時にローダーの基盤として使用 |

## 機能種別

抽象クラス / リソース読み込み基盤

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| manager | LoadingManager | No | ローディングマネージャー（デフォルト: DefaultLoadingManager） | LoadingManagerインスタンス |

### 入力データソース

継承クラスを通じて提供されるリソースURL/パス

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| manager | LoadingManager | ローディングマネージャーへの参照 |
| crossOrigin | string | CORS設定（デフォルト: 'anonymous'） |
| withCredentials | boolean | 認証情報送信フラグ（デフォルト: false） |
| path | string | アセットのベースパス |
| resourcePath | string | 依存リソースのベースパス |
| requestHeader | Object | HTTPリクエストヘッダー |

### 出力先

継承クラスに機能を提供

## 処理フロー

### 処理シーケンス

```
1. コンストラクタ呼び出し
   └─ LoadingManagerの設定（引数またはDefaultLoadingManager）
2. 共通プロパティ初期化
   └─ crossOrigin = 'anonymous'
   └─ withCredentials = false
   └─ path = ''
   └─ resourcePath = ''
   └─ requestHeader = {}
3. DevToolsへの登録（開発環境の場合）
4. 継承クラスでのオーバーライド
   └─ load()メソッドの実装
   └─ parse()メソッドの実装（必要に応じて）
```

### フローチャート

```mermaid
flowchart TD
    A[開始: new ConcreteLoader] --> B[Loader コンストラクタ]
    B --> C{manager 引数あり?}
    C -->|Yes| D[指定されたmanagerを使用]
    C -->|No| E[DefaultLoadingManagerを使用]
    D --> F[共通プロパティ初期化]
    E --> F
    F --> G[DevTools登録]
    G --> H[終了: Loader インスタンス]

    I[load 呼び出し] --> J[パス解決]
    J --> K[manager.resolveURL]
    K --> L[リソース読み込み]
    L --> M[onLoad コールバック]
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-98-01 | デフォルトマネージャー | manager未指定時はDefaultLoadingManagerを使用 | 常時 |
| BR-98-02 | CORS設定 | crossOriginのデフォルトは'anonymous' | 常時 |
| BR-98-03 | 認証情報 | withCredentialsは同一オリジン以外で有効 | クロスオリジン時 |
| BR-98-04 | パス連結 | load時にpathがURLに連結される | load呼び出し時 |
| BR-98-05 | デフォルトマテリアル名 | Loader.DEFAULT_MATERIAL_NAME = '__DEFAULT' | マテリアル読み込み時 |

### 計算ロジック

URL解決: `finalURL = this.path + url` → `manager.resolveURL(finalURL)`

## データベース操作仕様

該当なし

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | CORS エラー | クロスオリジンリソースでCORS未設定 | crossOrigin設定確認 |
| - | 認証エラー | 認証必要リソースでwithCredentials未設定 | withCredentials = true |
| - | 404 Not Found | リソースが存在しない | パス設定確認 |

### リトライ仕様

基底クラスでは未実装。継承クラスで必要に応じて実装。

## トランザクション仕様

該当なし

## パフォーマンス要件

- ローダーインスタンスは再利用可能
- 同一リソースの重複読み込みはLoadingManagerで管理

## セキュリティ考慮事項

- withCredentialsはクロスオリジン時のセキュリティに影響
- 信頼できないソースからのリソース読み込みに注意

## 備考

- Loader.DEFAULT_MATERIAL_NAMEは静的プロパティとして定義されている
- abort()メソッドは継承クラスでオーバーライド可能

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | LoadingManager.js | `src/loaders/LoadingManager.js` | DefaultLoadingManagerの仕組みを理解 |

**読解のコツ**: LoaderとLoadingManagerの関係を先に理解することが重要。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Loader.js | `src/loaders/Loader.js` | 基底クラスの全メソッドを確認 |

**主要処理フロー**:
1. **8行目**: クラス定義開始 `class Loader`
2. **15行目**: コンストラクタ - manager引数を受け取る
3. **23行目**: manager設定 - DefaultLoadingManagerまたは指定されたmanager
4. **32行目**: crossOrigin = 'anonymous'
5. **40行目**: withCredentials = false
6. **47行目**: path = ''
7. **54行目**: resourcePath = ''
8. **62行目**: requestHeader = {}
9. **64-68行目**: DevToolsへの登録
10. **82行目**: load()メソッド - 抽象メソッド（空実装）
11. **91-101行目**: loadAsync()メソッド - Promiseラッパー
12. **110行目**: parse()メソッド - 抽象メソッド（空実装）
13. **119-124行目**: setCrossOrigin()メソッド
14. **135-140行目**: setWithCredentials()メソッド
15. **148-153行目**: setPath()メソッド
16. **161-166行目**: setResourcePath()メソッド
17. **175-180行目**: setRequestHeader()メソッド
18. **188-192行目**: abort()メソッド
19. **220行目**: 静的プロパティ DEFAULT_MATERIAL_NAME

#### Step 3: 継承クラスでの使用を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | FileLoader.js | `src/loaders/FileLoader.js` | Loaderを継承した具象クラスの例 |
| 3-2 | TextureLoader.js | `src/loaders/TextureLoader.js` | 別の継承例 |

### プログラム呼び出し階層図

```
Loader (基底クラス)
    │
    ├─ LoadingManager (管理)
    │      └─ DefaultLoadingManager (デフォルト)
    │
    ├─ FileLoader (継承)
    │
    ├─ ImageLoader (継承)
    │
    ├─ TextureLoader (継承)
    │      └─ ImageLoader (内部使用)
    │
    └─ ObjectLoader (継承)
           └─ 各種ローダー (内部使用)
```

### データフロー図

```
[入力]                    [処理]                      [出力]

manager (オプション) ───▶  Loader                    ───▶  設定されたLoaderインスタンス
                          (コンストラクタ)
                          │
                          ├─ manager設定
                          │
                          ├─ CORS設定
                          │
                          └─ パス設定

[使用時]

URL                  ───▶  load() / loadAsync()     ───▶  読み込まれたデータ
                           │
                           ├─ path連結
                           │
                           ├─ resolveURL
                           │
                           └─ 実際の読み込み
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Loader.js | `src/loaders/Loader.js` | ソース | ローダー基底クラス |
| LoadingManager.js | `src/loaders/LoadingManager.js` | ソース | 読み込み管理クラス |
| FileLoader.js | `src/loaders/FileLoader.js` | ソース | 汎用ファイルローダー |
| ImageLoader.js | `src/loaders/ImageLoader.js` | ソース | 画像ローダー |
| TextureLoader.js | `src/loaders/TextureLoader.js` | ソース | テクスチャローダー |
| ObjectLoader.js | `src/loaders/ObjectLoader.js` | ソース | Three.js JSONローダー |
