# 通知設計書 11-finished

## 概要

本ドキュメントは、Three.jsのアニメーションシステムにおける `finished` イベント通知の設計を記述する。このイベントは、AnimationActionがアニメーションの再生を完了した際に発火する。

### 本通知の処理概要

この通知は、アニメーションアクションが最終フレームまで再生を完了した際に、AnimationMixer経由でリスナーに通知を行うイベントシステムである。

**業務上の目的・背景**：3Dアプリケーションにおいて、アニメーションの再生完了は重要なイベントである。例えば、キャラクターの攻撃アニメーションが終了したら待機状態に戻る、ドアの開閉アニメーションが完了したら次のシーケンスを開始するなど、アニメーション駆動のロジックを実装するために必要である。この通知により、開発者はアニメーション完了を検知して次のアクションをトリガーできる。

**通知の送信タイミング**：アニメーションアクションが以下の条件を満たした場合に発火する：（1）LoopOnceモードでアニメーションが終端に到達した場合、（2）LoopRepeatまたはLoopPingPongモードで設定されたrepetitions回数を消化した場合。具体的には、AnimationAction._updateTime()メソッド内で時間更新処理中に判定される。

**通知の受信者**：AnimationMixerインスタンスに対して `addEventListener('finished', callback)` でリスナーを登録したすべてのオブジェクト。アプリケーション開発者が登録するコールバック関数がイベントを受信する。

**通知内容の概要**：イベントオブジェクトには、`type`（'finished'）、`action`（終了したAnimationActionインスタンス）、`direction`（再生方向を示す1または-1）が含まれる。

**期待されるアクション**：受信者はイベントを受け取った後、次のアニメーションへの遷移、状態の更新、UIの変更、ゲームロジックの進行など、アプリケーション固有の処理を実行することが期待される。

## 通知種別

イベント通知（EventDispatcher経由のカスタムイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 高 |
| リトライ | なし |

### 送信先決定ロジック

AnimationMixerインスタンスに登録されたすべての'finished'イベントリスナーに対して、登録順に同期的にコールバックが呼び出される。EventDispatcher.dispatchEvent()メソッドがリスナー配列を走査し、各リスナー関数を順次実行する。

## 通知テンプレート

### イベントオブジェクト形式

```javascript
{
  type: 'finished',
  action: AnimationAction,  // 終了したアニメーションアクション
  direction: 1 | -1         // 1: 順方向再生で終了, -1: 逆方向再生で終了
}
```

### イベントリスナー登録例

```javascript
const mixer = new THREE.AnimationMixer(model);
mixer.addEventListener('finished', function(event) {
  console.log('Animation finished:', event.action.getClip().name);
  console.log('Direction:', event.direction);
});
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| type | イベントタイプ識別子 | ハードコード値 'finished' | Yes |
| action | 終了したAnimationActionインスタンス | AnimationAction._updateTime()内のthis参照 | Yes |
| direction | 再生方向（1:順方向, -1:逆方向） | deltaTime > 0 ? 1 : -1 の計算結果 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 内部処理 | AnimationMixer.update()呼び出し | アニメーション時間が終端に到達 | LoopOnceモードで duration または 0 に到達 |
| 内部処理 | AnimationMixer.update()呼び出し | 反復回数を消化完了 | LoopRepeat/LoopPingPongモードで repetitions 回完了 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| action.enabled === false | アクションが無効化されている場合、イベントは発火しない |
| deltaTime === 0 | 時間が進んでいない場合、終了判定は行われない |
| ループ継続中 | repetitions に達していない場合は 'loop' イベントが発火 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[AnimationMixer.update呼び出し] --> B[AnimationAction._update呼び出し]
    B --> C[_updateTime実行]
    C --> D{終了条件チェック}
    D -->|LoopOnce終端到達| E[clampWhenFinishedチェック]
    D -->|反復回数消化| E
    D -->|継続| F[通常再生継続]
    E -->|true| G[paused = true]
    E -->|false| H[enabled = false]
    G --> I[finishedイベント発火]
    H --> I
    I --> J[dispatchEvent実行]
    J --> K[リスナーコールバック呼び出し]
    K --> L[終了]
    F --> L
```

## データベース参照・更新仕様

本通知はデータベースを使用しない。すべての状態はメモリ上のオブジェクト内で管理される。

### 参照オブジェクト一覧

| オブジェクト | 用途 | 備考 |
|------------|------|------|
| AnimationAction | アニメーション状態の参照 | time, loop, repetitions等 |
| AnimationMixer._listeners | リスナー一覧の参照 | EventDispatcher継承 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー内例外 | リスナーコールバック内で例外発生 | 例外は伝播し、後続のリスナーは呼び出されない可能性がある |
| 無効なアクション | アクションが破棄済み | 通常は発生しない（内部管理による） |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（フレームレートに依存） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。AnimationMixer.update()が呼び出されるフレームごとに判定される。

## セキュリティ考慮事項

- イベントリスナーはアプリケーションコードによって登録されるため、信頼されたコードのみがイベントを受信する
- イベントオブジェクトにはAnimationActionへの参照が含まれるため、リスナー内でアクションの状態を変更可能

## 備考

- finishedイベント発火後、clampWhenFinished=trueの場合はアクションが一時停止状態（paused=true）となり、最終フレームで維持される
- clampWhenFinished=falseの場合はアクションが無効化（enabled=false）され、アニメーションは影響を与えなくなる
- 同一アニメーションを再度再生する場合は、action.reset()を呼び出す必要がある

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、イベントシステムの基盤となるEventDispatcherと、イベントオブジェクトの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | EventDispatcher.js | `src/core/EventDispatcher.js` | addEventListener/dispatchEventの仕組み |

**読解のコツ**: EventDispatcherは_listenersオブジェクトでイベントタイプごとにリスナー配列を管理する。dispatchEventはリスナー配列をコピーしてから走査するため、イベント処理中のリスナー追加・削除に対して安全である。

#### Step 2: エントリーポイントを理解する

イベント発火のトリガーとなるAnimationMixer.update()メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AnimationMixer.js | `src/animation/AnimationMixer.js` | update()メソッドがアクションを更新する流れ |

**主要処理フロー**:
1. **676-678行目**: deltaTimeにtimeScaleを乗算してグローバル時間を更新
2. **690-696行目**: アクティブなアクションに対して_update()を呼び出し
3. **700-707行目**: バインディングを適用してシーングラフを更新

#### Step 3: イベント発火ロジックを理解する

finishedイベントが発火する具体的な条件と処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AnimationAction.js | `src/animation/AnimationAction.js` | _updateTime()メソッド内の終了判定ロジック |

**主要処理フロー**:
- **742-767行目**: LoopOnceモードでの終了判定とイベント発火
- **805-819行目**: LoopRepeat/LoopPingPongモードでの終了判定とイベント発火
- **763-766行目**: dispatchEvent呼び出しでfinishedイベントを発火

### プログラム呼び出し階層図

```
AnimationMixer.update(deltaTime)
    |
    +-- AnimationAction._update(time, deltaTime, timeDirection, accuIndex)
           |
           +-- AnimationAction._updateTime(deltaTime)
                  |
                  +-- [終了条件判定]
                  |      |
                  |      +-- LoopOnce: time >= duration または time < 0
                  |      |
                  |      +-- LoopRepeat/PingPong: repetitions - loopCount <= 0
                  |
                  +-- AnimationMixer.dispatchEvent({ type: 'finished', action, direction })
                         |
                         +-- EventDispatcher.dispatchEvent(event)
                                |
                                +-- リスナーコールバック呼び出し
```

### データフロー図

```
[入力]                      [処理]                           [出力]

deltaTime ──────────> AnimationMixer.update() ─────> 更新されたシーン状態
                           |
                           v
                    AnimationAction._update()
                           |
                           v
                    AnimationAction._updateTime()
                           |
                           v
                    [終了条件判定]
                           |
                     ┌─────┴─────┐
                     |           |
                     v           v
               終了判定OK    継続
                     |
                     v
              finishedイベント ────────> リスナーコールバック
              {
                type: 'finished',
                action: this,
                direction: 1 or -1
              }
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AnimationAction.js | `src/animation/AnimationAction.js` | ソース | アニメーションアクションの状態管理とイベント発火 |
| AnimationMixer.js | `src/animation/AnimationMixer.js` | ソース | アニメーション再生の統括とイベントディスパッチ |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベントシステムの基盤クラス |
| AnimationClip.js | `src/animation/AnimationClip.js` | ソース | アニメーションデータの保持 |
| constants.js | `src/constants.js` | ソース | LoopOnce, LoopRepeat, LoopPingPong定数の定義 |
