# 通知設計書 12-loop

## 概要

本ドキュメントは、Three.jsのアニメーションシステムにおける `loop` イベント通知の設計を記述する。このイベントは、AnimationActionがアニメーションの1サイクルを完了し、次のループに入る際に発火する。

### 本通知の処理概要

この通知は、アニメーションアクションがループ再生中に1サイクル分の再生を完了し、次のループ（反復）を開始する際に、AnimationMixer経由でリスナーに通知を行うイベントシステムである。

**業務上の目的・背景**：ループアニメーションにおいて、各ループの境界を検知することは様々なユースケースで重要である。例えば、キャラクターの歩行サイクルごとに足音を再生する、アニメーションのループ回数をカウントする、特定のループ回数に達したら処理を変更するなど。この通知により、開発者はアニメーションの反復タイミングを正確に把握し、それに連動した処理を実装できる。

**通知の送信タイミング**：アニメーションアクションがLoopRepeatまたはLoopPingPongモードで再生中に、アニメーション時間がdurationを超えて次のループに折り返す瞬間（または逆再生時に0を下回る瞬間）に発火する。具体的には、AnimationAction._updateTime()メソッド内でループ境界を跨ぐ時間更新が行われた際に判定される。

**通知の受信者**：AnimationMixerインスタンスに対して `addEventListener('loop', callback)` でリスナーを登録したすべてのオブジェクト。アプリケーション開発者が登録するコールバック関数がイベントを受信する。

**通知内容の概要**：イベントオブジェクトには、`type`（'loop'）、`action`（ループしたAnimationActionインスタンス）、`loopDelta`（今回のループで進んだループ回数、正負の値）が含まれる。

**期待されるアクション**：受信者はイベントを受け取った後、サウンドエフェクトの再生、パーティクルのスポーン、ループカウンターの更新、条件に基づく状態変更など、アプリケーション固有の処理を実行することが期待される。

## 通知種別

イベント通知（EventDispatcher経由のカスタムイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 高 |
| リトライ | なし |

### 送信先決定ロジック

AnimationMixerインスタンスに登録されたすべての'loop'イベントリスナーに対して、登録順に同期的にコールバックが呼び出される。EventDispatcher.dispatchEvent()メソッドがリスナー配列を走査し、各リスナー関数を順次実行する。

## 通知テンプレート

### イベントオブジェクト形式

```javascript
{
  type: 'loop',
  action: AnimationAction,  // ループしたアニメーションアクション
  loopDelta: number         // ループ回数の変化量（符号付き整数）
}
```

### イベントリスナー登録例

```javascript
const mixer = new THREE.AnimationMixer(model);
mixer.addEventListener('loop', function(event) {
  console.log('Animation looped:', event.action.getClip().name);
  console.log('Loop delta:', event.loopDelta);
});
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| type | イベントタイプ識別子 | ハードコード値 'loop' | Yes |
| action | ループしたAnimationActionインスタンス | AnimationAction._updateTime()内のthis参照 | Yes |
| loopDelta | ループ回数の変化量 | Math.floor(time / duration) の計算結果 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 内部処理 | AnimationMixer.update()呼び出し | time >= duration かつ loop !== LoopOnce かつ 反復継続中 | 順方向再生でのループ |
| 内部処理 | AnimationMixer.update()呼び出し | time < 0 かつ loop !== LoopOnce かつ 反復継続中 | 逆方向再生でのループ |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| loop === LoopOnce | LoopOnceモードではloopイベントは発火しない（代わりにfinishedが発火） |
| action.enabled === false | アクションが無効化されている場合、イベントは発火しない |
| repetitions消化完了 | repetitionsに達した場合はfinishedイベントが発火 |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[AnimationMixer.update呼び出し] --> B[AnimationAction._update呼び出し]
    B --> C[_updateTime実行]
    C --> D{LoopOnceモード?}
    D -->|Yes| E[finishedイベント処理へ]
    D -->|No| F{time >= duration または time < 0?}
    F -->|No| G[通常再生継続]
    F -->|Yes| H[loopDelta計算]
    H --> I{repetitions消化完了?}
    I -->|Yes| J[finishedイベント発火]
    I -->|No| K[_loopCount更新]
    K --> L[loopイベント発火]
    L --> M[dispatchEvent実行]
    M --> N[リスナーコールバック呼び出し]
    N --> O[終了]
    G --> O
    E --> O
    J --> O
```

## データベース参照・更新仕様

本通知はデータベースを使用しない。すべての状態はメモリ上のオブジェクト内で管理される。

### 参照オブジェクト一覧

| オブジェクト | 用途 | 備考 |
|------------|------|------|
| AnimationAction | アニメーション状態の参照 | time, loop, repetitions, _loopCount等 |
| AnimationMixer._listeners | リスナー一覧の参照 | EventDispatcher継承 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー内例外 | リスナーコールバック内で例外発生 | 例外は伝播し、後続のリスナーは呼び出されない可能性がある |
| 無効なアクション | アクションが破棄済み | 通常は発生しない（内部管理による） |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（アニメーション速度に依存） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。AnimationMixer.update()が呼び出されるフレームごとに判定される。

## セキュリティ考慮事項

- イベントリスナーはアプリケーションコードによって登録されるため、信頼されたコードのみがイベントを受信する
- イベントオブジェクトにはAnimationActionへの参照が含まれるため、リスナー内でアクションの状態を変更可能

## 備考

- loopDeltaは通常1または-1だが、大きなdeltaTimeが渡された場合は複数ループを一度にスキップする可能性があり、その場合は絶対値が1より大きくなる
- LoopPingPongモードでは、アニメーションの再生方向が各ループで反転する
- loopイベント発火後、_loopCountが更新され、repetitionsに達すると次回のループ境界でfinishedイベントが発火する

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ループ関連のプロパティとイベントオブジェクトの構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | AnimationAction.js | `src/animation/AnimationAction.js` | loop, repetitions, _loopCountプロパティの定義 |
| 1-2 | constants.js | `src/constants.js` | LoopOnce, LoopRepeat, LoopPingPong定数 |

**読解のコツ**: AnimationActionのコンストラクタで初期化されるloop=LoopRepeat（デフォルト）、repetitions=Infinity、_loopCount=-1を確認する。_loopCountは-1で初期化され、最初のループ開始時に0になる。

#### Step 2: エントリーポイントを理解する

イベント発火のトリガーとなるAnimationMixer.update()メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | AnimationMixer.js | `src/animation/AnimationMixer.js` | update()メソッドがアクションを更新する流れ |

**主要処理フロー**:
1. **676-678行目**: deltaTimeにtimeScaleを乗算してグローバル時間を更新
2. **690-696行目**: アクティブなアクションに対して_update()を呼び出し

#### Step 3: ループイベント発火ロジックを理解する

loopイベントが発火する具体的な条件と処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | AnimationAction.js | `src/animation/AnimationAction.js` | _updateTime()メソッド内のループ処理ロジック |

**主要処理フロー**:
- **770-792行目**: ループモード初期化処理（_loopCount === -1の場合）
- **794-846行目**: ループ境界を跨ぐ時間の処理
- **798-801行目**: loopDelta計算とtime調整
- **821-846行目**: ループ継続時の処理とイベント発火
- **842-844行目**: dispatchEvent呼び出しでloopイベントを発火

### プログラム呼び出し階層図

```
AnimationMixer.update(deltaTime)
    |
    +-- AnimationAction._update(time, deltaTime, timeDirection, accuIndex)
           |
           +-- AnimationAction._updateTime(deltaTime)
                  |
                  +-- [LoopOnce以外のモードの場合]
                  |      |
                  |      +-- time >= duration または time < 0 の判定
                  |      |
                  |      +-- loopDelta = Math.floor(time / duration)
                  |      |
                  |      +-- loopCount += Math.abs(loopDelta)
                  |      |
                  |      +-- repetitions - loopCount > 0 の判定（継続判定）
                  |
                  +-- AnimationMixer.dispatchEvent({ type: 'loop', action, loopDelta })
                         |
                         +-- EventDispatcher.dispatchEvent(event)
                                |
                                +-- リスナーコールバック呼び出し
```

### データフロー図

```
[入力]                      [処理]                           [出力]

deltaTime ──────────> AnimationMixer.update() ─────> 更新されたシーン状態
                           |
                           v
                    AnimationAction._update()
                           |
                           v
                    AnimationAction._updateTime()
                           |
                           v
                    [ループ境界判定]
                           |
                     ┌─────┴─────┐
                     |           |
                     v           v
              ループ継続     反復完了
                     |           |
                     v           v
              loopイベント   finishedイベント
              {
                type: 'loop',
                action: this,
                loopDelta: n
              }
                     |
                     v
              リスナーコールバック
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| AnimationAction.js | `src/animation/AnimationAction.js` | ソース | アニメーションアクションの状態管理とイベント発火 |
| AnimationMixer.js | `src/animation/AnimationMixer.js` | ソース | アニメーション再生の統括とイベントディスパッチ |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベントシステムの基盤クラス |
| constants.js | `src/constants.js` | ソース | LoopOnce, LoopRepeat, LoopPingPong定数の定義 |
