# 通知設計書 13-change

## 概要

本ドキュメントは、Three.jsのノードシステムにおける `change` イベント通知の設計を記述する。このイベントは、ScriptableValueNodeの値が変更された際に発火する。

### 本通知の処理概要

この通知は、ScriptableValueNodeクラスのvalueプロパティが変更された際に、EventDispatcher経由でリスナーに変更を通知するイベントシステムである。

**業務上の目的・背景**：ノードベースのマテリアルシステムやビジュアルプログラミング環境において、値の変更を検知して依存する処理を更新することは重要である。例えば、ユーザーがパラメータを調整した際にマテリアルの再コンパイルをトリガーする、エディタUIを更新する、依存するノードグラフを再評価するなど。この通知により、開発者は値の変更に反応したリアクティブな処理を実装できる。

**通知の送信タイミング**：ScriptableValueNodeのvalueセッター経由で値が変更された際に発火する。具体的には、新しい値が現在の値と異なる場合（this._value !== val）にのみイベントが発火する。同一の値を設定した場合はイベントは発火しない。

**通知の受信者**：ScriptableValueNodeインスタンスのeventsプロパティ（EventDispatcherインスタンス）に対して `addEventListener('change', callback)` でリスナーを登録したすべてのオブジェクト。

**通知内容の概要**：イベントオブジェクトには、`type`（'change'）のみが含まれる。変更前後の値は含まれないため、必要な場合はリスナー側でノードのvalueプロパティを参照する必要がある。

**期待されるアクション**：受信者はイベントを受け取った後、ノードグラフの再評価、シェーダーの再コンパイル、UIの更新など、値の変更に応じた処理を実行することが期待される。

## 通知種別

イベント通知（EventDispatcher経由のカスタムイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 中 |
| リトライ | なし |

### 送信先決定ロジック

ScriptableValueNode.eventsプロパティ（EventDispatcherインスタンス）に登録されたすべての'change'イベントリスナーに対して、登録順に同期的にコールバックが呼び出される。

## 通知テンプレート

### イベントオブジェクト形式

```javascript
{
  type: 'change'
}
```

### イベントリスナー登録例

```javascript
const valueNode = new ScriptableValueNode(initialValue);
valueNode.events.addEventListener('change', function(event) {
  console.log('Value changed to:', valueNode.value);
});
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| type | イベントタイプ識別子 | ハードコード値 'change' | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| プロパティ変更 | valueセッター呼び出し | this._value !== val | 新しい値が現在の値と異なる場合 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| 値が同一 | this._value === val の場合、イベントは発火しない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[valueセッター呼び出し] --> B{値が変更された?}
    B -->|No| C[早期リターン]
    B -->|Yes| D{inputType === 'URL' かつ キャッシュあり?}
    D -->|Yes| E[URL.revokeObjectURL呼び出し]
    D -->|No| F[_value更新]
    E --> G[_cache = null]
    G --> F
    F --> H[changeイベント発火]
    H --> I[dispatchEvent実行]
    I --> J[リスナーコールバック呼び出し]
    J --> K[refresh呼び出し]
    K --> L[終了]
    C --> L
```

## データベース参照・更新仕様

本通知はデータベースを使用しない。すべての状態はメモリ上のオブジェクト内で管理される。

### 参照オブジェクト一覧

| オブジェクト | 用途 | 備考 |
|------------|------|------|
| ScriptableValueNode._value | 値の参照・比較 | プライベートプロパティ |
| ScriptableValueNode._cache | URLキャッシュの管理 | inputType='URL'の場合に使用 |
| ScriptableValueNode.events._listeners | リスナー一覧の参照 | EventDispatcherインスタンス |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー内例外 | リスナーコールバック内で例外発生 | 例外は伝播し、refresh()の呼び出しが行われない可能性がある |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。valueプロパティが変更されるたびに発火。

## セキュリティ考慮事項

- イベントリスナーはアプリケーションコードによって登録されるため、信頼されたコードのみがイベントを受信する
- inputType='URL'の場合、変更時に古いObject URLがrevokeされるため、メモリリークを防止

## 備考

- changeイベント発火後、自動的にrefresh()メソッドが呼び出され、'refresh'イベントも発火する
- inputType='URL'かつ値がArrayBufferの場合、古いキャッシュされたObject URLが自動的に解放される
- 値の変更検知は厳密等価（===）で行われる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ScriptableValueNodeのプロパティ構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | _value, _cache, inputType, events プロパティの定義 |
| 1-2 | EventDispatcher.js | `src/core/EventDispatcher.js` | イベントシステムの基盤 |

**読解のコツ**: ScriptableValueNodeはNodeを継承し、eventsプロパティとしてEventDispatcherインスタンスを持つ。値の管理は_valueプライベートプロパティで行い、publicなvalueプロパティはゲッター/セッターで制御される。

#### Step 2: イベント発火ロジックを理解する

changeイベントが発火する具体的な条件と処理を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | valueセッター（92-110行目） |

**主要処理フロー**:
- **94行目**: 値の同一性チェック（早期リターン）
- **96-101行目**: URL型でキャッシュがある場合のクリーンアップ
- **104行目**: _valueの更新
- **106行目**: changeイベントの発火
- **108行目**: refresh()メソッドの呼び出し

### プログラム呼び出し階層図

```
ScriptableValueNode.value = newValue （セッター呼び出し）
    |
    +-- [値の同一性チェック]
    |      |
    |      +-- 同一なら早期リターン
    |
    +-- [URLキャッシュクリーンアップ（条件付き）]
    |      |
    |      +-- URL.revokeObjectURL(_cache)
    |      +-- _cache = null
    |
    +-- this._value = val
    |
    +-- this.events.dispatchEvent({ type: 'change' })
    |      |
    |      +-- EventDispatcher.dispatchEvent(event)
    |             |
    |             +-- リスナーコールバック呼び出し
    |
    +-- this.refresh()
           |
           +-- this.events.dispatchEvent({ type: 'refresh' })
```

### データフロー図

```
[入力]                      [処理]                           [出力]

新しい値 ─────────> valueセッター ─────────> 更新された_value
                         |
                         v
                  [同一性チェック]
                         |
                   ┌─────┴─────┐
                   |           |
                   v           v
                同一         異なる
                   |           |
                   v           v
                終了    changeイベント発火
                              |
                              v
                        リスナーコールバック
                              |
                              v
                        refreshイベント発火
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | ソース | 値ノードの実装とイベント発火 |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベントシステムの基盤クラス |
| Node.js | `src/nodes/core/Node.js` | ソース | ノードの基底クラス |
| ScriptableNode.js | `src/nodes/code/ScriptableNode.js` | ソース | ScriptableValueNodeを使用するノード |
