# 通知設計書 14-refresh

## 概要

本ドキュメントは、Three.jsのノードシステムにおける `refresh` イベント通知の設計を記述する。このイベントは、ScriptableValueNodeがリフレッシュされた際に発火する。

### 本通知の処理概要

この通知は、ScriptableValueNodeクラスのrefresh()メソッドが呼び出された際に、EventDispatcher経由でリスナーに更新を通知するイベントシステムである。

**業務上の目的・背景**：ノードベースのシステムにおいて、ノードの状態が更新されたことを依存するコンポーネントに通知することは重要である。refreshイベントは、値の変更だけでなく、ノードの再評価が必要なあらゆる状況で使用される。例えば、ノードグラフのキャッシュ無効化、レンダリングパイプラインの更新トリガー、エディタUIの再描画など。この通知により、開発者はノードの状態更新に反応した処理を実装できる。

**通知の送信タイミング**：ScriptableValueNodeのrefresh()メソッドが呼び出された際に発火する。主に以下の2つのケースがある：（1）valueセッター経由で値が変更された後に自動的に呼び出される、（2）外部コードからrefresh()メソッドが直接呼び出される。

**通知の受信者**：ScriptableValueNodeインスタンスのeventsプロパティ（EventDispatcherインスタンス）に対して `addEventListener('refresh', callback)` でリスナーを登録したすべてのオブジェクト。

**通知内容の概要**：イベントオブジェクトには、`type`（'refresh'）のみが含まれる。リフレッシュの原因（値変更や外部呼び出し）は含まれないため、必要な場合はリスナー側で判断する必要がある。

**期待されるアクション**：受信者はイベントを受け取った後、キャッシュの無効化、依存するノードの再評価、UIの更新など、ノード状態の更新に応じた処理を実行することが期待される。

## 通知種別

イベント通知（EventDispatcher経由のカスタムイベント）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 中 |
| リトライ | なし |

### 送信先決定ロジック

ScriptableValueNode.eventsプロパティ（EventDispatcherインスタンス）に登録されたすべての'refresh'イベントリスナーに対して、登録順に同期的にコールバックが呼び出される。

## 通知テンプレート

### イベントオブジェクト形式

```javascript
{
  type: 'refresh'
}
```

### イベントリスナー登録例

```javascript
const valueNode = new ScriptableValueNode(initialValue);
valueNode.events.addEventListener('refresh', function(event) {
  console.log('Node refreshed, current value:', valueNode.value);
  // ノードグラフの再評価などの処理
});
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| type | イベントタイプ識別子 | ハードコード値 'refresh' | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 内部処理 | valueセッター完了 | 値が変更された場合 | changeイベント発火後に自動呼び出し |
| 外部呼び出し | refresh()メソッド呼び出し | 常に | 外部コードからの明示的な呼び出し |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| なし | refresh()メソッドが呼び出されると常にイベントが発火する |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[refresh呼び出し] --> B[refreshイベント発火]
    B --> C[dispatchEvent実行]
    C --> D[リスナーコールバック呼び出し]
    D --> E[終了]
```

### 値変更時の完全なフロー

```mermaid
flowchart TD
    A[valueセッター呼び出し] --> B{値が変更された?}
    B -->|No| C[早期リターン]
    B -->|Yes| D[_value更新]
    D --> E[changeイベント発火]
    E --> F[changeリスナー呼び出し]
    F --> G[refresh呼び出し]
    G --> H[refreshイベント発火]
    H --> I[refreshリスナー呼び出し]
    I --> J[終了]
    C --> J
```

## データベース参照・更新仕様

本通知はデータベースを使用しない。すべての状態はメモリ上のオブジェクト内で管理される。

### 参照オブジェクト一覧

| オブジェクト | 用途 | 備考 |
|------------|------|------|
| ScriptableValueNode.events._listeners | リスナー一覧の参照 | EventDispatcherインスタンス |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| リスナー内例外 | リスナーコールバック内で例外発生 | 例外は伝播し、後続のリスナーは呼び出されない可能性がある |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。refresh()メソッドが呼び出されるたびに発火。

## セキュリティ考慮事項

- イベントリスナーはアプリケーションコードによって登録されるため、信頼されたコードのみがイベントを受信する

## 備考

- refreshイベントは常にchangeイベントの後に発火する（値変更の場合）
- refresh()メソッドは値変更なしでも直接呼び出し可能であり、その場合はrefreshイベントのみが発火する
- refreshイベントはノードの再評価をトリガーするために使用されることが多い

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、ScriptableValueNodeのプロパティ構造とrefresh()メソッドを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | refresh()メソッドの定義（126-130行目） |
| 1-2 | EventDispatcher.js | `src/core/EventDispatcher.js` | イベントシステムの基盤 |

**読解のコツ**: refresh()メソッドは非常にシンプルで、eventsプロパティのdispatchEventを呼び出すだけ。値変更時にはvalueセッターから自動的に呼び出される。

#### Step 2: イベント発火ロジックを理解する

refreshイベントが発火するコンテキストを理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | valueセッターからのrefresh()呼び出し（108行目） |
| 2-2 | ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | refresh()メソッド定義（126-130行目） |

**主要処理フロー**:
- **108行目**: valueセッター内でrefresh()が呼び出される
- **128行目**: refresh()内でrefreshイベントが発火

### プログラム呼び出し階層図

```
[値変更時]
ScriptableValueNode.value = newValue （セッター呼び出し）
    |
    +-- this._value = val
    |
    +-- this.events.dispatchEvent({ type: 'change' })
    |
    +-- this.refresh()
           |
           +-- this.events.dispatchEvent({ type: 'refresh' })
                  |
                  +-- EventDispatcher.dispatchEvent(event)
                         |
                         +-- リスナーコールバック呼び出し

[直接呼び出し時]
ScriptableValueNode.refresh()
    |
    +-- this.events.dispatchEvent({ type: 'refresh' })
           |
           +-- EventDispatcher.dispatchEvent(event)
                  |
                  +-- リスナーコールバック呼び出し
```

### データフロー図

```
[入力]                      [処理]                           [出力]

refresh()呼び出し ──> events.dispatchEvent() ──> リスナーコールバック
     ^                                                    |
     |                                                    v
値変更 ───────────────────────────────────────> 依存処理の更新
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| ScriptableValueNode.js | `src/nodes/code/ScriptableValueNode.js` | ソース | 値ノードの実装とイベント発火 |
| EventDispatcher.js | `src/core/EventDispatcher.js` | ソース | イベントシステムの基盤クラス |
| Node.js | `src/nodes/core/Node.js` | ソース | ノードの基底クラス |
| ScriptableNode.js | `src/nodes/code/ScriptableNode.js` | ソース | ScriptableValueNodeを使用するノード |
