# 通知設計書 18-warnOnce

## 概要

本ドキュメントは、Three.jsのユーティリティ関数における `warnOnce` 通知（重複防止警告出力）の設計を記述する。この通知は、同一の警告メッセージを1回のみコンソールに出力する。

### 本通知の処理概要

この通知は、Three.jsライブラリ内部から警告メッセージを出力するためのユーティリティ関数であり、同一のメッセージは1回のみ出力される。内部キャッシュにより重複を防止する。

**業務上の目的・背景**：3Dアプリケーション開発において、ループ処理や頻繁に呼び出される関数内で警告を出力すると、コンソールが大量の同一メッセージで埋め尽くされることがある。warnOnce関数は、このような状況を防ぎ、開発者が重要な警告を見逃さないようにするために使用される。例えば、非推奨のAPIが毎フレーム呼び出される場合でも、警告は1回だけ表示される。

**通知の送信タイミング**：Three.jsライブラリ内部のコードからwarnOnce()関数が呼び出された際に、そのメッセージが初回であれば発火する。2回目以降の同一メッセージは抑止される。

**通知の受信者**：ブラウザのコンソール（Developer Tools）。内部的にはwarn()関数を呼び出すため、setConsoleFunction()でカスタム関数が設定されている場合は、そのカスタム関数がメッセージを受信する。

**通知内容の概要**：初回のメッセージのみがwarn()関数経由でコンソールに出力される。

**期待されるアクション**：開発者はコンソールで警告を確認し、問題を修正する。同一警告が繰り返されないため、コンソールの可読性が維持される。

## 通知種別

コンソール出力（warn関数経由、重複防止機能付き）

## 送信仕様

### 基本情報

| 項目 | 内容 |
|-----|------|
| 送信方式 | 同期 |
| 優先度 | 中（警告レベル） |
| リトライ | なし |

### 送信先決定ロジック

1. メッセージ全体を結合してキャッシュキーを生成
2. キャッシュに存在する場合は早期リターン（出力なし）
3. キャッシュに存在しない場合はキャッシュに追加してwarn()関数を呼び出す

## 通知テンプレート

### 出力形式

```javascript
// 内部的にwarn()関数を呼び出す
warn(...params)
// 結果として console.warn('THREE.' + params[0], ...rest) が実行される
```

### 使用例

```javascript
import { warnOnce } from 'three';

// 毎フレーム呼び出されても、警告は1回だけ出力される
function render() {
  if (deprecatedFeatureUsed) {
    warnOnce('DeprecatedFeature:', 'This feature will be removed in the next version.');
  }
  // ...
}
```

## テンプレート変数

| 変数名 | 説明 | データ取得元 | 必須 |
|--------|------|-------------|-----|
| message | キャッシュキー（引数全体を結合） | params.join(' ') | Yes |
| params | warn()に渡される引数 | 関数引数 | Yes |

## 送信トリガー・条件

### トリガー一覧

| トリガー種別 | トリガーイベント | 送信条件 | 説明 |
|------------|----------------|---------|------|
| 関数呼び出し | warnOnce()関数呼び出し | message not in _cache | 初回のメッセージのみ出力 |

### 送信抑止条件

| 条件 | 説明 |
|-----|------|
| メッセージがキャッシュ済み | 同一メッセージは2回目以降は出力されない |

## 処理フロー

### 送信フロー

```mermaid
flowchart TD
    A[warnOnce関数呼び出し] --> B[メッセージ結合]
    B --> C["message = params.join(' ')"]
    C --> D{キャッシュに存在?}
    D -->|Yes| E[早期リターン]
    D -->|No| F[キャッシュに追加]
    F --> G["_cache[message] = true"]
    G --> H[warn関数呼び出し]
    H --> I[終了]
    E --> I
```

## データベース参照・更新仕様

本通知はデータベースを使用しない。キャッシュはメモリ上のオブジェクト（_cache）で管理される。

### 参照オブジェクト一覧

| オブジェクト | 用途 | 備考 |
|------------|------|------|
| _cache | 出力済みメッセージのキャッシュ | オブジェクト形式、キーはメッセージ文字列 |

## エラー処理

### エラーケース一覧

| エラー種別 | 発生条件 | 対処方法 |
|----------|---------|---------|
| warn関数内例外 | warn関数が例外を発生 | 例外は呼び出し元に伝播 |

### リトライ仕様

| 項目 | 内容 |
|-----|------|
| リトライ回数 | なし |
| リトライ間隔 | N/A |
| リトライ対象エラー | N/A |

## 配信設定

### レート制限

| 項目 | 内容 |
|-----|------|
| 1分あたり上限 | なし（重複防止機能が実質的なレート制限として機能） |
| 1日あたり上限 | なし |

### 配信時間帯

制限なし。

## セキュリティ考慮事項

- キャッシュはメモリ上に永続的に保持されるため、大量のユニークなメッセージでメモリリークの可能性がある
- 実際にはライブラリ内部で使用されるため、メッセージ数は限定的

## 備考

- キャッシュはページのライフサイクル中永続する（クリア機能なし）
- ページのリロードでキャッシュはクリアされる
- メッセージの比較は引数を空白で結合した文字列で行われる

---

## コードリーディングガイド

本通知を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

まず、キャッシュの管理構造を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | utils.js | `src/utils.js` | _cache変数の定義（158行目） |

**読解のコツ**: _cacheは単純なオブジェクトで、メッセージ文字列をキーとし、値はtrue。in演算子でキャッシュの存在を確認する。

#### Step 2: warnOnce関数の実装を理解する

warnOnce関数の実装と動作を理解する。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | utils.js | `src/utils.js` | warnOnce関数（282-292行目） |

**主要処理フロー**:
- **284行目**: メッセージ結合（params.join(' ')）
- **286行目**: キャッシュ存在チェック（早期リターン）
- **288行目**: キャッシュに追加
- **290行目**: warn関数呼び出し

### プログラム呼び出し階層図

```
warnOnce(...params)
    |
    +-- const message = params.join(' ')
    |
    +-- [キャッシュチェック]
    |      |
    |      +-- message in _cache → 早期リターン
    |
    +-- _cache[message] = true
    |
    +-- warn(...params)
           |
           +-- const message = 'THREE.' + params.shift()
           |
           +-- console.warn(message, ...params)
```

### データフロー図

```
[入力]                      [処理]                           [出力]

params ─────────> warnOnce() ─────────────> warn() → コンソール
                    |                          ^
                    v                          |
            [キャッシュチェック]                |
                    |                          |
              ┌─────┴─────┐                   |
              |           |                   |
              v           v                   |
          存在あり    存在なし               |
              |           |                   |
              v           v                   |
            終了      キャッシュ追加 ─────────┘
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| utils.js | `src/utils.js` | ソース | warnOnce/warn関数の定義 |
| Three.Core.js | `src/Three.Core.js` | ソース | utilsのエクスポート |
