# API設計書

## 概要

本ドキュメントは、VBCorLibライブラリの公開API（クラス、インターフェース、メソッド）について記載しています。VBCorLibは.NET Frameworkの主要なクラスをVB6/VBAで再実装したライブラリであり、コレクション、暗号化、エンコーディング、ファイル操作、グローバリゼーションなど幅広い機能を提供します。

> **注意**: VBCorLibはREST APIではなく、COMライブラリとして提供されるVB6/VBAライブラリです。本ドキュメントでは、ライブラリの公開クラスとメソッドを「API」として文書化しています。

## 共通仕様

### ライブラリ参照

```
TypeLibrary: CorType.tlb (VB.EXT: CorLib Type Library 4.0)
UUID: 1705040f-4494-4ab3-8a97-a9f84820636d
Version: 4.0
```

### 使用方法

VB6/VBAプロジェクトから参照設定で「VBCorLib」を追加して使用します。

### 共通インターフェース

| インターフェース名 | 説明 |
| --- | --- |
| IObject | オブジェクトの基本機能（Equals, GetHashCode, ToString）を定義 |
| ICloneable | オブジェクトの複製機能を定義 |
| IComparable | オブジェクトの比較機能を定義 |
| IFormattable | 書式指定による文字列変換機能を定義 |
| IFormatProvider | カルチャ固有の書式情報を提供 |
| ICollection | コレクションの基本機能を定義 |
| IEnumerable | For Each文でのイテレーション機能を定義 |
| IEnumerator | イテレータの基本機能を定義 |
| IList | リストの基本機能を定義 |
| IDictionary | ディクショナリの基本機能を定義 |
| IComparer | オブジェクト比較機能を定義 |
| IEqualityComparer | 等価性比較機能を定義 |

### 共通エラー

| エラーコード | 定数名 | 説明 |
| --- | --- | --- |
| 5 | vbInvalidProcedureCall | 無効なプロシージャ呼び出し |
| 6 | vbOverflow | オーバーフロー |
| 7 | vbOutOfMemory | メモリ不足 |
| 9 | vbSubscriptOutOfRange | 添え字が範囲外 |
| 11 | vbDivisionByZero | ゼロ除算 |
| 13 | vbTypeMismatch | 型不一致 |
| 53 | vbFileNotFound | ファイルが見つからない |
| 76 | vbPathNotFound | パスが見つからない |
| 91 | vbObjectNotSet | オブジェクトが設定されていない |

## API一覧

| カテゴリ | クラス/モジュール | 説明 |
| --- | --- | --- |
| System | CorArray | 配列操作ユーティリティ |
| System | CorString | 文字列操作ユーティリティ |
| System | CorDateTime | 日付時刻操作 |
| System | TimeSpan | 時間間隔表現 |
| System | TimeZone | タイムゾーン情報 |
| System | Version | バージョン情報 |
| System | Guid | GUID操作 |
| System | Random | 乱数生成 |
| System | BitConverter | バイト配列変換 |
| System | Buffer | バイトバッファ操作 |
| System | Convert | 型変換ユーティリティ |
| System | Environment | 環境変数アクセス |
| System | Console | コンソール入出力 |
| System | OperatingSystem | OS情報 |
| System.Collections | ArrayList | 動的配列 |
| System.Collections | Hashtable | ハッシュテーブル |
| System.Collections | SortedList | ソート済みリスト |
| System.Collections | Queue | キュー（FIFO） |
| System.Collections | Stack | スタック（LIFO） |
| System.Collections | BitArray | ビット配列 |
| System.Text | StringBuilder | 文字列構築 |
| System.Text | Encoding | エンコーディング |
| System.Text | UTF8Encoding | UTF-8エンコーディング |
| System.Text | UTF7Encoding | UTF-7エンコーディング |
| System.Text | UnicodeEncoding | UTF-16エンコーディング |
| System.Text | UTF32Encoding | UTF-32エンコーディング |
| System.Text | ASCIIEncoding | ASCIIエンコーディング |
| System.IO | File | ファイル操作 |
| System.IO | Directory | ディレクトリ操作 |
| System.IO | Path | パス操作 |
| System.IO | FileStream | ファイルストリーム |
| System.IO | MemoryStream | メモリストリーム |
| System.IO | BinaryReader | バイナリ読み取り |
| System.IO | BinaryWriter | バイナリ書き込み |
| System.IO | StreamReader | テキスト読み取り |
| System.IO | StreamWriter | テキスト書き込み |
| System.IO | StringReader | 文字列読み取り |
| System.IO | StringWriter | 文字列書き込み |
| System.IO | DriveInfo | ドライブ情報 |
| System.IO | FileInfo | ファイル情報 |
| System.IO | DirectoryInfo | ディレクトリ情報 |
| System.Globalization | CultureInfo | カルチャ情報 |
| System.Globalization | NumberFormatInfo | 数値書式情報 |
| System.Globalization | DateTimeFormatInfo | 日時書式情報 |
| System.Globalization | Calendar | カレンダー |
| System.Globalization | GregorianCalendar | グレゴリオ暦 |
| System.Globalization | JapaneseCalendar | 日本暦 |
| System.Globalization | HebrewCalendar | ヘブライ暦 |
| System.Globalization | HijriCalendar | イスラム暦 |
| System.Globalization | JulianCalendar | ユリウス暦 |
| System.Globalization | KoreanCalendar | 韓国暦 |
| System.Globalization | TaiwanCalendar | 台湾暦 |
| System.Globalization | ThaiBuddhistCalendar | タイ仏暦 |
| System.Security.Cryptography | MD5 | MD5ハッシュ |
| System.Security.Cryptography | SHA1 | SHA-1ハッシュ |
| System.Security.Cryptography | SHA256 | SHA-256ハッシュ |
| System.Security.Cryptography | SHA384 | SHA-384ハッシュ |
| System.Security.Cryptography | SHA512 | SHA-512ハッシュ |
| System.Security.Cryptography | RIPEMD160 | RIPEMD-160ハッシュ |
| System.Security.Cryptography | HMAC | HMAC認証 |
| System.Security.Cryptography | DES | DES暗号化 |
| System.Security.Cryptography | TripleDES | Triple DES暗号化 |
| System.Security.Cryptography | RC2 | RC2暗号化 |
| System.Security.Cryptography | Rijndael | AES/Rijndael暗号化 |
| System.Security.Cryptography | RSA | RSA暗号化 |
| System.Security.Cryptography | DSA | DSA署名 |
| System.Security.Cryptography | RNGCryptoServiceProvider | 暗号論的乱数生成 |
| System.Security.Cryptography | Rfc2898DeriveBytes | パスワードベースキー導出 |
| System.Numerics | BigInteger | 任意精度整数 |
| System.Resources | ResourceReader | リソース読み取り |
| System.Resources | ResourceWriter | リソース書き込み |
| System.Resources | ResourceManager | リソース管理 |
| Microsoft.Win32 | Registry | レジストリアクセス |
| Microsoft.Win32 | RegistryKey | レジストリキー |

## 各APIクラス定義

### System

#### 1. CorArray（配列操作）

配列の作成、操作、検索、ソートのためのユーティリティメソッドを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | CorArray |
| 名前空間 | System |
| 種類 | スタティッククラス（モジュール） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| BinarySearch(Arr, Value, [Comparer]) | Long | ソート済み配列から値を二分探索 |
| BinarySearchEx(Arr, Index, Count, Value, [Comparer]) | Long | 範囲指定で二分探索 |
| Clear(Arr, Index, Count) | (なし) | 配列要素をクリア |
| Copy(SourceArray, DestinationArray, Length) | (なし) | 配列をコピー |
| CopyEx(SourceArray, SourceIndex, DestinationArray, DestinationIndex, Length) | (なし) | 範囲指定でコピー |
| CreateInstance(ElementType, Length) | Variant | 配列を作成 |
| Exists(Arr, AddressOfPredicate) | Boolean | 条件に一致する要素の存在確認 |
| Find(Arr, AddressOfPredicate) | Variant | 条件に一致する最初の要素を取得 |
| FindIndex(Arr, AddressOfPredicate, [Index], [Count]) | Long | 条件に一致する最初の要素のインデックス |
| FindAll(Arr, AddressOfPredicate) | Variant() | 条件に一致する全要素を取得 |
| FindLast(Arr, AddressOfPredicate) | Variant | 条件に一致する最後の要素を取得 |
| FindLastIndex(Arr, AddressOfPredicate, [Index], [Count]) | Long | 条件に一致する最後の要素のインデックス |
| IndexOf(Arr, Value, [StartIndex], [Count]) | Long | 値のインデックスを取得 |
| LastIndexOf(Arr, Value, [StartIndex], [Count]) | Long | 値の最後のインデックスを取得 |
| Reverse(Arr, [Index], [Count]) | (なし) | 配列を反転 |
| Sort(Arr, [Comparer]) | (なし) | 配列をソート |
| SortEx(Arr, Index, Count, [Comparer]) | (なし) | 範囲指定でソート |

**使用例**

```vb
Dim arr() As Long
arr = Array(5, 3, 8, 1, 9)

' ソート
CorArray.Sort arr

' 二分探索
Dim index As Long
index = CorArray.BinarySearch(arr, 5)

' 反転
CorArray.Reverse arr
```

---

#### 2. CorString（文字列操作）

文字列の操作、比較、検索のためのユーティリティメソッドを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | CorString |
| 名前空間 | System |
| 種類 | スタティッククラス（モジュール） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Compare(s1, s2, [CompareMethod]) | Long | 文字列を比較 |
| CompareOrdinal(s1, s2) | Long | 序数比較 |
| Concat(Args()) | String | 文字列を連結 |
| Contains(s, Value) | Boolean | 部分文字列の存在確認 |
| EndsWith(s, Value, [CompareMethod]) | Boolean | 指定文字列で終わるか確認 |
| Equals(s1, s2, [CompareMethod]) | Boolean | 文字列の等価性を確認 |
| Format(Format, Args()) | String | 書式文字列を適用 |
| IndexOf(s, Value, [StartIndex], [Count]) | Long | 部分文字列のインデックス |
| IndexOfAny(s, AnyOf(), [StartIndex], [Count]) | Long | いずれかの文字のインデックス |
| IsNullOrEmpty(s) | Boolean | Null/空文字列の確認 |
| IsNullOrWhiteSpace(s) | Boolean | Null/空白文字列の確認 |
| Join(Separator, Values) | String | 配列を区切り文字で連結 |
| LastIndexOf(s, Value, [StartIndex], [Count]) | Long | 部分文字列の最後のインデックス |
| PadLeft(s, TotalWidth, [PaddingChar]) | String | 左パディング |
| PadRight(s, TotalWidth, [PaddingChar]) | String | 右パディング |
| Remove(s, StartIndex, [Count]) | String | 部分文字列を削除 |
| Replace(s, OldValue, NewValue) | String | 文字列を置換 |
| Split(s, Separator, [Options]) | String() | 文字列を分割 |
| StartsWith(s, Value, [CompareMethod]) | Boolean | 指定文字列で始まるか確認 |
| Substring(s, StartIndex, [Length]) | String | 部分文字列を取得 |
| ToLower(s, [Culture]) | String | 小文字に変換 |
| ToUpper(s, [Culture]) | String | 大文字に変換 |
| Trim(s, [TrimChars]) | String | 前後の空白を除去 |
| TrimEnd(s, [TrimChars]) | String | 末尾の空白を除去 |
| TrimStart(s, [TrimChars]) | String | 先頭の空白を除去 |

**使用例**

```vb
Dim s As String
s = "Hello, World!"

' 部分文字列の検索
If CorString.Contains(s, "World") Then
    Debug.Print "Found!"
End If

' 分割
Dim parts() As String
parts = CorString.Split(s, ",")

' 書式文字列
Dim formatted As String
formatted = CorString.Format("Name: {0}, Age: {1}", "John", 30)
```

---

#### 3. CorDateTime（日付時刻）

日付と時刻の操作のためのクラスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | CorDateTime |
| 名前空間 | System |
| 種類 | クラス |
| インターフェース | IObject, IComparable, IFormattable |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Year | Long | 年 |
| Month | Long | 月 (1-12) |
| Day | Long | 日 (1-31) |
| Hour | Long | 時 (0-23) |
| Minute | Long | 分 (0-59) |
| Second | Long | 秒 (0-59) |
| Millisecond | Long | ミリ秒 (0-999) |
| Ticks | Currency | 100ナノ秒単位のティック |
| DayOfWeek | DayOfWeek | 曜日 |
| DayOfYear | Long | 年初からの日数 |
| Date | CorDateTime | 日付部分のみ |
| TimeOfDay | TimeSpan | 時刻部分のみ |
| Kind | DateTimeKind | 日時の種類（Local/UTC） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Add(Value As TimeSpan) | CorDateTime | 時間間隔を加算 |
| AddDays(Value) | CorDateTime | 日数を加算 |
| AddHours(Value) | CorDateTime | 時間を加算 |
| AddMinutes(Value) | CorDateTime | 分を加算 |
| AddMonths(Value) | CorDateTime | 月を加算 |
| AddSeconds(Value) | CorDateTime | 秒を加算 |
| AddYears(Value) | CorDateTime | 年を加算 |
| AddMilliseconds(Value) | CorDateTime | ミリ秒を加算 |
| AddTicks(Value) | CorDateTime | ティックを加算 |
| Subtract(Value) | TimeSpan/CorDateTime | 減算 |
| ToLocalTime() | CorDateTime | ローカル時刻に変換 |
| ToUniversalTime() | CorDateTime | UTC時刻に変換 |
| ToOADate() | Double | OLE日付に変換 |
| ToFileTime() | Currency | ファイル時刻に変換 |
| ToString([Format], [Provider]) | String | 文字列に変換 |
| CompareTo(Value) | Long | 比較 |
| Equals(Value) | Boolean | 等価性確認 |

**スタティックメソッド（CorDateTimeStatic）**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Now | CorDateTime | 現在のローカル日時 |
| UtcNow | CorDateTime | 現在のUTC日時 |
| Today | CorDateTime | 今日の日付 |
| FromOADate(Value) | CorDateTime | OLE日付から変換 |
| FromFileTime(Value) | CorDateTime | ファイル時刻から変換 |
| Parse(s, [Provider]) | CorDateTime | 文字列から解析 |
| TryParse(s, Result, [Provider]) | Boolean | 文字列から解析（例外なし） |
| IsLeapYear(Year) | Boolean | 閏年かどうか |
| DaysInMonth(Year, Month) | Long | 月の日数 |

**使用例**

```vb
' 現在日時の取得
Dim now As CorDateTime
Set now = CorDateTime.Now

' 1週間後の日付
Dim nextWeek As CorDateTime
Set nextWeek = now.AddDays(7)

' 書式文字列で出力
Debug.Print now.ToString("yyyy-MM-dd HH:mm:ss")

' 日付の比較
If now.CompareTo(nextWeek) < 0 Then
    Debug.Print "Now is before next week"
End If
```

---

#### 4. TimeSpan（時間間隔）

時間間隔を表すクラスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | TimeSpan |
| 名前空間 | System |
| 種類 | クラス |
| インターフェース | IObject, IComparable, IFormattable |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Days | Long | 日数部分 |
| Hours | Long | 時間部分 (0-23) |
| Minutes | Long | 分部分 (0-59) |
| Seconds | Long | 秒部分 (0-59) |
| Milliseconds | Long | ミリ秒部分 (0-999) |
| TotalDays | Double | 日数の合計 |
| TotalHours | Double | 時間の合計 |
| TotalMinutes | Double | 分の合計 |
| TotalSeconds | Double | 秒の合計 |
| TotalMilliseconds | Double | ミリ秒の合計 |
| Ticks | Currency | ティックの合計 |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Add(ts) | TimeSpan | 加算 |
| Subtract(ts) | TimeSpan | 減算 |
| Negate() | TimeSpan | 符号反転 |
| Duration() | TimeSpan | 絶対値 |
| CompareTo(Value) | Long | 比較 |
| Equals(Value) | Boolean | 等価性確認 |
| ToString([Format]) | String | 文字列に変換 |

**スタティックメソッド（TimeSpanStatic）**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| FromDays(Value) | TimeSpan | 日数から作成 |
| FromHours(Value) | TimeSpan | 時間から作成 |
| FromMinutes(Value) | TimeSpan | 分から作成 |
| FromSeconds(Value) | TimeSpan | 秒から作成 |
| FromMilliseconds(Value) | TimeSpan | ミリ秒から作成 |
| FromTicks(Value) | TimeSpan | ティックから作成 |
| Parse(s) | TimeSpan | 文字列から解析 |
| TryParse(s, Result) | Boolean | 文字列から解析（例外なし） |
| Zero | TimeSpan | ゼロ間隔 |
| MaxValue | TimeSpan | 最大値 |
| MinValue | TimeSpan | 最小値 |

---

### System.Collections

#### 5. ArrayList（動的配列）

動的にサイズが変更されるリストを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | ArrayList |
| 名前空間 | System.Collections |
| 種類 | クラス |
| インターフェース | IObject, ICollection, IEnumerable, ICloneable, IList |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Capacity | Long | 内部配列の容量 |
| Count | Long | 要素数 |
| IsFixedSize | Boolean | 固定サイズかどうか |
| IsReadOnly | Boolean | 読み取り専用かどうか |
| Item(Index) | Variant | インデックスで要素にアクセス |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Add(Value) | Long | 要素を追加 |
| AddRange(Source) | (なし) | コレクションを追加 |
| BinarySearch(Value, [Comparer]) | Long | 二分探索 |
| BinarySearchEx(Index, Count, Value, [Comparer]) | Long | 範囲指定で二分探索 |
| Clear() | (なし) | 全要素を削除 |
| Clone() | Object | 浅いコピーを作成 |
| Contains(Value) | Boolean | 要素の存在確認 |
| CopyTo(Array, [ArrayIndex]) | (なし) | 配列にコピー |
| GetEnumerator() | IEnumerator | イテレータを取得 |
| GetRange(Index, Count) | ArrayList | 範囲を取得 |
| IndexOf(Value, [StartIndex], [Count]) | Long | 要素のインデックス |
| Insert(Index, Value) | (なし) | 位置を指定して挿入 |
| InsertRange(Index, Source) | (なし) | 位置を指定してコレクションを挿入 |
| LastIndexOf(Value, [StartIndex], [Count]) | Long | 要素の最後のインデックス |
| Remove(Value) | (なし) | 要素を削除 |
| RemoveAt(Index) | (なし) | インデックスで削除 |
| RemoveRange(Index, Count) | (なし) | 範囲を削除 |
| Reverse([Index], [Count]) | (なし) | 反転 |
| SetRange(Index, Source) | (なし) | 範囲を置換 |
| Sort([Comparer]) | (なし) | ソート |
| SortEx(Index, Count, [Comparer]) | (なし) | 範囲指定でソート |
| ToArray([ArrayType]) | Variant() | 配列に変換 |
| TrimToSize() | (なし) | 容量を要素数に合わせる |

**使用例**

```vb
Dim list As New ArrayList

' 要素の追加
list.Add "Apple"
list.Add "Banana"
list.Add "Cherry"

' インデックスでアクセス
Debug.Print list(1)  ' "Banana"

' 検索
If list.Contains("Banana") Then
    Debug.Print "Found Banana!"
End If

' ソート
list.Sort

' For Eachでイテレーション
Dim item As Variant
For Each item In list
    Debug.Print item
Next
```

---

#### 6. Hashtable（ハッシュテーブル）

キーと値のペアを格納するハッシュテーブルを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Hashtable |
| 名前空間 | System.Collections |
| 種類 | クラス |
| インターフェース | IObject, ICollection, IEnumerable, ICloneable, IDictionary |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Count | Long | 要素数 |
| IsFixedSize | Boolean | 固定サイズかどうか |
| IsReadOnly | Boolean | 読み取り専用かどうか |
| Item(Key) | Variant | キーで値にアクセス |
| Keys | ICollection | 全キーのコレクション |
| Values | ICollection | 全値のコレクション |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Add(Key, Value) | (なし) | キーと値を追加 |
| Clear() | (なし) | 全要素を削除 |
| Clone() | Object | 浅いコピーを作成 |
| Contains(Key) | Boolean | キーの存在確認 |
| ContainsKey(Key) | Boolean | キーの存在確認 |
| ContainsValue(Value) | Boolean | 値の存在確認 |
| CopyTo(Array, ArrayIndex) | (なし) | 配列にコピー |
| GetEnumerator() | IDictionaryEnumerator | イテレータを取得 |
| Remove(Key) | (なし) | キーで削除 |

**使用例**

```vb
Dim ht As New Hashtable

' 要素の追加
ht.Add "name", "John"
ht.Add "age", 30
ht.Add "city", "Tokyo"

' キーでアクセス
Debug.Print ht("name")  ' "John"

' キーの存在確認
If ht.ContainsKey("age") Then
    Debug.Print "Age: " & ht("age")
End If

' 全キーのイテレーション
Dim key As Variant
For Each key In ht.Keys
    Debug.Print key & ": " & ht(key)
Next
```

---

#### 7. Queue（キュー）

FIFO（先入れ先出し）のキューを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Queue |
| 名前空間 | System.Collections |
| 種類 | クラス |
| インターフェース | IObject, ICollection, IEnumerable, ICloneable |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Count | Long | 要素数 |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Clear() | (なし) | 全要素を削除 |
| Clone() | Object | 浅いコピーを作成 |
| Contains(Value) | Boolean | 要素の存在確認 |
| CopyTo(Array, ArrayIndex) | (なし) | 配列にコピー |
| Dequeue() | Variant | 先頭要素を取り出す |
| Enqueue(Value) | (なし) | 末尾に要素を追加 |
| GetEnumerator() | IEnumerator | イテレータを取得 |
| Peek() | Variant | 先頭要素を参照（削除しない） |
| ToArray() | Variant() | 配列に変換 |
| TrimToSize() | (なし) | 容量を要素数に合わせる |

---

#### 8. Stack（スタック）

LIFO（後入れ先出し）のスタックを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Stack |
| 名前空間 | System.Collections |
| 種類 | クラス |
| インターフェース | IObject, ICollection, IEnumerable, ICloneable |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Count | Long | 要素数 |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Clear() | (なし) | 全要素を削除 |
| Clone() | Object | 浅いコピーを作成 |
| Contains(Value) | Boolean | 要素の存在確認 |
| CopyTo(Array, ArrayIndex) | (なし) | 配列にコピー |
| GetEnumerator() | IEnumerator | イテレータを取得 |
| Peek() | Variant | 先頭要素を参照（削除しない） |
| Pop() | Variant | 先頭要素を取り出す |
| Push(Value) | (なし) | 先頭に要素を追加 |
| ToArray() | Variant() | 配列に変換 |

---

#### 9. SortedList（ソート済みリスト）

キーでソートされたキー/値ペアのリストを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | SortedList |
| 名前空間 | System.Collections |
| 種類 | クラス |
| インターフェース | IObject, ICollection, IEnumerable, ICloneable, IDictionary |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Capacity | Long | 内部配列の容量 |
| Count | Long | 要素数 |
| Item(Key) | Variant | キーで値にアクセス |
| Keys | ICollection | 全キーのコレクション |
| Values | ICollection | 全値のコレクション |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Add(Key, Value) | (なし) | キーと値を追加 |
| Clear() | (なし) | 全要素を削除 |
| Clone() | Object | 浅いコピーを作成 |
| Contains(Key) | Boolean | キーの存在確認 |
| ContainsKey(Key) | Boolean | キーの存在確認 |
| ContainsValue(Value) | Boolean | 値の存在確認 |
| GetByIndex(Index) | Variant | インデックスで値を取得 |
| GetKey(Index) | Variant | インデックスでキーを取得 |
| GetKeyList() | IList | キーリストを取得 |
| GetValueList() | IList | 値リストを取得 |
| IndexOfKey(Key) | Long | キーのインデックス |
| IndexOfValue(Value) | Long | 値のインデックス |
| Remove(Key) | (なし) | キーで削除 |
| RemoveAt(Index) | (なし) | インデックスで削除 |
| SetByIndex(Index, Value) | (なし) | インデックスで値を設定 |
| TrimToSize() | (なし) | 容量を要素数に合わせる |

---

### System.Text

#### 10. StringBuilder（文字列構築）

効率的な文字列の構築と操作のためのクラスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | StringBuilder |
| 名前空間 | System.Text |
| 種類 | クラス |
| インターフェース | IObject |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Capacity | Long | 内部バッファの容量 |
| Length | Long | 文字列の長さ |
| Chars(Index) | Integer | インデックスで文字にアクセス |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Append(Value, [StartIndex], [Count]) | StringBuilder | 値を追加 |
| AppendChar(Char, [Count]) | StringBuilder | 文字を追加 |
| AppendFormat(Format, Args()) | StringBuilder | 書式文字列を追加 |
| AppendFormatEx(Provider, Format, Args()) | StringBuilder | カルチャ指定で書式文字列を追加 |
| AppendLine([s]) | StringBuilder | 行を追加 |
| AppendString(s) | (なし) | 文字列を追加（高速版） |
| Clear() | StringBuilder | クリア |
| CopyTo(SourceIndex, Destination, DestinationIndex, Count) | (なし) | 文字配列にコピー |
| EnsureCapacity(Capacity) | Long | 容量を確保 |
| Insert(Index, Value) | StringBuilder | 位置を指定して挿入 |
| Remove(StartIndex, Length) | StringBuilder | 範囲を削除 |
| Replace(OldValue, NewValue, [StartIndex], [Count]) | StringBuilder | 置換 |
| ToString([StartIndex], [Length]) | String | 文字列に変換 |

**使用例**

```vb
Dim sb As New StringBuilder

' 文字列の追加
sb.Append "Hello"
sb.Append ", "
sb.Append "World!"

' 行の追加
sb.AppendLine
sb.AppendLine "This is a new line."

' 書式文字列
sb.AppendFormat "Value: {0}, Date: {1}", 100, Now

' 文字列に変換
Debug.Print sb.ToString()
```

---

#### 11. Encoding（エンコーディング）

文字エンコーディングの基底クラスと各種エンコーディングを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Encoding |
| 名前空間 | System.Text |
| 種類 | 抽象クラス |

**スタティックプロパティ（EncodingStatic）**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| ASCII | ASCIIEncoding | ASCIIエンコーディング |
| UTF7 | UTF7Encoding | UTF-7エンコーディング |
| UTF8 | UTF8Encoding | UTF-8エンコーディング |
| Unicode | UnicodeEncoding | UTF-16 LEエンコーディング |
| BigEndianUnicode | UnicodeEncoding | UTF-16 BEエンコーディング |
| UTF32 | UTF32Encoding | UTF-32エンコーディング |
| Default | Encoding | システムデフォルト |

**主要メソッド（共通）**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| GetBytes(s) | Byte() | 文字列をバイト配列に変換 |
| GetBytesEx(Chars, CharIndex, CharCount, Bytes, ByteIndex) | Long | 文字配列をバイト配列に変換 |
| GetByteCount(s) | Long | バイト数を取得 |
| GetChars(Bytes) | Integer() | バイト配列を文字配列に変換 |
| GetCharsEx(Bytes, ByteIndex, ByteCount, Chars, CharIndex) | Long | バイト配列を文字配列に変換 |
| GetCharCount(Bytes) | Long | 文字数を取得 |
| GetString(Bytes, [Index], [Count]) | String | バイト配列を文字列に変換 |
| GetMaxByteCount(CharCount) | Long | 最大バイト数を取得 |
| GetMaxCharCount(ByteCount) | Long | 最大文字数を取得 |
| GetPreamble() | Byte() | BOMを取得 |
| GetDecoder() | Decoder | デコーダを取得 |
| GetEncoder() | Encoder | エンコーダを取得 |

**使用例**

```vb
' UTF-8でエンコード
Dim utf8 As Encoding
Set utf8 = Encoding.UTF8

Dim bytes() As Byte
bytes = utf8.GetBytes("Hello, World!")

' UTF-8でデコード
Dim s As String
s = utf8.GetString(bytes)
```

---

### System.IO

#### 12. File（ファイル操作）

ファイル操作のスタティックメソッドを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | File |
| 名前空間 | System.IO |
| 種類 | スタティッククラス（モジュール） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| AppendAllText(Path, Contents, [Encoding]) | (なし) | テキストを追記 |
| Copy(SourceFileName, DestFileName, [Overwrite]) | (なし) | ファイルをコピー |
| Create(Path) | FileStream | ファイルを作成 |
| Delete(Path) | (なし) | ファイルを削除 |
| Exists(Path) | Boolean | 存在確認 |
| GetAttributes(Path) | FileAttributes | 属性を取得 |
| GetCreationTime(Path) | CorDateTime | 作成日時を取得 |
| GetLastAccessTime(Path) | CorDateTime | 最終アクセス日時を取得 |
| GetLastWriteTime(Path) | CorDateTime | 最終更新日時を取得 |
| Move(SourceFileName, DestFileName) | (なし) | ファイルを移動 |
| Open(Path, Mode, [Access], [Share]) | FileStream | ファイルを開く |
| OpenRead(Path) | FileStream | 読み取り用に開く |
| OpenText(Path) | StreamReader | テキスト読み取り用に開く |
| OpenWrite(Path) | FileStream | 書き込み用に開く |
| ReadAllBytes(Path) | Byte() | 全バイトを読み取り |
| ReadAllLines(Path, [Encoding]) | String() | 全行を読み取り |
| ReadAllText(Path, [Encoding]) | String | 全テキストを読み取り |
| ReadLines(Path, [Encoding]) | IEnumerable | 行を遅延読み取り |
| SetAttributes(Path, Attributes) | (なし) | 属性を設定 |
| SetCreationTime(Path, CreationTime) | (なし) | 作成日時を設定 |
| SetLastAccessTime(Path, LastAccessTime) | (なし) | 最終アクセス日時を設定 |
| SetLastWriteTime(Path, LastWriteTime) | (なし) | 最終更新日時を設定 |
| WriteAllBytes(Path, Bytes) | (なし) | 全バイトを書き込み |
| WriteAllLines(Path, Contents, [Encoding]) | (なし) | 全行を書き込み |
| WriteAllText(Path, Contents, [Encoding]) | (なし) | 全テキストを書き込み |

---

#### 13. Directory（ディレクトリ操作）

ディレクトリ操作のスタティックメソッドを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Directory |
| 名前空間 | System.IO |
| 種類 | スタティッククラス（モジュール） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| CreateDirectory(Path) | DirectoryInfo | ディレクトリを作成 |
| Delete(Path, [Recursive]) | (なし) | ディレクトリを削除 |
| Exists(Path) | Boolean | 存在確認 |
| GetCreationTime(Path) | CorDateTime | 作成日時を取得 |
| GetCurrentDirectory() | String | カレントディレクトリを取得 |
| GetDirectories(Path, [SearchPattern], [SearchOption]) | String() | サブディレクトリを取得 |
| GetDirectoryRoot(Path) | String | ルートを取得 |
| GetFiles(Path, [SearchPattern], [SearchOption]) | String() | ファイルを取得 |
| GetFileSystemEntries(Path, [SearchPattern]) | String() | ファイルとディレクトリを取得 |
| GetLastAccessTime(Path) | CorDateTime | 最終アクセス日時を取得 |
| GetLastWriteTime(Path) | CorDateTime | 最終更新日時を取得 |
| GetLogicalDrives() | String() | 論理ドライブを取得 |
| GetParent(Path) | DirectoryInfo | 親ディレクトリを取得 |
| Move(SourceDirName, DestDirName) | (なし) | ディレクトリを移動 |
| SetCreationTime(Path, CreationTime) | (なし) | 作成日時を設定 |
| SetCurrentDirectory(Path) | (なし) | カレントディレクトリを設定 |
| SetLastAccessTime(Path, LastAccessTime) | (なし) | 最終アクセス日時を設定 |
| SetLastWriteTime(Path, LastWriteTime) | (なし) | 最終更新日時を設定 |

---

#### 14. Path（パス操作）

パス文字列の操作のスタティックメソッドを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Path |
| 名前空間 | System.IO |
| 種類 | スタティッククラス（モジュール） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| ChangeExtension(Path, Extension) | String | 拡張子を変更 |
| Combine(Path1, Path2) | String | パスを結合 |
| GetDirectoryName(Path) | String | ディレクトリ名を取得 |
| GetExtension(Path) | String | 拡張子を取得 |
| GetFileName(Path) | String | ファイル名を取得 |
| GetFileNameWithoutExtension(Path) | String | 拡張子なしファイル名を取得 |
| GetFullPath(Path) | String | フルパスを取得 |
| GetPathRoot(Path) | String | ルートを取得 |
| GetRandomFileName() | String | ランダムファイル名を生成 |
| GetTempFileName() | String | 一時ファイル名を取得 |
| GetTempPath() | String | 一時パスを取得 |
| HasExtension(Path) | Boolean | 拡張子があるか確認 |
| IsPathRooted(Path) | Boolean | ルートパスか確認 |

---

#### 15. FileStream（ファイルストリーム）

ファイルへの読み書きストリームを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | FileStream |
| 名前空間 | System.IO |
| 種類 | クラス |
| 継承 | Stream |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| CanRead | Boolean | 読み取り可能か |
| CanSeek | Boolean | シーク可能か |
| CanWrite | Boolean | 書き込み可能か |
| Length | Currency | ストリームの長さ |
| Position | Currency | 現在位置 |
| Name | String | ファイル名 |
| Handle | Long | ファイルハンドル |
| SafeFileHandle | SafeFileHandle | 安全なファイルハンドル |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Close() | (なし) | ストリームを閉じる |
| Flush() | (なし) | バッファをフラッシュ |
| Lock(Position, Length) | (なし) | ファイルをロック |
| Read(Buffer, Offset, Count) | Long | 読み取り |
| ReadByte() | Long | 1バイト読み取り |
| Seek(Offset, Origin) | Currency | シーク |
| SetLength(Value) | (なし) | 長さを設定 |
| Unlock(Position, Length) | (なし) | ロック解除 |
| Write(Buffer, Offset, Count) | (なし) | 書き込み |
| WriteByte(Value) | (なし) | 1バイト書き込み |

---

### System.Security.Cryptography

#### 16. HashAlgorithm（ハッシュアルゴリズム）

ハッシュアルゴリズムの基底クラスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | HashAlgorithm |
| 名前空間 | System.Security.Cryptography |
| 種類 | 抽象クラス |

**派生クラス**

| クラス名 | 説明 |
| --- | --- |
| MD5CryptoServiceProvider | MD5ハッシュ（128ビット） |
| SHA1CryptoServiceProvider | SHA-1ハッシュ（160ビット） |
| SHA1Managed | SHA-1マネージド実装 |
| SHA256Managed | SHA-256ハッシュ（256ビット） |
| SHA384Managed | SHA-384ハッシュ（384ビット） |
| SHA512Managed | SHA-512ハッシュ（512ビット） |
| RIPEMD160Managed | RIPEMD-160ハッシュ（160ビット） |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| Hash | Byte() | 計算されたハッシュ値 |
| HashSize | Long | ハッシュサイズ（ビット） |
| CanTransformMultipleBlocks | Boolean | 複数ブロック変換可能か |
| InputBlockSize | Long | 入力ブロックサイズ |
| OutputBlockSize | Long | 出力ブロックサイズ |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Clear() | (なし) | リソースを解放 |
| ComputeHash(Buffer, [Offset], [Count]) | Byte() | バイト配列からハッシュ計算 |
| ComputeHashFromStream(Stream) | Byte() | ストリームからハッシュ計算 |
| TransformBlock(InputBuffer, InputOffset, InputCount, OutputBuffer, OutputOffset) | Long | ブロック変換 |
| TransformFinalBlock(InputBuffer, InputOffset, InputCount) | Byte() | 最終ブロック変換 |

**使用例**

```vb
' SHA256でハッシュ計算
Dim sha256 As SHA256
Set sha256 = SHA256Managed.Create()

Dim data() As Byte
data = Encoding.UTF8.GetBytes("Hello, World!")

Dim hash() As Byte
hash = sha256.ComputeHash(data)

' 16進数文字列に変換
Dim hexString As String
Dim i As Long
For i = 0 To UBound(hash)
    hexString = hexString & Right("0" & Hex(hash(i)), 2)
Next
Debug.Print hexString
```

---

#### 17. SymmetricAlgorithm（対称暗号アルゴリズム）

対称暗号アルゴリズムの基底クラスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | SymmetricAlgorithm |
| 名前空間 | System.Security.Cryptography |
| 種類 | 抽象クラス |

**派生クラス**

| クラス名 | 説明 |
| --- | --- |
| DESCryptoServiceProvider | DES暗号化（56ビットキー） |
| TripleDESCryptoServiceProvider | Triple DES暗号化（112/168ビットキー） |
| RC2CryptoServiceProvider | RC2暗号化（40-128ビットキー） |
| RijndaelManaged | Rijndael/AES暗号化（128/192/256ビットキー） |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| BlockSize | Long | ブロックサイズ（ビット） |
| FeedbackSize | Long | フィードバックサイズ |
| IV | Byte() | 初期化ベクトル |
| Key | Byte() | 暗号化キー |
| KeySize | Long | キーサイズ（ビット） |
| LegalBlockSizes | KeySizes() | 有効なブロックサイズ |
| LegalKeySizes | KeySizes() | 有効なキーサイズ |
| Mode | CipherMode | 暗号化モード（CBC等） |
| Padding | PaddingMode | パディングモード |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Clear() | (なし) | リソースを解放 |
| CreateDecryptor([Key], [IV]) | ICryptoTransform | 復号化トランスフォームを作成 |
| CreateEncryptor([Key], [IV]) | ICryptoTransform | 暗号化トランスフォームを作成 |
| GenerateIV() | (なし) | IVを生成 |
| GenerateKey() | (なし) | キーを生成 |
| ValidKeySize(BitLength) | Boolean | キーサイズの検証 |

**使用例**

```vb
' AES暗号化
Dim aes As RijndaelManaged
Set aes = New RijndaelManaged
aes.KeySize = 256
aes.GenerateKey
aes.GenerateIV

' 暗号化
Dim encryptor As ICryptoTransform
Set encryptor = aes.CreateEncryptor()

Dim plainText() As Byte
plainText = Encoding.UTF8.GetBytes("Secret Message")

Dim encrypted() As Byte
encrypted = encryptor.TransformFinalBlock(plainText, 0, UBound(plainText) + 1)

' 復号化
Dim decryptor As ICryptoTransform
Set decryptor = aes.CreateDecryptor()

Dim decrypted() As Byte
decrypted = decryptor.TransformFinalBlock(encrypted, 0, UBound(encrypted) + 1)

Debug.Print Encoding.UTF8.GetString(decrypted)
```

---

#### 18. RSACryptoServiceProvider（RSA暗号化）

RSA公開鍵暗号化を提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | RSACryptoServiceProvider |
| 名前空間 | System.Security.Cryptography |
| 種類 | クラス |
| 継承 | RSA, AsymmetricAlgorithm |
| インターフェース | ICspAsymmetricAlgorithm |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| KeySize | Long | キーサイズ（ビット） |
| PublicOnly | Boolean | 公開鍵のみか |
| CspKeyContainerInfo | CspKeyContainerInfo | キーコンテナ情報 |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Clear() | (なし) | リソースを解放 |
| Decrypt(Data, UseOAEP) | Byte() | 復号化 |
| Encrypt(Data, UseOAEP) | Byte() | 暗号化 |
| ExportParameters(IncludePrivateParameters) | RSAParameters | パラメータをエクスポート |
| ImportParameters(Parameters) | (なし) | パラメータをインポート |
| SignData(Buffer, HashAlgorithm) | Byte() | データに署名 |
| SignHash(Hash, OID) | Byte() | ハッシュに署名 |
| VerifyData(Buffer, HashAlgorithm, Signature) | Boolean | 署名を検証 |
| VerifyHash(Hash, OID, Signature) | Boolean | ハッシュ署名を検証 |

---

### System.Numerics

#### 19. BigInteger（任意精度整数）

任意精度の整数演算を提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | BigInteger |
| 名前空間 | System.Numerics |
| 種類 | クラス |
| インターフェース | IObject, IComparable, IFormattable |

**プロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| IsEven | Boolean | 偶数か |
| IsOne | Boolean | 1か |
| IsPowerOfTwo | Boolean | 2の累乗か |
| IsZero | Boolean | ゼロか |
| Sign | Long | 符号（-1, 0, 1） |

**主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Abs() | BigInteger | 絶対値 |
| Add(Value) | BigInteger | 加算 |
| CompareTo(Value) | Long | 比較 |
| Divide(Divisor) | BigInteger | 除算 |
| DivRem(Divisor, Remainder) | BigInteger | 除算と剰余 |
| Equals(Value) | Boolean | 等価性確認 |
| GetHashCode() | Long | ハッシュコード |
| GreatestCommonDivisor(Value) | BigInteger | 最大公約数 |
| Max(Value) | BigInteger | 最大値 |
| Min(Value) | BigInteger | 最小値 |
| Modulus(Divisor) | BigInteger | 剰余 |
| ModPow(Exponent, Modulus) | BigInteger | 冪剰余 |
| Multiply(Value) | BigInteger | 乗算 |
| Negate() | BigInteger | 符号反転 |
| Pow(Exponent) | BigInteger | 累乗 |
| ShiftLeft(Shift) | BigInteger | 左シフト |
| ShiftRight(Shift) | BigInteger | 右シフト |
| Subtract(Value) | BigInteger | 減算 |
| ToByteArray() | Byte() | バイト配列に変換 |
| ToString([Format]) | String | 文字列に変換 |

**スタティックメソッド（BigIntegerStatic）**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Parse(s) | BigInteger | 文字列から解析 |
| TryParse(s, Result) | Boolean | 文字列から解析（例外なし） |
| Zero | BigInteger | ゼロ |
| One | BigInteger | 1 |
| MinusOne | BigInteger | -1 |

---

### Microsoft.Win32

#### 20. Registry / RegistryKey（レジストリアクセス）

Windowsレジストリへのアクセスを提供します。

**基本情報**

| 項目 | 内容 |
| --- | --- |
| クラス名 | Registry, RegistryKey |
| 名前空間 | Microsoft.Win32 |
| 種類 | スタティッククラス、クラス |

**Registryスタティックプロパティ**

| プロパティ名 | 型 | 説明 |
| --- | --- | --- |
| ClassesRoot | RegistryKey | HKEY_CLASSES_ROOT |
| CurrentConfig | RegistryKey | HKEY_CURRENT_CONFIG |
| CurrentUser | RegistryKey | HKEY_CURRENT_USER |
| LocalMachine | RegistryKey | HKEY_LOCAL_MACHINE |
| Users | RegistryKey | HKEY_USERS |

**RegistryKey主要メソッド**

| メソッド名 | 戻り値 | 説明 |
| --- | --- | --- |
| Close() | (なし) | キーを閉じる |
| CreateSubKey(SubKey) | RegistryKey | サブキーを作成 |
| DeleteSubKey(SubKey, [ThrowOnMissingSubKey]) | (なし) | サブキーを削除 |
| DeleteSubKeyTree(SubKey) | (なし) | サブキーツリーを削除 |
| DeleteValue(Name, [ThrowOnMissingValue]) | (なし) | 値を削除 |
| Flush() | (なし) | 変更をフラッシュ |
| GetSubKeyNames() | String() | サブキー名を取得 |
| GetValue(Name, [DefaultValue]) | Variant | 値を取得 |
| GetValueKind(Name) | RegistryValueKind | 値の種類を取得 |
| GetValueNames() | String() | 値名を取得 |
| OpenSubKey(Name, [Writable]) | RegistryKey | サブキーを開く |
| SetValue(Name, Value, [ValueKind]) | (なし) | 値を設定 |

## 例外クラス

VBCorLibは.NET Frameworkと同様の例外階層を提供します。

| 例外クラス | 基底クラス | 説明 |
| --- | --- | --- |
| Exception | (なし) | 全例外の基底 |
| SystemException | Exception | システム例外の基底 |
| ApplicationException | Exception | アプリケーション例外の基底 |
| ArgumentException | SystemException | 引数例外 |
| ArgumentNullException | ArgumentException | Null引数例外 |
| ArgumentOutOfRangeException | ArgumentException | 範囲外引数例外 |
| ArithmeticException | SystemException | 算術例外 |
| DivideByZeroException | ArithmeticException | ゼロ除算例外 |
| OverflowException | ArithmeticException | オーバーフロー例外 |
| FormatException | SystemException | 書式例外 |
| IndexOutOfRangeException | SystemException | インデックス範囲外例外 |
| InvalidCastException | SystemException | 無効キャスト例外 |
| InvalidOperationException | SystemException | 無効操作例外 |
| NotSupportedException | SystemException | 非サポート例外 |
| ObjectDisposedException | InvalidOperationException | オブジェクト破棄済み例外 |
| OutOfMemoryException | SystemException | メモリ不足例外 |
| RankException | SystemException | 配列次元例外 |
| UnauthorizedAccessException | SystemException | アクセス拒否例外 |
| IOException | SystemException | IO例外 |
| FileNotFoundException | IOException | ファイル未検出例外 |
| DirectoryNotFoundException | IOException | ディレクトリ未検出例外 |
| PathTooLongException | IOException | パス長超過例外 |
| EndOfStreamException | IOException | ストリーム終端例外 |
| DriveNotFoundException | IOException | ドライブ未検出例外 |
| CryptographicException | SystemException | 暗号化例外 |
| SerializationException | SystemException | シリアライゼーション例外 |

## 備考

- VBCorLibは.NET Framework 2.0相当のAPIを提供しています
- 全てのクラスはCOM互換であり、VB6およびVBA（Excel、Access等）から利用可能です
- ライブラリの詳細なドキュメントは `http://www.kellyethridge.com/vbcorlib/doc/CorLib.html` で参照できます
- ライセンスはMITライセンスです
